/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.component;

import java.awt.Component;
import java.awt.Dimension;
import java.text.ParseException;

import javax.swing.Icon;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.ListCellRenderer;
import javax.swing.border.EmptyBorder;

import no.g9.client.support.G9Document;

/**
 * The combo box renderer used to render list values.
 */
@SuppressWarnings("rawtypes")
public class ComboBoxRenderer extends JLabel implements ListCellRenderer {

    /** Reference to the document model used to format the values */
    private G9Document documentModel;

    /**
     * Constructs a new combo box renderer.
     *
     * @param documentModel The document model to use when rendering values
     * @param outputJustification the output justification
     */
    public ComboBoxRenderer(G9Document documentModel,
            int outputJustification) {
        super();
        setOpaque(true);
        setBorder(new EmptyBorder(1, 2, 1, 1));
        this.setHorizontalAlignment(outputJustification);
        this.documentModel = documentModel;
    }

    /**
     * Gets the preffered size of this renderer.
     *
     * @return the preffered size of this renderer.
     */
    @Override
    public Dimension getPreferredSize() {
        Dimension size;

        if ((this.getText() == null) || (this.getText().equals(""))) {
            setText(" ");
            size = super.getPreferredSize();
            setText("");
        } else {
            size = super.getPreferredSize();
        }
        return size;
    }


    /**
     * Gets the component that will be responsible for rendering the specified
     * value.
     *
     * @param list the list this renderer is created for
     * @param value the value to render
     * @param index the list index of the value
     * @param isSelected boolean flag indicating if the cell is selected or not
     * @param cellHasFocus boolean flag indicating if the cell has focus or not
     * @return the component responsible for rendering the value
     */
    @Override
    public Component getListCellRendererComponent(JList list, Object value,
            int index, boolean isSelected, boolean cellHasFocus) {
        setBackground(isSelected ? list.getSelectionBackground() : list
                .getBackground());
        setForeground(isSelected ? list.getSelectionForeground() : list
                .getForeground());
        setFont(list.getFont());
        if (value instanceof Icon) {
            setIcon((Icon) value);
        } else {
            setText(value == null ? "" : format(value));

        }

        return this;
    }


    /**
     * Internal use.
     * <p>
     * Formats the specified value to the approriate String represenatsion.
     * First the actual value to use is obtained by invoking the (possibly
     * overridden) toValue method. Then the document model is used on that
     * value, formatting it to an appropriate String.
     *
     * @param value either the value it self or a domain object with the value.
     * @return the approriate String representation of the specified value.
     */
    protected String format(Object value) {
        if (value == null) {
            return null;
        }
        String formatValue = null;
        Object actualValue = getDisplayValue(value);
        if (actualValue != null) {
            formatValue = actualValue.toString();
            if (documentModel != null) {
                documentModel.setValue(actualValue);
                formatValue = documentModel.getOutputText();
            }
        }
        return formatValue;
    }


    /**
     * Internal use.
     * <p>
     * Overridden in generated code. Returns the correct value for the specified
     * domain object. E.g. if this combo box is used to display
     * "domainObject.foo", the overridden method returns domainObject.getFoo().
     * The returned value is then formatted to a String using the display rules
     * of that attribute. Finally the String is rendered and displayed.
     *
     * @param domainObject the domain object with the attribute that should be
     *            displayed by this renderer.
     * @return the value to use when rendering this domain object.
     */
    protected Object getDisplayValue(Object domainObject) {
        return domainObject;
    }


    /**
     * Parses the specified string to an appropriate value object.
     *
     * @param str the string to parse
     * @return the value object
     * @throws ParseException if the specified string could not be parsed into a
     *             value object.
     */
    public Object parse(String str) throws ParseException {
        return documentModel.parse(str);
    }

}
