/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.component;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.Action;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.RootPaneContainer;

import no.g9.client.support.DialogBlocker;


/**
 * The class representing a button
 */
public class G9Button extends JButton {

    private boolean inToolbar = false;

    /** Property set to <code>true</code> if button is blocked. */
    private boolean isBlocked = false;

    /** The last focused component */
    private Component lastFocused;


    /**
     * Creates a button with no set text or icon.
     */
    public G9Button() {
        this(null, null);
    }

    /**
     * Creates a button with text.
     *
     * @param text the text of the button
     */
    public G9Button(String text) {
        this(text, null);
    }

    /**
     * Creates a button where properties are taken from the <code>Action</code>
     * supplied.
     *
     * @param a the <code>Action</code> used to specify the new button
     * @since 1.3
     */
    public G9Button(Action a) {
        this();
        setAction(a);
    }

    /**
     * Creates a button with an icon.
     *
     * @param icon the Icon image to display on the button
     */
    public G9Button(Icon icon) {
        this(null, icon);
    }

    /**
     * Creates a button with initial text and an icon.
     *
     * @param text  the text of the button
     * @param icon  the Icon image to display on the button
     */
    public G9Button(String text, Icon icon) {
        super(text, icon);
        // Special focus handling if in toolbar.
        addActionListener(new ActionHandel());
        addFocusListener(new FocusHandel());
        addMouseListener(new MouseAdapter() {

            // If dialog is in a blocked state, we still might need to process
            // this mouse event in order for a button to be clicked.
            // Typically this situation occurs when a focus lost (or
            // similar) event is triggered by this button being pressed, and
            // that event is resulting in a dialog block.
            @Override public void mouseReleased(MouseEvent e) {
                Component parent = getParent();
                while (parent != null && !(parent instanceof RootPaneContainer)) {
                    parent = parent.getParent();
                }

                boolean wantEvent = DialogBlocker.isBlocked(parent)
                        && getModel().isPressed()
                        && contains(e.getPoint());
                if (wantEvent) {
                    doClick();
                }
            }
        });
    }

    /**
     * Focus adapter keeping track of the component that lost focus
     * prior to this button gaining focus.
     */
    private class FocusHandel extends FocusAdapter {

        // Keep last focused component
        @Override
        public void focusGained(FocusEvent e) {
            if (isInToolbar()) {
                lastFocused = e.getOppositeComponent();
            }
        }
    }

    /**
     * Action listener, ensuring that the focus is given back to the previously
     * focused component before the buttons clicked actions are invoked. This
     * only applies if this button resides in a toolbar.
     */
    private class ActionHandel implements ActionListener {

        // Give back focus before other actions are performed.
        // This ensures that the components focus-lost method will
        // run if pressing this button moves focus to a third component
        // (typically another window etc.).
        @Override
        public void actionPerformed(ActionEvent e) {
            if (isInToolbar()) {
                if (lastFocused != null) {
                    lastFocused.requestFocus();
                }
            }
        }
    }

    /**
     * Internal use.
     *
     * @return <code>true</code> if this button is placed in a toolbar.
     */
    public final boolean isInToolbar() {
        return inToolbar;
    }


    /**
     * Internal use.
     * <p>
     * Sets the <code>inToolbar</code> property, which shuould be
     * <code>true</code> if this button is used in a toolbar.
     *
     * @param inToolbar the value of the toolbar property.
     */
    public void setInToolbar(boolean inToolbar) {
        this.inToolbar = inToolbar;
    }

    @Override
    public void grabFocus() {
        boolean oldEnabled = isEnabled();
        setEnabled(false);
        super.grabFocus();
        setEnabled(oldEnabled);
    }

    /**
     * Internal use only. Sets the isBlocked property. If set to true, this
     * button will be considered as blocked. A blocked button will not receive
     * events when reciding in a blocked toolbar.
     *
     * @param isBlocked <code>true</code> to block.
     */
    public void setBlocked(boolean isBlocked) {
        this.isBlocked = isBlocked;
    }

    @Override
    protected void processMouseEvent(MouseEvent e) {
        if (!isBlocked) {
            super.processMouseEvent(e);
        }
    }

}
