/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.component;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.util.Iterator;

import javax.swing.AbstractButton;
import javax.swing.ButtonGroup;
import javax.swing.ButtonModel;
import javax.swing.JComponent;
import javax.swing.SwingUtilities;
import javax.swing.event.EventListenerList;

import no.g9.client.event.G9ValueState;
import no.g9.client.event.G9ValueChangedEvent;
import no.g9.client.event.G9ValueChangedListener;

/** 
 * The button group class, used to group radio buttons. 
 */
public class G9ButtonGroup extends ButtonGroup implements
        FocusListener, G9ValueState, ActionListener {

    /**
     * The old value of this button group.
     */
    private Object oldVal;
    
    /**
     * The initial value of this button group
     */
    private Object initialVal;
    
    /** 
     * An invisible radio button used when no selection is made
     */
    private G9RadioButton clearButtonValue = new G9RadioButton();

    /**
     * list of value changed listeners
     */
    protected EventListenerList vcListeners = new EventListenerList();
    
    /**
     * List of event listeners
     */
    protected EventListenerList listenerList = new EventListenerList();

    /**
     * Flag indicating wheter this group has focus
     */
    private boolean isInFocus = false;
    
    /**
     * Group as a component
     */
    private JComponent componentGroup;
    
    /**
     * Default constructor.
     * 
     */
    public G9ButtonGroup() {
        super();
        componentGroup = new ComponentGroup(this);
        super.add(clearButtonValue);
    }
    
    /**
     * Returns a component view of this group
     * @return a component view of this group
     */
    public JComponent asComponent() {
        return componentGroup;
    }

    @Override
    public void add(AbstractButton b) {
        super.add(b);
        if(b instanceof G9RadioButton) {
        	((G9RadioButton) b).setGroup(this);
        }
        b.addFocusListener(this);
        b.addActionListener(this);
        updateSelection();
    }

    @Override
    public void focusGained(FocusEvent e) {
        if (isInFocus) {
            return;
        }
        isInFocus = true;
        oldVal = getSelection();
        fireFocusEvent(e);
    }

    /**
     * Clears the radio button group. After invoking this method, none of 
     * the visible radio buttons are selected.
     *
     */
    public void clear() {
        clearButtonValue.setSelected(true);
        initialVal = getSelection();
        oldVal = getSelection();
    }
    
    @Override
    public void focusLost(final FocusEvent e) {
        
        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
                // check if another radio button in 
                // this group has focus.
                Iterator<AbstractButton> it = buttons.iterator();
                boolean otherFocus = false;
                while (!otherFocus && it.hasNext()) {
                    AbstractButton b = it.next();
                    otherFocus = b.isFocusOwner();
                }
                if (!otherFocus) {
                    // fire events!
                    isInFocus = false;
                    fireFocusEvent(e);
                    if (oldVal != null && !oldVal.equals(getSelection())
                            || oldVal == null && getSelection() != null) {
                        fireG9ValueChangedEvent(
                                G9ValueChangedEvent.VALUE_CHANGED, oldVal,
                                getSelection());
                    } 
                }
            }
        });
    }

    @Override
    public void addValueChangedListener(G9ValueChangedListener listener) {
        vcListeners.add(G9ValueChangedListener.class, listener);
    }
    
    /**
     * Adds an action listener to the group of listeners that should get
     * notified when an action event occurs.
     * @param listener the action listener
     */
    public void addActionListener(ActionListener listener) {
        listenerList.add(ActionListener.class, listener);
    }

    /**
     * Adds a focus listener to the list of listeners that should be notified
     * when a focus event occurs.
     * @param listener the focus listener
     */
    public void addFocusListener(FocusListener listener) {
        listenerList.add(FocusListener.class, listener);
    }
    
    /**
     * Fires an action event, notifying any registred listeneres interrested in
     * this type of event.
     * @param e the action event to fire. 
     */
    public void fireActionEvent(ActionEvent e) {
        Object[] listeners = listenerList.getListenerList();
        ActionEvent event = new ActionEvent(this, e.getID(), e.getActionCommand());
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == ActionListener.class) {
                ((ActionListener) listeners[i + 1]).actionPerformed(event);
            }
        }
    }
    
    /**
     * Fires a focus event, notifying any registred listeners interrested
     * in this typeof event.
     * @param e the focus event to fire.
     */
    public void fireFocusEvent(FocusEvent e) {
        Object[] listeners = listenerList.getListenerList();
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == FocusListener.class) {
                switch (e.getID()) {
                    case FocusEvent.FOCUS_GAINED : 
                        ((FocusListener) listeners[i + 1]).focusGained(e);
                        break;
                    case FocusEvent.FOCUS_LOST :
                        ((FocusListener) listeners[i + 1]).focusLost(e);
                        break;
                    default:
                        break;
                }
            }
        }
    }
    
    /**
     * Fires a g9 value changed event.
     * @param id the type of event
     * @param oldValue the old value
     * @param newValue the new value
     */
    protected void fireG9ValueChangedEvent(int id, Object oldValue,
            Object newValue) {
        Object[] listeners = vcListeners.getListenerList();
        G9ValueChangedEvent e = null;

        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (e == null) {
                e = new G9ValueChangedEvent(this, id, oldValue,
                        newValue);
            }
            if (listeners[i] == G9ValueChangedListener.class) {
                switch (e.getID()) {
                    case G9ValueChangedEvent.VALUE_CHANGED:
                        ((G9ValueChangedListener) listeners[i + 1])
                                .valueChanged(e);
                        break;
                    default:
                        break;
                }
            }
        }
    }
    
    @Override
    public void resetState() {
        oldVal = getSelection();
        initialVal = oldVal;
    }

    @Override
    public boolean isChanged() {
        Object cur = getSelection();
        if (cur == null) {
            return initialVal != null;
        }
        
        return !cur.equals(initialVal);
    }
    
    /**
     * Check if the radio group is clear, viz. that no selection is made.
     * @return <code>true</code> if no selection is made.
     */
    public boolean isClear() {
		return this.getSelection() == this.clearButtonValue.getModel()
				|| this.getSelection() == null;
	}

    /**
     * Updates initial selected value of this group.
     *
     */
    public void updateSelection() {
        initialVal = getSelection();
    }

    @Override
    public void display(Object o) {
        ButtonModel m = (ButtonModel) o;
        setSelected(m, m.isSelected());
    }

    @Override
    public Object getInitialValue() {
        return initialVal;
    }

    @Override
    public void setInitialValue(Object value) {
        initialVal = value;
        oldVal = initialVal;
    }

	/**
     * Since the radio button group some times needs to be considered 
     * as a Swing component, this class can act on behalf of the radio 
     * group.
     */
    public static class ComponentGroup extends JComponent implements G9ValueState {

        /** Reference to the radio button group. */
        private G9ButtonGroup group;
        
        /**
         * Creates the component
         * @param group the radio button group
         */
        ComponentGroup(G9ButtonGroup group) {
            this.group = group;
        }
        
        
        @Override
        public void addValueChangedListener(G9ValueChangedListener listener) {
            group.addValueChangedListener(listener);
            
        }

        @Override
        public void resetState() {
            group.resetState();
            
        }

        @Override
        public boolean isChanged() {
           return group.isChanged();
        }

        @Override
        public void display(Object o) {
            group.display(o);
            
        }

        @Override
        public Object getInitialValue() {
            return group.getInitialValue();
        }

        @Override
        public void setInitialValue(Object value) {
            group.setInitialValue(value);
            
        }
        
        /**
         * Returns the radio button group.
         * @return the radio button group.
         */
        public G9ButtonGroup getGroup() {
            return group;
        }
        
    }

    @Override
    public void actionPerformed(ActionEvent e) {
        fireActionEvent(e);
    }

}