/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.component;

import java.awt.Component;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.Icon;
import javax.swing.JCheckBox;
import javax.swing.RootPaneContainer;
import javax.swing.event.EventListenerList;

import no.g9.client.event.G9ValueChangedEvent;
import no.g9.client.event.G9ValueChangedListener;
import no.g9.client.event.G9ValueState;
import no.g9.client.support.DialogBlocker;

/**
 * In addition to being a JCheckBox, the G9CheckBox implements the
 * G9ValueState interface, which makes it possible to quire about the change
 * status of the check box.
 */
public class G9CheckBox extends JCheckBox implements G9ValueState {

    /**  The old valu of this check box, used to trigger value changed */
    private boolean oldVal;

    /**
     * The initial value of this check box, used to determine if this component
     * is changed or not.
     */
    private boolean initialVal;

    /** The list of value change listeners */
    protected EventListenerList vcListeners = new EventListenerList();

    /**
     * Creates an initially unselected check box button with no text, no icon.
     */
    public G9CheckBox() {
        this(null, null, false);
    }

    /**
     * Creates an initially unselected check box with text.
     *
     * @param text the text of the check box.
     */
    public G9CheckBox(String text) {
        this(text, null, false);
    }

    /**
     * Creates a check box with text and specifies whether or not it is
     * initially selected.
     *
     * @param text the text of the check box.
     * @param selected a boolean value indicating the initial selection state.
     *            If <code>true</code> the check box is selected
     */
    public G9CheckBox(String text, boolean selected) {
        this(text, null, selected);
    }

    /**
     * Creates a check box where properties are taken from the Action supplied.
     *
     * @param a the action perforemed when the button is pressed.
     * @since 1.3
     */
    public G9CheckBox(Action a) {
        this();
        setAction(a);
    }

    /**
     * Creates an initially unselected check box with an icon.
     *
     * @param icon the Icon image to display
     */
    public G9CheckBox(Icon icon) {
        this(null, icon, false);
    }

    /**
     * Creates a check box with an icon and specifies whether or not it is
     * initially selected.
     *
     * @param icon the Icon image to display
     * @param selected a boolean value indicating the initial selection state.
     *            If <code>true</code> the check box is selected
     */
    public G9CheckBox(Icon icon, boolean selected) {
        this(null, icon, selected);
    }

    /**
     * Creates an initially unselected check box with the specified text and
     * icon.
     *
     * @param text the text of the check box.
     * @param icon the Icon image to display
     */
    public G9CheckBox(String text, Icon icon) {
        this(text, icon, false);
    }

    /**
     * Creates a check box with text and icon, and specifies whether or not it
     * is initially selected.
     *
     * @param text the text of the check box.
     * @param icon the Icon image to display
     * @param selected a boolean value indicating the initial selection state.
     *            If <code>true</code> the check box is selected
     */
    public G9CheckBox(String text, Icon icon, boolean selected) {
        super(text, icon, selected);
        initialVal = selected;

        addFocusListener(new FocusAdapter() {
            @Override
            public void focusGained(FocusEvent e) {
                oldVal = isSelected();
            }

            @Override
            public void focusLost(FocusEvent e) {
                if (oldVal != isSelected()) {
                    fireG9ValueChangedEvent(
                            G9ValueChangedEvent.VALUE_CHANGED,
                            new Boolean(oldVal), new Boolean(isSelected()));
                }
            }
        });

        addMouseListener(new MouseAdapter() {

            // If dialog is in a blocked state, we still might need to process
            // this mouse event in order for a button to be clicked.
            // Typically this situation occurs when a focus lost (or
            // similar) event is triggered by this button being pressed, and
            // that event is resulting in a dialog block.
            @Override
            public void mouseReleased(MouseEvent e) {
                Component parent = getParent();
                while (parent != null && !(parent instanceof RootPaneContainer)) {
                    parent = parent.getParent();
                }

                boolean wantEvent = DialogBlocker.isBlocked(parent)
                && getModel().isPressed()
                && contains(e.getPoint());
                if (wantEvent) {
                    doClick();
                }
            }
        });


        // The insets needed are already calcluated by g9.
        setBorder(BorderFactory.createEmptyBorder());
    }

    @Override
    public void resetState() {
        oldVal = isSelected();
        initialVal = oldVal;
    }

    @Override
    public void addValueChangedListener(G9ValueChangedListener listener) {
        vcListeners.add(G9ValueChangedListener.class, listener);
    }

    /**
     * Fires a g9 value changed event.
     *
     * @param id the type of event
     * @param oldValue the old value
     * @param newValue the new value
     */
    protected void fireG9ValueChangedEvent(int id, Object oldValue,
            Object newValue) {
        Object[] listeners = vcListeners.getListenerList();
        G9ValueChangedEvent e = null;

        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (e == null) {
                e = new G9ValueChangedEvent(this, id, oldValue,
                        newValue);
            }
            if (listeners[i] == G9ValueChangedListener.class)
                switch (e.getID()) {
                    case G9ValueChangedEvent.VALUE_CHANGED:
                        ((G9ValueChangedListener) listeners[i + 1])
                                .valueChanged(e);
                        break;
                    default:
                        break;
                }
        }
    }

    @Override
    public boolean isChanged() {
        return initialVal != isSelected();
    }

    @Override
    public void setSelected(boolean b) {
        initialVal = b;
        super.setSelected(b);
    }

    @Override
    public void display(Object o) {
      super.setSelected(((Boolean) o).booleanValue());
    }

    @Override
    public Object getInitialValue() {
        return new Boolean(initialVal);
    }

    @Override
    public void setInitialValue(Object value) {
        if (value != null) {
            initialVal = ((Boolean) value).booleanValue();
        } else {
            initialVal = false;
        }
        oldVal = initialVal;
    }

}
