/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.component;


import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.text.ParseException;
import java.util.Collection;

import javax.swing.ComboBoxModel;
import javax.swing.DefaultCellEditor;
import javax.swing.JComboBox;
import javax.swing.JRootPane;
import javax.swing.JTextField;
import javax.swing.MutableComboBoxModel;
import javax.swing.SwingUtilities;
import javax.swing.event.EventListenerList;
import javax.swing.event.PopupMenuEvent;
import javax.swing.event.PopupMenuListener;

import no.esito.util.EnumUtil;
import no.g9.client.event.G9SelectedEvent;
import no.g9.client.event.G9SelectionListener;
import no.g9.client.event.G9ValueChangedEvent;
import no.g9.client.event.G9ValueChangedListener;
import no.g9.client.event.G9ValueState;
import no.g9.client.support.G9FieldValue;
import no.g9.exception.G9BaseException;
import no.g9.exception.G9ClientFrameworkException;
import no.g9.message.CRuntimeMsg;
import no.g9.message.Message;
import no.g9.message.MessageSystem;


/**
 * The class representing a combo box.
 */
@SuppressWarnings("rawtypes")
public class G9ComboBox extends JComboBox implements G9FieldValue, G9ValueState {

    /** Flag indicating case sensetive search */
    private boolean caseSensitiveSearch = false;

    /** List of value changed listeners */
    protected EventListenerList vcListeners = new EventListenerList();

    /** The intitial displayed value */
    private Object initialValue = null;

    /** The value when this component gained focus */
    private Object valueWhenGainedFocus = null;

    /** The last selected value */
    private Object selectionReminder = null;

    /** The is searchable property */
    private boolean isSearchable = false;

    /** The search agent used for a searchable combo */
    private SearchAgent searchAgent = null;

    /** The property used to determine if selected event should fire. */
    private boolean dontFire;

    /** Used internaly to tell a searchable combo from an editable combo. */
    private boolean customEdit = false;

    /**
     * Creates a <code>JComboBox</code> with a default data model.
     * The default data model is an empty list of objects.
     * Use <code>addItem</code> to add items. By default the first item
     * in the data model becomes selected.
     *
     * @see javax.swing.DefaultComboBoxModel
     */
    @SuppressWarnings("unchecked")
    public G9ComboBox() {
        super();

        G9ComboBoxModel model = new G9ComboBoxModel();
        setModel(model);
        model.setComboBox(this);
        SelectionManager selectionManager = new SelectionManager();
        setKeySelectionManager(selectionManager);
        addKeyListener(selectionManager);

        setEditor(new G9ComboBoxEditor(this));

        getEditor().getEditorComponent().addFocusListener(new FocusWhenPopupManager());
        addFocusListener(new FocusWhenPopupManager());

        getEditor().getEditorComponent().addFocusListener(new FocusManager());
        addFocusListener(new FocusManager());


        // Determine if selection event should fire.
        addActionListener(new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent e) {
                maybeFireSelectedEvent();
            }
        });

        // Determine if selection event should fire.
        addKeyListener(new KeyAdapter() {

            @Override
            public void keyReleased(KeyEvent e) {
                if (e.getKeyCode() == KeyEvent.VK_ENTER)  {
                    maybeFireSelectedEvent();
                } else if (e.getKeyCode() == KeyEvent.VK_ESCAPE) {
                    setSelectedItem(selectionReminder);
                }
            }

        });

        getEditor().getEditorComponent().addKeyListener(new KeyAdapter() {

            @Override
            public void keyReleased(KeyEvent e) {
                if ((e.getKeyCode() == KeyEvent.VK_ESCAPE)) {
                    setSelectedItem(selectionReminder);
                }
            }

        });

        // For searchable, not editable comboes, special handeling if
        // selection is made, and editor is empty.
        addPopupMenuListener(new PopupMenuListener() {

            @Override
            public void popupMenuWillBecomeVisible(PopupMenuEvent e) {
                // Not implemented
            }

            @Override
            public void popupMenuWillBecomeInvisible(PopupMenuEvent e) {
                if (isSearchable() && !isCustomEditable()) {
                    getEditor().setItem(getSelectedItem());
                }
            }

            @Override
            public void popupMenuCanceled(PopupMenuEvent e) {
                // Not implemented
            }

        });

    }

    /**
     * Fires a g9 value changed event.
     *
     * @param id the type of event
     * @param oldValue the old value
     * @param newValue the new value
     */
    protected void fireG9ValueChangedEvent(int id, Object oldValue, Object newValue) {
        Object[] listeners = vcListeners.getListenerList();
        G9ValueChangedEvent e = null;

        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (e == null) {
                e = new G9ValueChangedEvent(this, id, oldValue, newValue);
            }
            if (listeners[i] == G9ValueChangedListener.class) {
                switch (e.getID()) {
                    case G9ValueChangedEvent.VALUE_CHANGED:
                        ((G9ValueChangedListener) listeners[i + 1]).valueChanged(e);
                        break;
                    default:
                        break;
                }
            }
        }
    }


    /**
     * Returns the String value used when displaying the specified object
     *
     * @param anObject the object holding the value
     * @return the value used for displaying the object.
     */
    public String format(Object anObject) {
        return ((ComboBoxRenderer) getRenderer()).format(anObject);
    }

    /**
     * Formats the selected value into the approriate string representation.
     *
     * @return the selected value as a String
     */
    @Override
    public String format() {
        Object selectedItem = getSelectedItem();
        if (selectedItem != null) {
            return ((ComboBoxRenderer) getRenderer()).format(selectedItem);
        }
        return null;
    }

    /**
     * Gets the selected value object. E.g. if this combo displays
     * domainObject.foo, an instance of foo is returned.
     *
     * @return the selected item's display value.
     */
    @Override
    public Object getValue() {
        return getDisplayValue(getSelectedItem());
    }

    @Override
    public Object parse(String formattedString) {
        Object o = null;
        try {
            o = ((ComboBoxRenderer) getRenderer()).parse(formattedString);
        } catch (ParseException e) {
            Object[] args = { getClass(), getName(), formattedString };
            Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.CF_DOCUMENT_PARSEEXCEPTION, args);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9ClientFrameworkException(msg);
        }
        return o;
    }

    /**
     * Selects a value, this is the same as invoking display(value).
     *
     * @param value the value to set
     * @see #display(Object)
     */
    @Override
    public void setValue(Object value) {
        display(value);
    }

    @Override
    public void addValueChangedListener(G9ValueChangedListener listener) {
        vcListeners.add(G9ValueChangedListener.class, listener);

    }

    /**
     * Sets the specified object as the selected object. If the specified object
     * is "null", and the combo is not editable, it will only become the selcted
     * value if the list contains a null value, otherwise no selection is made.
     * If the specified Object is a String, the corresponding item will be
     * selected (if it exists). Note that invoking this method does <em>not</em>
     * trigger a selected event. If you want the selected event to be triggered,
     * use {@link #setSelectedItem(Object)}.
     *
     * @param anObject the object to select.
     */
    @Override
    public void display(Object anObject) {

        if (anObject instanceof String) {
            Object item = getItem((String) anObject);
            if (item != null) {
                anObject = item;
            } else if (!isCustomEditable() && getIndexOfNullValue() < 0) {
                return;
            }
        }

        if (!hasItem(anObject) && !isCustomEditable()) {
            if (anObject == null) {
                anObject = getItemAt(0);
            } else {
                return;
            }
        }

        dontFire = true;
        setSelectedItem(anObject);
        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
                dontFire = false;
            }
        });

    }

    /**
     * Returns the index of the corresponding item
     *
     * @param itemAsString - the item
     * @return the item index or -1 if no such item is found
     */
    private int getItemIndexFromString(String itemAsString) {
        // Make sure the string is properly formatted.
        itemAsString = ((ComboBoxRenderer) getRenderer()).format(itemAsString);

        int index = -1;
        for (int i = 0; i < getItemCount() && index == -1; i++) {
            String displayString = format(getItemAt(i));
            if (displayString != null && displayString.equals(itemAsString)) {
                index = i;
            }
        }
        return index;
    }

    /**
     * Returns the index of the first null value in the list, or -1 if no such
     * element is found.
     *
     * @return the index of null value.
     */
    private int getIndexOfNullValue() {
        int nullIndex = -1;
        for (int i = 0; i < getItemCount() && nullIndex == -1; i++) {
            Object o = getItemAt(i);
            if (o == null) {
                nullIndex = i;
            }
        }
        return nullIndex;
    }

    @Override
    public Object getInitialValue() {
        return initialValue;
    }

    @Override
    public boolean isChanged() {
        Object currentSelection = getSelectedItem();
        // special handling of empty, editable comboboxes...
        if (isEditable() && getItemCount() == 0) {
            currentSelection = editor.getItem();
        }
        if (initialValue == null) {
            return currentSelection != null;
        }
        return !currentSelection.equals(initialValue);
    }


    @Override
    public Object getSelectedItem() {
        if (!isEditable()) {
            return super.getSelectedItem();
        }

        String itemValue = ((G9TextField) getEditor().getEditorComponent()).getText();
        Object item = getItem(itemValue);
        if (item == null) {
            item = ((G9TextField) getEditor().getEditorComponent()).getValue();
        }
        return item;
    }

    /**
     * Resets the current displayed value and the state of the combo box.<br>
     * It the combo list contains any elements, the first element is chosen.
     * Otherwise, if the combo is editable, the editfield is cleared.
     */
    public void reset() {
        if (getItemCount() > 0) {
            initialValue = getItemAt(0);
            valueWhenGainedFocus = initialValue;
            selectionReminder = initialValue;
            setSelectedIndex(0);
        } else if (isCustomEditable()) {
            initialValue = null;
            valueWhenGainedFocus = initialValue;
            selectionReminder = initialValue;
            setSelectedItem(null);
        }
    }

    @Override
    public void resetState() {
        initialValue = getSelectedItem();
        valueWhenGainedFocus = initialValue;
        selectionReminder = initialValue;
    }

    /**
     * {@inheritDoc}
     * <p>
     * If the specified initial value is "null", and this the combo is not
     * editable, and the list does not contain "null", the first item is choosen
     * as the initial value. If the specified value is a String, the list is
     * searched for an item that matches the string. If found, that item is
     * used.
     */
    @Override
    public void setInitialValue(Object value) {
        if (value instanceof String) {
            Object item = getItem((String) value);
            if (item != null) {
                value = item;
            } else if (!isCustomEditable() && getIndexOfNullValue() < 0) {
                return;
            }
        }
        if (!hasItem(value) && !isCustomEditable()) {
            if (value == null) {
                value = getItemAt(0);
            } else {
                return;
            }
        }
        initialValue = value;
        valueWhenGainedFocus = initialValue;
        selectionReminder = value;
    }

    /**
     * Get's the item that matches the specified String.
     *
     * @param stringValue the string representation of the item
     * @return the item matching the string
     */
    private Object getItem(String stringValue) {
        Object item = getItemAt(getItemIndexFromString(stringValue));
        if (item == null && !isCustomEditable()) {
            item = getItemAt(getIndexOfNullValue());
        }

        return item;
    }


    /**
     * Internal use.
     * <p> Sets the select all on focus property.
     *
     * @param selectAllOnFocus (missing javadoc)
     */
    public void setSelectAllOnFocus(boolean selectAllOnFocus) {
        if (isEditable()) {
            if (getEditor() instanceof G9ComboBoxEditor) {
                ((G9ComboBoxEditor) getEditor())
                        .setSelectAllOnFocus(selectAllOnFocus);
            }
        }
    }

    /**
     * Returns the string representation of the selected item (viz. the
     * displayed text).
     *
     * @return the selected item's displayed text.
     */
    public String getSelectedItemAsString() {
        String selection = null;
        if (isEditable()) {
            selection = ((JTextField) getEditor().getEditorComponent())
                    .getText();
        } else {
            selection = ((ComboBoxRenderer) getRenderer())
                    .format(getSelectedItem());
        }
        return selection;
    }

    /**
     * Sets the selected item to be the specified value.
     *
     * @param value the value to select
     * @see JComboBox#setSelectedItem(Object)
     * @deprecated This method is now obsolete and should not be used. Use
     *             {@link #setSelectedItem(Object)} instead.
     */
    @Deprecated
    public void setSelectedItemG9(Object value) {
        setSelectedItem(value);
    }


    @SuppressWarnings("unchecked")
    @Override
    public void addItem(Object anObject) {
        dontFire = true;
        super.addItem(anObject);
        reset();
        dontFire = false;
    }


    /**
     * Adds a collection of items to the item list. This method works only if
     * the <code>JComboBox</code> uses a mutable data model.
     * <p>
     * <strong>Warning:</strong> Focus and keyboard navigation problems may
     * arise if you add duplicate String objects. A workaround is to add new
     * objects instead of String objects and make sure that the toString()
     * method is defined, see {@link JComboBox#addItem(Object)}
     *
     * @param items the Collection of objects to add to the list
     * @see MutableComboBoxModel
     */
    public void addItem(Collection<? extends Object> items) {
        if (!(getModel() instanceof MutableComboBoxModel)) {
            throw new G9BaseException("Cannot use this method with a non-Mutable data model.");
        }

        dontFire = true;
        ((G9ComboBoxModel) getModel()).addElements(items);
        reset();
        dontFire = false;
    }



    /**
     * Adds a collection of items to the combo box. Forwards the call to
     * {@link #addItem(Collection)}.
     *
     * @param items a collection of items.
     * @see #addItem(Collection)
     */
    public void addAllItems(Collection<? extends Object> items) {
        addItem(items);
    }


    /**
     * Adds an enumeration to this combo.
     *
     * @param enumTitle the title of the enum (current value)
     * @param enumType the type of enum
     */
    public void addItem(String enumTitle, Class<?> enumType) {
        String enumName = enumType.getName();
        Object anEnum= EnumUtil.getEnumObject(enumName, enumTitle);
        addItem(anEnum);
    }


    /**
     * Sets the input justification
     *
     * @param justification value specifying the input justification. See
     *            {@link JTextField#setHorizontalAlignment(int)} for a list of
     *            valid parameters.
     */
    public void setInputJustification(int justification) {
        ((G9TextField) getEditor().getEditorComponent())
                .setInputJustification(justification);

    }

    /**
     * Sets the output justification.
     *
     * @param outputJustification value specifying the output justification. See
     *            {@link JTextField#setHorizontalAlignment(int)} for a list of
     *            valid parameters.
     */
    public void setOutputJustification(int outputJustification) {
        ((ComboBoxRenderer) getRenderer())
                .setHorizontalAlignment(outputJustification);

    }

    /**
     * Sets the horizontal alignment.
     *
     * @param alignment the alignment, see
     *            {@link JTextField#setHorizontalAlignment(int)} for a list of
     *            valid parameters.
     */
    public void setHorizontalAlignment(int alignment) {
        ((ComboBoxRenderer) getRenderer())
                .setHorizontalAlignment(alignment);

    }

    /**
     * Sets the isSorted property. If set to true, the contents of this combo
     * will be sorted in the natural, descending order.
     *
     * @param isSorted (missing javadoc)
     */
    public void setSorted(boolean isSorted) {
        ((G9ComboBoxModel) getModel()).setSorted(isSorted);

    }

    /**
     * Gets the value used for displaying <code>anObject</code>.
     *
     * @param anObject the object to get the display value of
     * @return the value used for displaying the specified object.
     */
    protected Object getDisplayValue(Object anObject) {
        return ((ComboBoxRenderer) getRenderer())
                .getDisplayValue(anObject);
    }

    /**
     * The class for managing selections by key stroke on a non-searchable combo
     * box. The last typed character is the character used for searching,
     * ignoring case. Thus in a combo containing month-names, typing "jun" will
     * select "November".
     */
    public class SelectionManager extends KeyAdapter implements
            JComboBox.KeySelectionManager {

        /**
         * {@inheritDoc}
         * Selects the next match, ignoring key.
         * @param aKey the character used for searching
         * @param aModel the model containing the data to search for.
         */
        @Override
        public int selectionForKey(char aKey, ComboBoxModel aModel) {
            if (!(aKey == KeyEvent.CHAR_UNDEFINED
                    || Character.isISOControl(aKey))) {
                showPopup();
            }

            aKey = Character.toLowerCase(aKey);

            int currentSelection = -1;
            Object selectedItem = getModel().getSelectedItem();
            // get current selection index.
            if (selectedItem != null) {
                for (int i = 0; i < getModel().getSize(); i++) {
                    if (selectedItem == getModel().getElementAt(i)) {
                        currentSelection = i;
                        break;
                    }
                }
            }



            // search from current position + 1
            for (int i = currentSelection + 1; i < getModel().getSize(); i++) {
                String element = format(getModel().getElementAt(i));
                if (element != null && element.length() > 0) {

                    element = element.toLowerCase();
                    if (element.charAt(0) == aKey) {
                        return i;
                    }
                }
            }

            // search from top
            for (int i = 0; i < currentSelection; i++) {
                String element = format(getModel().getElementAt(i));
                if (element != null && element.length() > 0) {
                    element = element.toLowerCase();
                    if (element.charAt(0) == aKey) {
                        return i;
                    }
                }
            }

            // no element found.
            return -1;

        }

    }

    /**
     * Sets the combo to be editable. Note that even though a combo is set to
     * uneditble, it still might be editable if it is searchable. The difference
     * is then determined in how the search is performed. In either case,
     * editing the combo field does not affect the combo list.
     *
     * @param aFlag a boolean value where true indicates that this combo is
     *            editable.
     */
    @Override
    public void setEditable(boolean aFlag) {
        if (aFlag) {
            super.setEditable(aFlag);
            customEdit = aFlag;
        } else {
            super.setEditable(isSearchable);
            customEdit = aFlag;
        }
    }

    /**
     * Returns the number of possible search hits using the specified search
     * string. E.g. if the list contains mont-names, the string "Ju" will give
     * two possible hits (mathces both "June" and "Juli"). Note that even if a
     * search string hits more than one item in the list, it does not mean that
     * an item can not be uniquly identified using the search string. E.g. if
     * the list contains (among other items) the texts "foo" and "foo bar", the
     * search string "foo" gives two hits, but is sufficently accurate to
     * uniquely identify the "foo" item. The second parameter determines if case
     * is significant or not.
     *
     * @param searchString the string used for searching.
     * @param caseSensitive a boolean value where false indicates that case is
     *            not significant.
     * @return the number of itemes starting with the specified string
     */
    public int getNumberOfPossibleHits(String searchString,
            boolean caseSensitive) {
        int hitCount = 0;
        if (!caseSensitive) {
            searchString = searchString.toLowerCase();
        }
        for (int i = 0; i < getItemCount(); i++) {
            String item = format(getItemAt(i));
            if (!caseSensitive && item != null) {
                item = item.toLowerCase();
            }
            if (item != null && item.startsWith(searchString)) {
                hitCount++;
            }
        }
        return hitCount;
    }

    /**
     * Returns true if an item (viz. the string represenatation of that item)
     * equals the search string.
     *
     * @param searchString the string used for searching
     * @return <code>true</code> if an exact match is found.
     */
    public boolean isSearchHit(String searchString) {
        boolean searchHit = false;
        for (int i = 0; i < getItemCount() && !searchHit; i++) {
            String item = format(getItemAt(i));
            searchHit = (item != null && item.equals(searchString))
                    || (item == null && (searchString == null || searchString
                            .equals("")));
        }
        return searchHit;
    }

    /**
     * The search agent used for managing search in the combo The search is
     * either case sensitive or not, depending on the
     * {@link G9ComboBox#isCaseSenstitiveSearch()} property.
     */
    private class SearchAgent extends KeyAdapter {
        /** The editor component */
        protected JTextField editor;

        /** Constructs a new search agent. */
        private SearchAgent() {
            editor = (JTextField) getEditor().getEditorComponent();
        }

        /**
         * Invoked when a key is pressed. Conditionally decides to show popup.
         *
         * @param e the key event.
         */
        @Override
        public void keyPressed(KeyEvent e) {
            char ch = e.getKeyChar();
            if ((ch == KeyEvent.CHAR_UNDEFINED
                    || Character.isISOControl(ch) || e.getModifiers() != 0)
                    && !isPopupTrigger(ch)) {
                return;
            }
            // SUP-283
//            e.consume();
            showPopup();
        }

        /**
         * Invoked when a key is released. Conditionally performs search.
         *
         * @param e the key event.
         */
        @Override
        public void keyReleased(KeyEvent e) {
            char ch = e.getKeyChar();
            if (ch == KeyEvent.CHAR_UNDEFINED
                    || Character.isISOControl(ch)
                    || e.getModifiers() != 0
                    && (editor.getText() != null && editor.getText()
                            .length() > 0)) {
                return;
            }
            e.consume();
            search();
        }

        /**
         * Determine if the undefined char should trigger the popup. This is
         * true for delete and backspace and maybe others.
         *
         * @param ch the char
         * @return true if popup should show
         */
        private boolean isPopupTrigger(char ch) {
            boolean isPopupTrigger = false;
            switch (ch) {
                // YES - FALL TRHOUGH.
                case KeyEvent.VK_DELETE :
                case KeyEvent.VK_BACK_SPACE :
                    isPopupTrigger = true;
                    break;
                default : isPopupTrigger = false;
            }
            return isPopupTrigger;
        }

        /**
         * Perfoms a search in the combo box list based on the current caret
         * position, and displays the search.
         */
        public void search() {
            int pos = editor.getCaretPosition();

            String searchString = editor.getText();
            if (!isCustomEditable()) {
                searchString = getFirstHitSearchString(searchString);
            }

            if (searchString.length() == 0) {
                if (!isCustomEditable()) {
                    editor.setText("");
                    if (getItemIndexFromString(searchString) >= 0) {
                        setSelectedIndex(getItemIndexFromString(searchString));
                    }
                }
                return;
            }

            if (pos > searchString.length()) {
                pos = searchString.length();
            }

            for (int itemIndex = 0; itemIndex < getItemCount(); itemIndex++) {
                String item = format(getItemAt(itemIndex));
                if (item == null) {
                    continue;
                }
                if (isCaseSenstitiveSearch()) {
                    if (!item.startsWith(searchString)) {
                        continue;
                    }
                } else {
                    if (!item.toLowerCase().startsWith(searchString.toLowerCase())) {
                        continue;
                    }
                }
                // If here, we have a hit
                int currentSelection = getSelectedIndex();
                if (itemIndex != currentSelection) {
                    setSelectedIndex(itemIndex);
                }
                editor.setText(item);
                editor.setCaretPosition(item.length());
                editor.moveCaretPosition(pos);
                break;
            }

        }

        /**
         * Returns the maximum string that will give a search hit.
         *
         * @param searchString the search string to use
         * @return the new maximum lenght search string
         */
        private String getFirstHitSearchString(String searchString) {

            if (!isCaseSenstitiveSearch()) {
                searchString = searchString.toLowerCase();
            }
            while (searchString.length() > 0) {
                for (int i = 0; i < getItemCount(); i++) {
                    String item = format(getItemAt(i));

                    if (item != null) {
                        if (!isCaseSenstitiveSearch()) {
                            item = item.toLowerCase();
                        }
                        if (item.startsWith(searchString)) {
                            return searchString;
                        }
                    }
                }
                searchString = searchString.substring(0, searchString
                        .length() - 1);
            }

            return searchString;
        }
    }

    private class FocusWhenPopupManager extends FocusAdapter {

        @Override
        public void focusLost(FocusEvent e) {
            if (menuOrToolbar(e.getOppositeComponent())) {
                return;
            }
            if (!isPopupVisible()) {
                return;
            }
            String currentValue = getSelectedItemAsString();

            if (!isCustomEditable() && isSearchable() && !isSearchHit(currentValue)) {
                setSelectedItem(selectionReminder);
                getEditor().setItem(selectionReminder);
                return;
            }
            int oldSelection = getIndexForItem(selectionReminder);
            int newSelection = getSelectedIndex();
            if (oldSelection != newSelection) {
                fireG9SelectedEvent(oldSelection, newSelection);
            }
        }

        /**
         * Check if the componet that lost focus is a menu or toolbar component
         *
         * @param opposite the component that lost focus
         * @return <code>true</code> if the component is a menu or toolbar
         *         component
         */
        private boolean menuOrToolbar(Component opposite) {
            return opposite == null
                    || opposite instanceof JRootPane
                    || (opposite instanceof G9Button && ((G9Button) opposite)
                            .isInToolbar());

        }
    }

    /**
     * The focus manager used by the combo and editor component. When focus is
     * gained, the current selected value is stored. If this is a non-mutable
     * combo, the value is checked when focus is lost, and possibly restored. If
     * the selected value is changed, a value changed event is fired.
     */
    private class FocusManager extends FocusAdapter {

        /**
         * Invoked when the component looses focus. Validates the selection and
         * fires a value changed event if the value is changed.
         *
         * @param e the focus event.
         */
        @Override
        public void focusLost(FocusEvent e) {
            if (menuOrToolbar(e.getOppositeComponent())) {
                return;
            }
            String currentValue = getSelectedItemAsString();

            if (!isCustomEditable() && isSearchable()
                    && !isSearchHit(currentValue)) {
                setSelectedItem(valueWhenGainedFocus);
                getEditor().setItem(valueWhenGainedFocus);
                return;
            }

            if ("".equals(currentValue) && !isSearchHit(currentValue)) {
                currentValue = null;
            }
            String oldValue = format(valueWhenGainedFocus);



            if ((oldValue != null ^ currentValue != null)
                    || oldValue != null && !oldValue.equals(currentValue)) {
                fireG9ValueChangedEvent(
                        G9ValueChangedEvent.VALUE_CHANGED,
                        valueWhenGainedFocus, getSelectedItem());
            }
        }

        /**
         * Invoked when the component gains focus. Keeps the selection value in
         * order to validate on focus lost.
         *
         * @param e the focus event.
         */
        @Override
        public void focusGained(FocusEvent e) {
            if (menuOrToolbar(e.getOppositeComponent())) {
                return;
            }
            if (getItemCount() > 0) {
                valueWhenGainedFocus = getSelectedItem();
            } else if (isEditable()){
                valueWhenGainedFocus = ((G9TextField) getEditor().getEditorComponent()).getValue();
            } else {
                valueWhenGainedFocus = null;
            }
            selectionReminder = valueWhenGainedFocus;
        }

        /**
         * Check if the componet that lost focus is a menu or toolbar component
         *
         * @param opposite the component that lost focus
         * @return <code>true</code> if the component is a menu or toolbar
         *         component
         */
        private boolean menuOrToolbar(Component opposite) {
            return opposite == null
                    || opposite instanceof JRootPane
                    || (opposite instanceof G9Button && ((G9Button) opposite)
                            .isInToolbar());

        }
    }

    /**
     * Sets whether searching is case sensitive or not (only applies to
     * searchable combos). Default value is false.
     *
     * @param aFlag a boolean value where true indicates that searching is case
     *            sensitive.
     */
    public void setCaseSensitiveSearch(boolean aFlag) {
        caseSensitiveSearch = aFlag;
    }

    /**
     * Returns the value used when determening if searching is case sensitive.
     *
     * @return the caseSensitiveSearch property
     * @see #setCaseSensitiveSearch(boolean)
     */
    public boolean isCaseSenstitiveSearch() {
        return caseSensitiveSearch;
    }

    /**
     * Sets this combo as searchable. Note that setting a combo to searchable,
     * implies that it is also editable (invoking <code>isEditable</code> on a
     * searchable combo results in <code>true</code>. In order to make it true
     * editable, one must invoke the setEditable method.
     *
     * @param searchable a boolean value, true indicating that this combo is
     *            searchable.
     */
    public void setSearchable(boolean searchable) {
        isSearchable = searchable;
        if (searchable) {
            super.setEditable(searchable);
        } else {
            setEditable(isCustomEditable());
        }
        isSearchable = searchable;
        JTextField textEditor = (JTextField) getEditor().getEditorComponent();
        if (searchable) {
            if (searchAgent == null) {
                searchAgent = new SearchAgent();
            }
            textEditor.addKeyListener(searchAgent);
        } else {
            if (searchAgent != null) {
                textEditor.removeKeyListener(searchAgent);
            }
        }
    }

    @Override
    public void setSelectedItem(Object anObject) {
        if (hasItem(anObject) || isCustomEditable()) {
            super.setSelectedItem(anObject);
            editor.setItem(anObject);
        }
    }

    private boolean hasItem(Object anObject) {
        if (anObject == null) {
            return getIndexOfNullValue() >= 0;
        } else if (anObject instanceof String) {
            return getItemIndexFromString((String) anObject) >= 0;
        } else {
            return getIndexForItem(anObject) >= 0;
        }
    }

    /**
     * Returns the isSearchable property
     *
     * @return <code>true</code> if this combo is searchable.
     * @see #setSearchable(boolean)
     */
    public boolean isSearchable() {
        return isSearchable;
    }

    /**
     * Returns the isCustomEditable property
     *
     * @return <code>true</code> if this combo is editable
     * @see #setEditable(boolean)
     */
    public boolean isCustomEditable() {
        return customEdit;
    }

    /**
     * Adds a g9 selected listener to this component.
     *
     * @param listener the listener
     */
    public void addG9SelectionListener(G9SelectionListener listener) {
        listenerList.add(G9SelectionListener.class, listener);
    }
    private void maybeFireSelectedEvent() {
        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
                // if popup is showing, user is traversing through selections. Don't fire!
                if (isPopupVisible() || dontFire) {
                    return;
                }
                if (!(selectionReminder == null && getSelectedItem() == null)) {
                    int oldIndex = getIndexForItem(selectionReminder);
                    int newIndex = getSelectedIndex();
                    if (oldIndex != newIndex) {
                        fireG9SelectedEvent(oldIndex, newIndex);
                    }
                }
            }
        });
    }

    private int getIndexForItem(Object anItem) {
        int index = -1;
        for (int i = 0; i < getItemCount() && index == -1; i++) {
            Object o = getItemAt(i);
            if (o != null && o.equals(anItem)) {
                index = i;
            }
        }
        return index;
    }

    /**
     * Fires a g9 selection event
     *
     * @param oldSelection the index of the old selection
     * @param newSelection the index of the new selection
     */
    public void fireG9SelectedEvent(int oldSelection, int newSelection) {
        selectionReminder = getSelectedItem();
        Object[] listeners = listenerList.getListenerList();
        G9SelectedEvent e = new G9SelectedEvent(this, oldSelection, newSelection);

        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == G9SelectionListener.class) {
                ((G9SelectionListener) listeners[i + 1]).selectionChanged(e);
            }
        }
    }

    /**
     * Internal use! Will not be implemented until editable listblock is
     * supported. Sets the default listblock cell editor.
     *
     * @param tx the cell editor.
     */
    public void setUsedAsListblockEditor(DefaultCellEditor tx) {
        // not implemented
    }

    /**
     * Internal use! Will not be implemented until editable listblock is
     * supported. Signals that edit is about to begin.
     */
    public void editingAboutToBegin() {
        // Not implemented.
    }

    /**
     * Internal use, returns the dontFire flag. Used when updating expressions,
     * to determine wheter the target combo should fire a selected event or not.
     *
     * @return the dontFire flag.
     */
    public boolean isDontFire() {
        return dontFire || isPopupVisible();
    }

}


