/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.component;

import java.awt.Component;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.Collections;
import java.util.List;
import java.util.ListIterator;

import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.Icon;
import javax.swing.JComponent;
import javax.swing.JRadioButton;
import javax.swing.RootPaneContainer;

import no.g9.client.event.G9ValueChangedListener;
import no.g9.client.event.G9ValueState;
import no.g9.client.support.DialogBlocker;

/**
 * The class representing a radio button
 */
public class G9RadioButton extends JRadioButton implements G9ValueState {

    /** The radio button group this button belongs to */
    private G9ButtonGroup group;
    
    /**
     * Creates an initially unselected radio button with no set text.
     */
    public G9RadioButton() {
        this(null, null, false);
    }

    /**
     * Creates an unselected radio button with the specified text.
     * 
     * @param text  the string displayed on the radio button
     */
    public G9RadioButton(String text) {
        this(text, null, false);
    }

    /**
     * Creates a radio button with the specified text and selection state.
     * 
     * @param text the string displayed on the radio button
     * @param selected if true, the button is initially selected; otherwise, the
     *            button is initially unselected
     */
    public G9RadioButton(String text, boolean selected) {
        this(text, null, selected);
    }

    /**
     * Creates a radiobutton where properties are taken from the Action
     * supplied.
     * 
     * @param a the action
     * @since 1.3
     */
    public G9RadioButton(Action a) {
        this();
        setAction(a);
    }

    /**
     * Creates an initially unselected radio button with the specified image but
     * no text.
     * 
     * @param icon the image that the button should display
     */
    public G9RadioButton(Icon icon) {
        this(null, icon, false);
    }

    /**
     * Creates a radio button with the specified image and selection state, but
     * no text.
     * 
     * @param icon the image that the button should display
     * @param selected if true, the button is initially selected; otherwise, the
     *            button is initially unselected
     */
    public G9RadioButton(Icon icon, boolean selected) {
        this(null, icon, selected);
    }

    /**
     * Creates a radio button that has the specified text and image, and that is
     * initially unselected.
     * 
     * @param text the string displayed on the radio button
     * @param icon the image that the button should display
     */
    public G9RadioButton(String text, Icon icon) {
        this(text, icon, false);
    }

    /**
     * Creates a radio button that has the specified text, image, and selection
     * state.
     * 
     * @param text the string displayed on the radio button
     * @param icon the image that the button should display
     * @param selected the selected state of this button
     */
    public G9RadioButton(String text, Icon icon, boolean selected) {
        super(text, icon, selected);
        // The insets needed are already calcluated by g9.
        setBorder(BorderFactory.createEmptyBorder());
        
        addMouseListener(new MouseAdapter() {
            // If dialog is in a blocked state, we still might need to process 
            // this mouse event in order for a button to be clicked.
            // Typically this situation occurs when a focus lost (or
            // similar) event is triggered by this button being pressed, and
            // that event is resulting in a dialog block.
            @Override
            public void mouseReleased(MouseEvent e) {
                Component parent = getParent();
                while (parent != null && !(parent instanceof RootPaneContainer)) {
                    parent = parent.getParent();
                }
                boolean wantEvent = DialogBlocker.isBlocked(parent)
                && getModel().isPressed()
                && contains(e.getPoint());
                if (wantEvent) {
                    doClick();
                }
            }
        });
    }
    
    
    
    /**
	 * @return the Parent button group
	 */
	public G9ButtonGroup getGroup() {
		return group;
	}

    /**
     * Set the parent button group
     * 
     * @param group the parent button group to set
     */
	void setGroup(G9ButtonGroup group) {
		this.group = group;
	}

	@Override
    public void setSelected(boolean selected) {
        super.setSelected(selected);
        if (group != null) {
            group.updateSelection();
        }
    }

    @Override
    public void addValueChangedListener(G9ValueChangedListener listener) {
        if (group != null) {
            group.addValueChangedListener(listener);
        }
        
    }

    @Override
    public void resetState() {
        if (group != null) {
            group.resetState();
        }
        
    }

    @Override
    public boolean isChanged() {
        if (group != null) {
            return group.isChanged();
        } 
        return false;
    }

    @Override
    public void display(Object o) {
        if (group != null) {
            group.display(o);
        }
        
    }

    @Override
    public Object getInitialValue() {
        if (group != null) {
            return group.getInitialValue();
        } 
        return null;
    }

    @Override
    public void setInitialValue(Object value) {
        if (group != null) {
            group.setInitialValue(value);
        }      
    }

    /**
     * {@inheritDoc} Set up left, down, right, up as radiogroup internal focus
     * traversal keys.
     * 
     * @see javax.swing.JComponent#processKeyEvent(java.awt.event.KeyEvent)
     */
	@Override
    protected void processKeyEvent(KeyEvent e) {
		super.processKeyEvent(e);		
		if (e.getID() == KeyEvent.KEY_PRESSED
				&& (e.getKeyCode() == KeyEvent.VK_RIGHT || e.getKeyCode() == KeyEvent.VK_DOWN)) {
			setFocus(+1);
			return;
		}
		if (e.getID() == KeyEvent.KEY_PRESSED
				&& (e.getKeyCode() == KeyEvent.VK_LEFT || e.getKeyCode() == KeyEvent.VK_UP)) {
			setFocus(-1);
			return;
		}
	}

    /**
     * Request fokus for the first showing objekt that is at least
     * <code>move</code> elements from this.
     * 
     * @param move Number of elements to move fokus.
     */
	private void setFocus(int move) {
		List<AbstractButton> l = Collections.list(group.getElements());
		ListIterator<AbstractButton> it= l.listIterator();
		while(it.hasNext()) {
			Object o = it.next();
			if(o instanceof JComponent && ((JComponent) o).isShowing()) 
				continue;
			it.remove();
		}
		int index= (l.indexOf(this) + move)%l.size();
		if(index<0) index+=l.size();
		((JComponent)  l.get(index)).requestFocus();		
	}
	
}