/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.component;

import java.awt.Component;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import javax.swing.Icon;
import javax.swing.JTabbedPane;

/**
 * The g9 tab pane manager manages enables showing and hiding of tab components.
 * <em>Note:</em> This class is not thread safe. 
 */
public class G9TabPaneManager {

    /** The pane managed by this tab pane manager */
    private final JTabbedPane tabPane;

    /** The immutable list of tab titles */
    private final List<String> tabTitles;
    
    /** The immutable list of tab components (shown and hidden). */
    private final List<Component> tabComponents;
    
    /** The immutable list of tab icons */
    private final List<Icon> tabIcons;
    
    /** The immutable list of tool tips. */
    private final List<String> toolTips;

    /**
     * Constructs a new tab pane manager for the specified tab pane.
     * @param tabPane
     *            the tab pane that should be managed
     */
    public G9TabPaneManager(JTabbedPane tabPane) {
        this.tabPane = tabPane;
        tabTitles = Collections.unmodifiableList(getShownTitles());
        tabComponents = Collections.unmodifiableList(getShownTabs());
        tabIcons = Collections.unmodifiableList(getShownIcons());
        toolTips = Collections.unmodifiableList(getShownToolTips());
    }

    
    /**
     * Shows or hides the specified tab component.
     * @param tabComponent the tab component to show or hide.
     * @param visible if <code>true</code> the tab is shown, otherwise hidden.
     */
    public void setVisible(Component tabComponent, boolean visible) {
        if (visible) {
            showTab(tabComponent);
        } else {
            hideTab(tabComponent);
        }
    }
    
    
    /**
     * Hide the specified tab component.
     * @param tabComponent the tab component to hide.
     */
    public void hideTab(Component tabComponent) {
        if (absoluteIndex(tabComponent) < 0) {
            throwEx(tabComponent);
        }
        List<Component> shownTabs = getShownTabs();
        int index = shownTabs.indexOf(tabComponent);
        if (index >= 0) {
            tabPane.remove(index);
        }
    }
    
    /**
     * Show the tab with the specified title.
     * @param tabComponent the tab page to show.
     */
    public void showTab(Component tabComponent) {
        int tabIndex = absoluteIndex(tabComponent);
        if (tabIndex < 0) {
            throwEx(tabComponent);
        }
        
        
        if (getShownTabs().contains(tabComponent)) {
            return; 
        }
        
        String title = tabTitles.get(tabIndex);
        Icon icon = tabIcons.get(tabIndex);
        String tip = toolTips.get(tabIndex);
        int insertIndex = getInsertIndex(tabComponent);
        tabPane.insertTab(title, icon, tabComponent, tip, insertIndex);
    }

    /**
     * Get the list of icons.
     * @return the icon list.
     */
    private List<Icon> getShownIcons() {
        int tabCount = tabPane.getTabCount();
        List<Icon> tmp = new ArrayList<Icon>();
        for (int i = 0; i < tabCount; i++) {
            tmp.add(tabPane.getIconAt(i));
        }
        return tmp;
    }
    
    /**
     * Get the list of tool tips.
     * @return the tool tip list.
     */
    private List<String> getShownToolTips() {
        int tabCount = tabPane.getTabCount();
        List<String> tmp = new ArrayList<String>();
        for (int i = 0; i < tabCount; i++) {
            tmp.add(tabPane.getToolTipTextAt(i));
        }
        return tmp;
    }
    
    /**
     * Get the absolute index of the specified tab component.
     * @param tabComponent the tab component.
     * @return the index of the component or -1 if not foune.
     */
    private int absoluteIndex(Component tabComponent) {
        return tabComponent == null ? - 1 : tabComponents.indexOf(tabComponent);
    }
    
    /**
     * Throws an exception due to no such component.
     * @param component the tab component that was not found.
     */
    private void throwEx(Component component) {
        String msg = "Managed tab paned does not contain the component " + component;
        throw new RuntimeException(msg);
    }
    
    /**
     * Get the list of shown titles. The contents of the list will depend upon
     * which tabs are shown.
     * 
     * @return the list of shown titles.
     */
    private List<String> getShownTitles() {
        int tabCount = tabPane.getTabCount();
        List<String> tmp = new ArrayList<String>();
        for (int i = 0; i < tabCount; i++) {
            tmp.add(tabPane.getTitleAt(i));
        }
        return tmp;
    }
    
    /**
     * Get the list of shown tab components.
     * @return
     */
    private List<Component> getShownTabs() {
        return Arrays.asList(tabPane.getComponents());
    }

    /**
     * Get the insert index of the tab component.
     * @param tabComponent
     * @return
     */
    private int getInsertIndex(Component tabComponent) {
        int absoluteIndex = absoluteIndex(tabComponent);
        if (absoluteIndex < 0) {
            throwEx(tabComponent);
        }
        
        Component[] shownTabs = getShownTabs().toArray(new Component[0]);
        for (int i = 0; i < shownTabs.length; i++) {
            int currentShownIndex = absoluteIndex(shownTabs[i]);
            if (absoluteIndex < currentShownIndex) {
                return i;
            }
        }
        
        return shownTabs.length;
    }

    @Override
    public String toString() {
        return "G9TabPaneManager tabTitles=" + tabTitles;
    }
}
