/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.component;

import java.awt.Component;
import java.awt.Dimension;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import javax.swing.JComponent;
import javax.swing.JToolBar;
import javax.swing.SwingConstants;

import no.g9.client.support.G9DialogFrame;

/**
 * Internal use!
 * <p>
 * Extends JToolBar with the capability of merging tool bars using the
 * {@link #mergeToolBars(G9DialogFrame)}. (The Java 5.0 swing framework does
 * not include support for multiple toolbars, as you find in e.g. MS Office. Due
 * to this fact, this implementations is not written to suppor multiple
 * toolbars.)
 */
public class G9ToolBar extends JToolBar {

    private static boolean useOldMerge;

	/** Because of serialization */
	private static final long serialVersionUID = 1L;

	/** The application main dialog */
	private G9DialogFrame parentFrame;

	/** The components of this toolbar */
	private List<Component> myComponents;

	/** If set, consecutive separators are reduced to one */
	private boolean removeConsecutiveSeparators = false;

	/** True if the last added element is a spacer */
	private boolean isLastAddedSpacer = false;

	/** True if the toolbar is modal */
	private boolean isModal;

    /**
     * Default constructor. Creates a new G9ToolBar
     *
     * @param parentFrame the owner of this tool bar.
     */
	public G9ToolBar(G9DialogFrame parentFrame) {
		super();
		if(parentFrame == null)
			throw new IllegalArgumentException("Parent frame can not be null");

		this.parentFrame = parentFrame;
	}

    /**
     * Constructor. Creates a new G9ToolBar
     *
     * @param parentFrame the owner of this tool bar
     * @param myComponents the components on this tool bar
     */
	public G9ToolBar(G9DialogFrame parentFrame, List<Component> myComponents) {
		super();
		if(parentFrame == null) {
			throw new IllegalArgumentException("Parent frame can not be null");
		}
		this.parentFrame = parentFrame;

		this.myComponents = myComponents;
	}

    /**
     * Returns an array containing all components added by using the
     * {@link #add(Component)} method.
     *
     * @return an array containing all the components of this toolbar.
     */
	public synchronized Component[] getG9Components() {
		if (myComponents == null) {
			return new Component[0];
		}
		Component[] retVal = new Component[myComponents.size()];
        return myComponents.toArray(retVal);
	}

    /**
     * If set to <code>true</code>, consecutive separators are reduced to one
     *
     * @param removeConsecutiveSeparators The flag to set.
     */
	public synchronized void setRemoveConsecutiveSeparators(boolean removeConsecutiveSeparators) {
		this.removeConsecutiveSeparators = removeConsecutiveSeparators;
	}

    /**
     * Return true if consecutive separators are to be reduced to one
     *
     * @return Returns true if consecutive separators are to be reduced to one.
     */
	public synchronized boolean isRemoveConsecutiveSeparators() {
		return removeConsecutiveSeparators;
	}

    /**
     * Enable or disable all toolbar items.
     *
     * @param enable if <code>true</code> enable, else disable.
     */
    public synchronized void setEnabledWholeToolBar(boolean enable) {
        if(parentFrame == null || parentFrame.getG9ToolBar() == null)
            return; //No components to disable

        Component mainElements[] = getG9Components();

        for(int i = 0; i < mainElements.length; i++) {
            mainElements[i].setEnabled(enable &&
                        parentFrame.isEnabledComponent(mainElements[i]));
        }
    }


    /**
     * Enable or disable the modal status of this tool bar. Note that modality
     * is only meaningfull for the application main frame tool bar. For other
     * internal dialogs, the behavior is to enable or disable the internal
     * dialogs own tool bar.
     *
     * @param isModal Puts this menu bar into modal mode if <code>true</code>
     *            otherwise the menu bar is put back to normal mode.
     */
	public synchronized void setModal(boolean isModal) {
		this.isModal = isModal;

		if(parentFrame == null || parentFrame.getG9ToolBar() == null)
			return; //No components to disable

		Component mainElements[] = getG9Components();

		for(int i = 0; i < mainElements.length; i++) {
			mainElements[i].setEnabled(!isModal &&
						parentFrame.isEnabledComponent(mainElements[i]));
		}
	}

    /**
     * Return <code>true</code> if this menubar is modal otherwise
     * <code>false</code>.
     *
     * @return Returns <code>true</code> if this menubar is modal otherwise
     *         <code>false</code>.
     */
	public synchronized boolean isModal() {
		return isModal;
	}

	/**
     * Merges the tool bar held by the argument dialog with the
     * applicaition tool bar.
     *
     * <p>
     * Tool bars are merged using one of the following algoritms:
     * <ul>
     *   <li>If the <code>useOldMerge</code> property is
     *       <code>false</code>, the merge position is intially last in the
     *       application tool bar. Then, the dialog's tool bar is scanned from
     *       the beginnig. For each component the following algoritm is applied:
     *     <ol>
     *       <li> If the component matches one in the application tool bar, the
     *            application tool bar component is replaced with the dialog's
     *            tool bar component, and merge position is set to the position
     *            of the component in the application tool bar.
     *
     *       <li> Else, if the component does not match a component in the
     *            application tool bar, the component is inserted at the current
     *            merge position, shifting subsequent components to the right.
     *     </ol>
     *     <p>Eksample: The application tool bar has the following components<br>
     *     {A, B, C, D} and the dialog tool bar has <br>
     *     {X, A, Y, B, D , Z}, the result is:<br>
     *     {A, Y, B, C, D, Z, X}
     *
     *   <li>If the <code>useOldMerge</code> property is <code>true</code>
     *       the following algoritm is applied:
     *     <ol>
     *       <li>Substitute components from the application tool bar with
     *           components from the dialog tool bar that match.
     *       <li>Add all components in the application tool bar
     *       <li>Add all components in the dialog tool bar
     *     </ol>
     *   <p>This algoritm will place un-merged application tool bar components
     *   first. Example:
     *
     *   <p>Application tool bar: {A, B, C}<br>
     *   Dialog tool bar: {A, X, B, Y} <br>
     *   Merged tool bar: {C, A, X, B, Y}
     *
     * </ul>
     *
     * @param activeDialog the dialog that holds the tool bar to be merged.
     */
	public void mergeToolBars(G9DialogFrame activeDialog) {
		merge(activeDialog);
		validate();
		repaint();
	}

    /**
     * @param activeDialog the currently active dialog
     * @see #mergeToolBars(G9DialogFrame) for details.
     */
	private synchronized void merge(G9DialogFrame activeDialog) {

		synchronized (getTreeLock()) {

			removeAll();

            if (!G9ToolBar.isUseOldMerge()) {
                Component[] bar = getMergedComponents(activeDialog);
                if (bar != null) {
                    display(bar);
                    return;
                }
            }

			Component mainElements[] = getG9Components();
			Component activeElements[] = (activeDialog != null && activeDialog
					.getG9ToolBar() != null) ? activeDialog
					.getG9ToolBar().getG9Components()
					: new Component[0];

			if (mainElements.length == 0) {
				display(activeElements);
				return;
			}

			if (activeElements.length == 0) {
				display(mainElements);
				return;
			}

			for (int i = 0; i < mainElements.length; i++) {
				String appElemName = getName(mainElements[i], parentFrame);
				if (appElemName != null) {
					for (int j = 0; j < activeElements.length; j++) {
						String diaElemName = getName(activeElements[j],
								activeDialog);
						if (appElemName.equals(diaElemName)) {
							mainElements[i] = activeElements[j];
							continue;
						}
					}
				}
				display(mainElements[i]);
			}
			display(activeElements);

		}
	}


    /**
     * Returns a merged array of components. See
     * {@link #merge(G9DialogFrame)} for details.
     *
     * @param frame the dialog frame with the tool bar to merge into the
     *            application tool bar
     * @return a merged array with components from the application and the
     *         specified frame.
     */
    private Component[] getMergedComponents(G9DialogFrame frame) {
        Component[] tmp = getG9Components();
        List<Component> merged = new ArrayList<Component>();
        for (int i = 0; i < tmp.length; i++) {
            merged.add(tmp[i]);
        }

        Component[] dialogElems = new Component[0];

        if (frame != null && frame.getG9ToolBar() != null) {
            dialogElems = frame.getG9ToolBar().getG9Components();
        }

        int mergePos = merged.size();

        for (int i = 0; i < dialogElems.length; i++) {
            String elemName = getName(dialogElems[i], frame);
            int matchPos = elemPos(elemName, merged);
            if (matchPos > - 1) {
                mergePos = matchPos;
                merged.remove(mergePos);
            }
            merged.add(mergePos, dialogElems[i]);
            mergePos++;
        }

        Component[] mergedComponents = new Component[merged.size()];
        for (int i = 0; i < merged.size(); i++) {
            mergedComponents[i] = merged.get(i);
        }

        return mergedComponents;
    }

    /**
     * Returns the position of the named element in the specified List.
     *
     * @param name the name of the element
     * @param merged the list of merged elements
     * @return the position of the named element or -1 if no such element is
     *         found.
     */
    private int elemPos(String name, List<Component> merged) {
        int pos = -1;
        for (int i = 0; i < merged.size() && pos == -1; i++) {
            JComponent comp = (JComponent) merged.get(i);
            if (name.equals(parentFrame.fromComponentToName(comp))) {
                pos = i;
            }
        }
        return pos;
    }

    /**
     * Adds the component argument to this tool bar. If c is a separator then
     * the orientation is set properly.
     *
     * @param c the component to add
     * @return the component that was added
     */
	@Override
    public synchronized Component add(Component c) {
		if (myComponents == null) {
			myComponents = new Vector<Component>();
		}
		myComponents.add(c);
		if (c instanceof G9Button) {
			G9Button gb = (G9Button) c;
			gb.setInToolbar(true);
		}
		return display(c);
	}

    /**
     * Displays the argument component on the tool bar.
     *
     * @param c the component to display
     * @return the component that was added
     */
	private Component display(Component c) {
		if (c instanceof Separator) {
			if (getOrientation() == VERTICAL)
				((Separator) c).setOrientation(SwingConstants.HORIZONTAL);
			else
				((Separator) c).setOrientation(SwingConstants.VERTICAL);
			if(!isLastAddedSpacer) {
				isLastAddedSpacer = removeConsecutiveSeparators;
				return super.add(c);
			}
			return c;
		}

		isLastAddedSpacer = false;
		return super.add(c);
	}

    /**
     * Adds all {@link javax.swing.JComponent}s in the argument list to this
     * tool bar.
     *
     * @param components the list of components to add
     */
	private void display(Component components[]) {
		if (components == null) {
			return;
        }
		int maxComponentHeight = 0;
		for(int i = 0; i < components.length; i++) {
            if (components[i] != null) {
                int cHeight = components[i].getPreferredSize().height;
                if (cHeight > maxComponentHeight) {
                    maxComponentHeight = cHeight;
                }
                display(components[i]);
            }
		}
        if (maxComponentHeight > 0) {
            // Silly hack to avoid disappearing tool bars, see SUP-3518
		    Dimension preferredToolbarSize = getPreferredSize();
		    preferredToolbarSize.setSize(preferredToolbarSize.width+1, preferredToolbarSize.height);
            setPreferredSize(preferredToolbarSize);
		}
	}

    /**
     * Looks up the <code>elem</code> name in the argument dialog. Null is
     * return either if <code>elem</code> is not a
     * {@link javax.swing.JComponent} or if <code>elem</code> is not a known
     * component of the given dialog.
     *
     * @param elem the element to lookup
     * @param dialog the dialog used in the lookup
     * @return the <code>elem</code> name
     */
	private String getName(Object elem, G9DialogFrame dialog) {
		if (elem instanceof JComponent) {
			return dialog.fromComponentToName((JComponent) elem);
		}
		return null;
	}

    /**
     * @return the useOldMerge
     * @see #mergeToolBars(G9DialogFrame)
     */
    public static boolean isUseOldMerge() {
        return G9ToolBar.useOldMerge;
    }

    /**
     * @param useOldMerge the useOldMerge to set
     * @see #mergeToolBars(G9DialogFrame)
     */
    public static void setUseOldMerge(boolean useOldMerge) {
        G9ToolBar.useOldMerge = useOldMerge;
    }

}
