/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.component.menu;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyVetoException;
import java.util.List;
import java.util.Vector;

import javax.swing.Action;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JInternalFrame;
import javax.swing.JMenuItem;
import javax.swing.JSeparator;

import no.g9.client.component.G9DesktopPane;
import no.g9.message.CRuntimeMsg;
import no.g9.support.Registry;

/**
 * The window menu
 */
public class G9WindowMenu extends G9Menu {

    /** Because of serialization */
    private static final long serialVersionUID = 1L;

    /* Default menu items for the WindowMenu */

    /** The cascade menu item */
    protected JMenuItem cascadeItem;

    /** The tile menu item */
    protected JMenuItem tileItem;

    /** The tool bar menu item */
    protected JMenuItem toolBarItem;

    /** The message bar menu item */
    protected JMenuItem messageBarItem;

    /** List of default window menu items */
    private static List<G9MenuItem> defaultWindowMenuItems;

    /** Reference to the desktop pane */
    private G9DesktopPane desktop;

    /** Flag indicating if icons should be used in the open window list */
    private boolean useIconsInWindowList = false;

    /**
     * Constructs a new <code>JMenu</code> with no text.
     * 
     * @param desktop the desktop to monitor current open dialogs
     */
    public G9WindowMenu(G9DesktopPane desktop) {
        this("", desktop);
    }

    /**
     * Constructs a new <code>JMenu</code> with the supplied string as its text.
     * 
     * @param s the text for the menu label
     * @param desktop the desktop to monitor current open dialogs
     */
    public G9WindowMenu(String s, G9DesktopPane desktop) {
        super(s, G9Menu.WINDOW_MENU);
        this.desktop = desktop;
        initDefaultItems();
    }

    /**
     * Constructs a new <code>JMenu</code> with the supplied string as its text
     * and specified as a tear-off menu or not.
     * 
     * @param s the text for the menu label
     * @param b can the menu be torn off (not yet implemented)
     * @param desktop the desktop to monitor current open dialogs
     */
    public G9WindowMenu(String s, boolean b, G9DesktopPane desktop) {
        this("", desktop);

    }

    /**
     * Constructs a new <code>JMenu</code> whose properties are taken from the
     * <code>Action</code> supplied.
     * 
     * @param a the <code>Action</code>
     * @param desktop the desktop to monitor current open dialogs
     * @since 1.3
     */
    public G9WindowMenu(Action a, G9DesktopPane desktop) {
        this("", desktop);
        setAction(a);
    }

    /**
     * Constructs the default menu items Cut, Copy, Paste and Select all and
     * adds those to the menu.
     */
    private synchronized void initDefaultItems() {

        if (defaultWindowMenuItems == null && desktop != null) {
            defaultWindowMenuItems = new Vector<G9MenuItem>();
            defaultWindowMenuItems.add(createMenuItem(desktop.getActionMap()
                    .get(G9DesktopPane.cascade), null,
                    CRuntimeMsg.CM_TEXT_CASCADE));
            defaultWindowMenuItems.add(createMenuItem(desktop.getActionMap()
                    .get(G9DesktopPane.tile), null,
                    CRuntimeMsg.CM_TEXT_TILE));
        }

        setDefaultItem(defaultWindowMenuItems);

        Registry registry= Registry.getRegistry();
        String useIcons= registry.getProperty("menu.windowMenu.useIcons");
        setUseIconsInWindowList("true".equals(useIcons));
    }

    /**
     * Creates the menu components on this window menu. The menu components
     * contains a list of opened dialogs.
     */
    @Override
    protected synchronized void createMenuComponents() {

        super.createMenuComponents();

        if (desktop == null) {
            return;
        }

        JInternalFrame[] frames = desktop.getAllFrames();
        if (frames.length > 0) {
            add(new JSeparator());
            for (int i = 0; i < frames.length; i++) {
                if (frames[i].isVisible()) {
                    FrameMenuItem menuItem = new FrameMenuItem(frames[i]);
                    menuItem.setSelected(frames[i].isSelected());

                    menuItem.addActionListener(new ActionListener() {
                        @Override
                        public void actionPerformed(ActionEvent e) {
                            JInternalFrame frame = ((FrameMenuItem) e
                                    .getSource()).getFrame();

                            try {
                                if (frame.isIcon()) {
                                    frame.setIcon(false);
                                }
                                frame.setSelected(true);
                            } catch (PropertyVetoException e1) {
                                // selection was vetoed, menu item should not be
                                // selected.
                                FrameMenuItem mi = (FrameMenuItem) e
                                        .getSource();
                                mi.setSelected(false);
                            }
                        }
                    });
                    if (isUseIconsInWindowList()) {
                        menuItem.setIcon(frames[i].getFrameIcon());
                    }
                    add(menuItem);

                }
            }

        }
    }

    /**
     * The menu item representing an opened window in the list of opened
     * windows.
     */
    private static class FrameMenuItem extends JCheckBoxMenuItem {

        /** Because of serialization */
        private static final long serialVersionUID = 1L;

        /** The internal frame represented by this menu item */
        private JInternalFrame frame;

        /**
         * Constructs a new Frame menu item
         * 
         * @param frame the internal frame
         */
        public FrameMenuItem(JInternalFrame frame) {
            super(frame.getTitle());
            this.frame = frame;
        }

        /**
         * Returns the internal frame represented by this menu item
         * 
         * @return the internal frame.
         */
        public JInternalFrame getFrame() {
            return frame;
        }

    }

    /**
     * Returns the useIconsInWindowList property. If this property is
     * <code>true</code> the open-window list will contain the dialog's icon.
     * 
     * @return the useIconsInWindowList property.
     */
    public synchronized boolean isUseIconsInWindowList() {
        return useIconsInWindowList;
    }

    /**
     * Sets the useIconsInWindowList property. If this property is
     * <code>true</code> the open-window list will use the dialog's icon.
     * 
     * @param useIconsInWindowList the useIconsInWindowList to set
     */
    public synchronized void setUseIconsInWindowList(
            boolean useIconsInWindowList) {
        this.useIconsInWindowList = useIconsInWindowList;
    }

}
