/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

/**
 * Internal use!
 * <p>
 * The AbstractAttributeComparator has the capability to chain comparators,
 * facilitating a generic sort based on various attributes.
 */
@SuppressWarnings("rawtypes")
public abstract class AbstractAttributeComparator implements Comparator, Comparable {

    /** The priority of this comparator */
    private int priority;

    /** The sort direction */
    private int direction;

    /** The column number */
    private int column;

    /** The attribute name */
    private String attributeName;

    /** The direction string */
    private String directionString;

    /** The comparator after this comparator */
    private AbstractAttributeComparator next;

    /** Constant denoting the sort direction. */
    public static final int ASCENDING = 1;

    /** Constant denoting the sort direction. */
    public static final int DESCENDING = 2;

    /**
     * Creates a new instance of this comparator.
     *
     * @param attributeName the name of the attribute this comparator compares.
     * @param column the column number for this comparator
     * @param priority the priority of this comparator
     * @param direction the sort direction of this comparator, either ASCENDING
     *            or DESCENDING.
     */
    public AbstractAttributeComparator(String attributeName, int column, int priority,
            int direction) {
        this.attributeName = attributeName;
        this.column = column;
        this.priority = priority;
        this.direction = direction;
        setDirectionString();
    }

    /**
     * Compares this object with the specified object for order.
     *
     * @param o the Object to be compared.
     * @return a negative integer, zero, or a positive integer as this object is
     *         less than, equal to, or greater than the specified object.
     */
    @Override
    public int compareTo(Object o) {
        return compareTo((AbstractAttributeComparator) o);
    }

    /**
     * Compares this object to another instance of AbstractAttributeComparator.
     * The comparison is based on the priority.
     *
     * @param another the AbstractAttributeComparator to compare with.
     * @return a negative integer, zero, or a positive integer as this object's
     *         priority is less than, equal to, or greater than the specified
     *         object's priority.
     */
    public int compareTo(AbstractAttributeComparator another) {
        return this.priority - another.priority;
    }

    /**
     * Chains this comparator to first comparator in the list, and then calls
     * that comparators chain method.
     *
     * @param comparators (missing javadoc)
     */
    public static void chainComparators(List comparators) {
        if (comparators == null || comparators.size() == 0) {
            return;
        }
        AbstractAttributeComparator compA = null;
        AbstractAttributeComparator compB;
        Iterator iter = comparators.iterator();
        while (iter.hasNext()) {
            compB = (AbstractAttributeComparator) iter.next();
            if (compA != null) {
                compA.next = compB;
            }
            compA = compB;
        }
        if (compA != null) {
            compA.next = null;
        }
    }

    /**
     * Get the comparator with the given column number
     *
     * @param comparators the list of comparators
     * @param column the column number
     * @return the comparator, null if not found
     */
    public static AbstractAttributeComparator getComparator(List comparators, int column) {
        AbstractAttributeComparator result = null;
        AbstractAttributeComparator comp;
       Iterator iter = comparators.iterator();
        while (iter.hasNext()) {
            comp = (AbstractAttributeComparator) iter.next();
            if (comp.getColumn() == column) {
                result = comp;
                break;
            }
        }
        return result;
    }

    /**
     * Gets the next comparator in line.
     *
     * @return the next comparator.
     */
    public AbstractAttributeComparator getNextComparator() {
        return next;
    }

    /**
     * Sets the next comparator. The next comparator is used if the comparison
     * is equal.
     *
     * @param nextComparator the next comparator in line.
     */
    public void setNextComparator(AbstractAttributeComparator nextComparator) {
        next = nextComparator;
    }

    /**
     * Returns the attribute name this comparator compares.
     *
     * @return the attributeName
     */
    public String getAttributeName() {
        return attributeName;
    }

    /**
     * Sets the name of the attribute this comparator compares.
     *
     * @param attributeName the attributeName to set
     */
    public void setAttributeName(String attributeName) {
        this.attributeName = attributeName;
    }

    /**
     * Gets the sort direction of this comparator. The direction is either
     * ASCENDING or DESCENDING.
     *
     * @return the direction
     */
    public int getDirection() {
        return direction;
    }

    /**
     * Sets the sort direction of this comparator. The sort direction should be
     * either ASCENDING or DESCENDING.
     *
     * @param direction the direction to set
     */
    public void setDirection(int direction) {
        this.direction = direction;
    }

    /**
     * Sets the direction string
     */
    private void setDirectionString() {
        directionString = direction == ASCENDING ? "ascending" : "descending";
    }

    /**
     * Toggles the sort direction of this comparator.
     */
    public void toggleDirection() {
        if (direction == ASCENDING) {
            direction = DESCENDING;
        }
        else {
            direction = ASCENDING;
        }
        setDirectionString();
    }

    /**
     * Gets the priority of this comparator.
     *
     * @return the priority
     */
    public int getPriority() {
        return priority;
    }

    /**
     * Sets the priority of this comparator.
     *
     * @param priority the priority to set
     */
    public void setPriority(int priority) {
        this.priority = priority;
    }

    /**
     * Mainly for debugging purposes. Returns a string representation of this
     * comparator. The string contains the name of the attribute this comparator
     * compares, the sort direction and the priority of this comparator.
     *
     * @return String representation of this comparator
     */
    @Override
    public String toString() {
        return attributeName + " " + directionString + " priority: " + priority + " column: " + column;
    }

    /**
     * Internal use.
     * @return the column
     */
    public int getColumn() {
        return column;
    }

    /**
     * Internal use.
     * @param column the column to set
     */
    public void setColumn(int column) {
        this.column = column;
    }

}
