/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.awt.Color;
import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JInternalFrame;
import javax.swing.SwingUtilities;

import no.g9.client.event.G9ValueState;
import no.g9.client.support.RoleObject.AssociationAccess;
import no.g9.domain.DomainObjectProxy;
import no.g9.domain.DomainUtil;
import no.g9.exception.G9ClientException;
import no.g9.exception.G9ClientFrameworkException;
import no.g9.message.*;
import no.g9.support.TypeTool;
import no.g9.support.Visitor;

/**
 * The class representing a target in the object selection. Implements common
 * methods for the object selection nodes.
 *
 * @param <T>
 *            The domain class type represented by this node
 */
@SuppressWarnings("unchecked")
public abstract class AbstractNode<T> implements ObjectSelectionNode {

    /** The current state of this target */
    private int currentState = ObjectSelectionNode.CLEARED;

    /** The overridden state of this target */
    private Integer stateOverride = null;

    /** The object proxy representing the domain object */
    @SuppressWarnings("rawtypes")
    protected DomainObjectProxy objectProxy;

    @SuppressWarnings("rawtypes")
    @Override
    public Set getChangedNoKeyFields() {
        Set s = getChangedFields();
        s.removeAll(getChangedKeyFields());
        return s;
    }

    @Override
    public int setState(int state) {
        return setState(state, true);
    }

    /**
     * INTERNAL USE. Clears the object selection role on edt
     */
    protected void clearOnEDT() {
        Runnable task = new Runnable() {

            @Override
            public void run() {
                clear();
            }
        };
        edt(task, "clear");
    }

    /**
     * INTERNAL USE. Clears on edt
     *
     * @see ObjectSelectionNode#clearKeepKeys()
     */
    protected void clearKeepKeysOnEDT() {
        Runnable task = new Runnable() {

            @Override
            public void run() {
                clearKeepKeys();
            }
        };
        edt(task, "clearKeepKeys");
    }

    /**
     * INTERNAL USE. Clears the specified field on the EDT
     *
     * @param s
     *            the field to clear
     */
    protected void clearOnEDT(final String s) {
        Runnable task = new Runnable() {

            @Override
            public void run() {
                clear(s);
            }
        };
        edt(task, "clear " + s);
    }

    /**
     * Appends the value from the specified field (if possible) to the
     * corresponding text area.
     *
     * @param domainInstance
     *            the domain instance with the field value to append
     * @param field
     *            the field
     */
    public abstract void append(T domainInstance, String field);

    /**
     * INTERNAL USE. Appends on edt
     *
     * @param domainInstance
     *            the domain instance
     * @param field
     *            the field
     * @see #append(Object, String)
     */
    protected void appendOnEDT(final T domainInstance, final String field) {
        Runnable task = new Runnable() {

            @Override
            public void run() {
                append(domainInstance, field);
            }
        };
        edt(task, "append " + field);
    }

    /**
     * Sets the state of this target. If the second parameter is
     * <code>true</code> the state is overridden, and the state is not
     * calculated when invoking <code>getState()</code>.
     *
     * @param state
     *            the state to set.
     * @param override
     *            if <code>true</code> the state is overridden
     * @return the new state of this target
     */
    protected int setState(int state, boolean override) {
        if (override) {
            stateOverride = new Integer(state);
        } else {
            stateOverride = null;
        }

        currentState = state;

        return getState();
    }

    @SuppressWarnings("rawtypes")
    @Override
    public int getState() {
        if (stateOverride != null) {
            return stateOverride.intValue();
        }
        Set s = getChangedFields();
        boolean altered = s.size() > 0;
        switch (currentState) {
        case ObjectSelectionNode.CLEARED:
            if (altered) {
                currentState = ObjectSelectionNode.EDITED;
            }
            break;
        case ObjectSelectionNode.EDITED:
            if (!altered) {
                currentState = ObjectSelectionNode.CLEARED;
            }
            break;
        case ObjectSelectionNode.CLEAN:
            if (altered) {
                currentState = ObjectSelectionNode.DIRTY;
            }
            break;
        case ObjectSelectionNode.DIRTY:
            if (!altered) {
                currentState = ObjectSelectionNode.CLEAN;
            }
            break;
        default:
            break;
        }
        return currentState;

    }

    /**
     * Displays the Set of domain objects. This method will be overridden by
     * classes capable of displaying several instances of the domain object. In
     * other cases the method does nothing.
     *
     * @param domainObjectCollection
     *            the collection of objects to display
     */
    @SuppressWarnings("rawtypes")
    public void recursiveDisplay(Collection domainObjectCollection) {
        // Empty, overridden in subclasses where appropriate.
    }

    /**
     * INTERNAL USE. Invokes display on edt
     *
     * @param domainInstance
     *            the domain instance to display
     */
    protected void displayOnEDT(final Object domainInstance) {
        Runnable task = new Runnable() {

            @Override
            public void run() {
                display(domainInstance);
            }
        };
        edt(task, "display " + domainInstance);
    }

    /**
     * INTERNAL USE. Invokes display on edt
     *
     * @param attributeName
     *            the name of the attribute to display
     * @param value
     *            the value of the attribute to display
     * @param resetState
     *            flag - if true state is reset
     */
    protected void displayOnEDT(final String attributeName, final Object value,
            final boolean resetState) {
        Runnable task = new Runnable() {

            @Override
            public void run() {
                display(attributeName, value, resetState);
            }
        };
        edt(task, "display " + attributeName + " " + value);
    }

    /**
     * Updates data of given attriubte in object selection in the dialog.
     *
     * @param attributeName
     *            Name of attribute to update.
     * @param value
     *            The new value.
     * @param resetState
     *            Set to false to avoid resetting the attribute state.
     */
    public abstract void display(String attributeName, Object value,
            boolean resetState);

    @SuppressWarnings("rawtypes")
    @Override
    public void displayOnEDT(final Collection resultObject,
            final List pathToTarget, final boolean findAll) {
        Runnable task = new Runnable() {

            @Override
            public void run() {
                displayAboveTarget(resultObject, pathToTarget, findAll);
            }
        };
        edt(task, "displayOnEDT");
    }

    @SuppressWarnings("rawtypes")
    @Override
    public abstract void displayAboveTarget(Collection resultObject,
            List pathToTarget, boolean findAll);

    /**
     * Obtains the Set of domain objects. This method will be overridden by
     * classes which are represented as several instances in the dialog (e.g. a
     * list block).
     *
     * @return a Set of objects.
     * @deprecated use {@link #obtainSet(boolean)}
     */
    @SuppressWarnings("rawtypes")
    @Deprecated
    public Set obtainSet() {
        return obtainSet(true);

    }

    /**
     * Updates data of given attriubte in object selection in the dialog.
     *
     * @param domainInstance
     *            Attributes are copied from this object.
     * @param attributeName
     *            The attribute to set.
     * @param resetState
     *            Set to false to avoid resetting the attribute state.
     */
    public abstract void display(T domainInstance, String attributeName,
            boolean resetState);

    /**
     * INTERNAL USE. Displays on edt
     *
     * @param domainInstance
     *            (missing javadoc)
     * @param attributeName
     *            (missing javadoc)
     * @param resetState
     *            (missing javadoc)
     * @see #display(Object, String, boolean)
     */
    protected void displayOnEDT(final T domainInstance,
            final String attributeName, final boolean resetState) {
        Runnable task = new Runnable() {

            @Override
            public void run() {
                display(domainInstance, attributeName, resetState);
            }
        };
        edt(task, "display");
    }

    /**
     * INTERNAL USE. Sets focus on edt.
     *
     * @param field
     *            (missing javadoc)
     * @see ObjectSelectionNode#setFocus(String)
     */
    protected void setFocusOnEDT(final String field) {
        Runnable task = new Runnable() {

            @Override
            public void run() {
                setFocus(field);
            }
        };
        edt(task, "setFocus " + field);
    }

    /**
     * INTERNAL USE. setShown on edt
     *
     * @param shown
     *            (missing javadoc)
     * @see ObjectSelectionNode#setShown(boolean)
     */
    protected void setShownOnEDT(final boolean shown) {
        Runnable task = new Runnable() {

            @Override
            public void run() {
                setShown(shown);
            }
        };
        edt(task, "setShown");
    }

    /**
     * INTERNAL USE. setShown on edt
     *
     * @param shown
     *            (missing javadoc)
     * @param field
     *            (missing javadoc)
     * @see ObjectSelectionNode#setShown(boolean, String)
     */
    protected void setShownOnEDT(final boolean shown, final String field) {
        Runnable task = new Runnable() {

            @Override
            public void run() {
                setShown(shown, field);
            }
        };
        edt(task, "setShown " + field);
    }

    /**
     * INTERNAL USE. Enables field on edt.
     *
     * @param enable
     *            (missing javadoc)
     * @param field
     *            (missing javadoc)
     * @see ObjectSelectionNode#setEnabled(boolean, String)
     */
    protected void setEnabledOnEDT(final boolean enable, final String field) {
        Runnable task = new Runnable() {

            @Override
            public void run() {
                setEnabled(enable, field);
            }
        };
        edt(task, "setEnabled " + field);
    }

    /**
     * INTERNAL USE. Enables field on edt
     *
     * @param enable
     *            (missing javadoc)
     * @see ObjectSelectionNode#setEnabled(boolean)
     */
    protected void setEnabledOnEDT(final boolean enable) {
        Runnable task = new Runnable() {

            @Override
            public void run() {
                setEnabled(enable);
            }
        };
        edt(task, "setEnabled");
    }

    /**
     * INTERNAL USE. Sets the style on edt
     *
     * @param foreground
     *            (missing javadoc)
     * @param background
     *            (missing javadoc)
     * @see ObjectSelectionNode#setStyle(Color, Color)
     */
    protected void setStyleOnEDT(final Color foreground, final Color background) {
        Runnable task = new Runnable() {

            @Override
            public void run() {
                setStyle(foreground, background);
            }
        };
        edt(task, "setStyle");
    }

    /**
     * Obtains a set of domain objects. This method will be overridden by
     * classes which are represented as several instances in the dialog (e.g. a
     * list block). Note that invoking this method has a side effect in the case
     * of listblocks if <code>createIfNull</code> flag is true: If a listblock
     * line holds a null-reference to a domain instance, a new object will be
     * created by the listblock line.
     *
     * @param createIfNull
     *            flag indicating if missing objects should be created.
     * @see ListblockLine#getObject(String, boolean)
     * @return a Set of objects
     */
    @Override
    @SuppressWarnings("rawtypes")
    public Set obtainSet(boolean createIfNull) {
        Set objectSet = TypeTool.getDefaultSet();
        objectSet.add(obtainAsObject());
        return objectSet;
    }

    /**
     * Recursivly releases the proxy on this target. The effect is that the
     * connection to the original domain object (as returned from the server) is
     * lost, an subsequent obtains will return a new instance of the domain
     * object, with only attributes from the dialog set.
     */
    @SuppressWarnings("rawtypes")
    public void nullProxy() {
        objectProxy = null;
        List related = getChildren();
        Iterator it = related.iterator();
        while (it.hasNext()) {
            AbstractNode child = (AbstractNode) it.next();
            child.nullProxy();
        }
    }

    @Override
    public boolean checkFind() {
        if (getController().ignoreCheckChanged()) {
            return true;
        }
        CheckFindTask cft = new CheckFindTask();

        if (!SwingUtilities.isEventDispatchThread()) {
            String msgID = null;
            Exception ex = null;
            try {
                SwingUtilities.invokeAndWait(cft);
            } catch (InterruptedException e) {
                msgID = CRuntimeMsg.CT_INTERRUPTED;
                ex = e;
            } catch (InvocationTargetException e) {
                msgID = CRuntimeMsg.CT_INVOCATION_TARGET;
                if (e.getCause() != null && e.getCause() instanceof Exception) {
                    ex = (Exception) e.getCause();
                } else {
                    ex = e;
                }
            } finally {
                if (msgID != null) {
                    Object[] msgArgs = { this.getClass(), "checkFind", ex };
                    Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
                    MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                    throw new G9ClientFrameworkException(ex, msg);
                }
            }
        } else {
            cft.result = cft.checkFind();
        }

        return cft.result;

    }

    @Override
    public boolean checkRowSelection(int oldSelection, int newSelection) {
        return checkRowSelection(oldSelection, newSelection, null);
    }

    /**
     * @param oldSelection
     *            (missing javadoc)
     * @param newSelection
     *            (missing javadoc)
     * @param foreignNodes
     *            (missing javadoc)
     * @return (missing javadoc)
     */
    @SuppressWarnings("rawtypes")
    public boolean checkRowSelection(int oldSelection, int newSelection,
            Set foreignNodes) {
        if (getController().ignoreCheckChanged()) {
            return true;
        }

        ECheckResult cr = getController().callHookCheckRowSelect(getRoleName(),
                oldSelection, newSelection);

        boolean ok = true;

        if (cr == ECheckResult.CHANGED) {
            ok = false;
        } else if (cr == ECheckResult.DEFAULT) {
            ok = checkChanged();
        }

        if (ok && cr != ECheckResult.CONTINUE) {
            Iterator it = foreignNodes != null ? foreignNodes.iterator() : null;
            if (it != null) {
                while (ok && it.hasNext()) {
                    String foreignNodeName = (String) it.next();
                    AbstractNode foreignNode = (AbstractNode) getController()
                            .getObjectSelectionNode(foreignNodeName);

                    ok = foreignNode.checkChanged();
                }
            }
        }

        if (!ok) {
            String msgID;
            if (oldSelection == newSelection) {
                msgID = CRuntimeMsg.CC_CHECK_UNSELECTION_MSG;
            } else {
                msgID = CRuntimeMsg.CC_CHECK_SELECTION_MSG;
            }

            Message message = MessageSystem.getMessageFactory().getMessage(msgID);
            MessageReply reply = Application.getMessageDispatcher(
                    getController().getWindow(),
                    getController().getApplication()).dispatch(message);
            ok = MessageReplyType.REPLY_OK.equals(reply);
        }
        return ok;
    }

    /**
     * @return (missing javadoc)
     */
    public boolean checkPrint() {
        if (getController().ignoreCheckChanged()) {
            return true;
        }

        CheckPrintTask cpt = new CheckPrintTask();
        if (!SwingUtilities.isEventDispatchThread()) {
            String msgID = null;
            Exception ex = null;
            try {
                SwingUtilities.invokeAndWait(cpt);
            } catch (InterruptedException e) {
                msgID = CRuntimeMsg.CT_INTERRUPTED;
                ex = e;
            } catch (InvocationTargetException e) {
                msgID = CRuntimeMsg.CT_INVOCATION_TARGET;
                if (e.getCause() != null && e.getCause() instanceof Exception) {
                    ex = (Exception) e.getCause();
                } else {
                    ex = e;
                }
            } finally {
                if (msgID != null) {
                    Object[] msgArgs = { this.getClass(), "checkPrint", ex };
                    Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
                    MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                }
            }
        } else {
            cpt.result = cpt.checkPrint();
        }

        return cpt.result;
    }

    @Override
    public boolean checkClose() {
        if (getController().ignoreCheckChanged()) {
            return true;
        }
        CheckCloseTask cct = new CheckCloseTask();

        if (!SwingUtilities.isEventDispatchThread()) {
            String msgID = null;
            Exception ex = null;
            try {
                SwingUtilities.invokeAndWait(cct);
            } catch (InterruptedException e) {
                msgID = CRuntimeMsg.CT_INTERRUPTED;
                ex = e;
            } catch (InvocationTargetException e) {
                msgID = CRuntimeMsg.CT_INVOCATION_TARGET;
                if (e.getCause() != null && e.getCause() instanceof Exception) {
                    ex = (Exception) e.getCause();
                } else {
                    ex = e;
                }
            } finally {
                if (msgID != null) {
                    Object[] msgArgs = { this.getClass(), "checkClose", ex };
                    Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
                    MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                    throw new G9ClientFrameworkException(ex, msg);
                }
            }
        } else {
            cct.result = cct.checkClose();
        }
        return cct.result;
    }

    @Override
    public boolean checkDelete() {
        if (getController().ignoreCheckChanged()) {
            return true;
        }
        CheckDeleteTask cdt = new CheckDeleteTask(this);

        if (!SwingUtilities.isEventDispatchThread()) {
            String msgID = null;
            Exception ex = null;
            try {
                SwingUtilities.invokeAndWait(cdt);
            } catch (InterruptedException e) {
                msgID = CRuntimeMsg.CT_INTERRUPTED;
                ex = e;
            } catch (InvocationTargetException e) {
                msgID = CRuntimeMsg.CT_INVOCATION_TARGET;
                if (e.getCause() != null && e.getCause() instanceof Exception) {
                    ex = (Exception) e.getCause();
                } else {
                    ex = e;
                }
            } finally {
                if (msgID != null) {
                    Object[] msgArgs = { this.getClass(), "checkDelete", ex };
                    Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
                    MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                    throw new G9ClientFrameworkException(ex, msg);
                }
            }
        } else {
            cdt.result = cdt.checkDelete();
        }
        return cdt.result;
    }

    @Override
    public boolean checkClear() {
        if (getController().ignoreCheckChanged()) {
            return true;
        }
        CheckClearTask cct = new CheckClearTask();
        if (!SwingUtilities.isEventDispatchThread()) {
            String msgID = null;
            Exception ex = null;
            try {
                SwingUtilities.invokeAndWait(cct);
            } catch (InterruptedException e) {
                msgID = CRuntimeMsg.CT_INTERRUPTED;
                ex = e;
            } catch (InvocationTargetException e) {
                msgID = CRuntimeMsg.CT_INVOCATION_TARGET;
                if (e.getCause() != null && e.getCause() instanceof Exception) {
                    ex = (Exception) e.getCause();
                } else {
                    ex = e;
                }
            } finally {
                if (msgID != null) {
                    Object[] msgArgs = { this.getClass(), "checkClear", ex };
                    Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
                    MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                    throw new G9ClientFrameworkException(ex, msg);
                }
            }
        } else {
            cct.result = cct.checkClear();
        }
        return cct.result;
    }

    @Override
    public boolean checkSave() {
        if (getController().ignoreCheckChanged()) {
            return true;
        }
        CheckSaveTask cst = new CheckSaveTask(this);

        if (!SwingUtilities.isEventDispatchThread()) {
            String msgID = null;
            Exception ex = null;
            try {
                SwingUtilities.invokeAndWait(cst);
            } catch (InterruptedException e) {
                msgID = CRuntimeMsg.CT_INTERRUPTED;
                ex = e;
            } catch (InvocationTargetException e) {
                msgID = CRuntimeMsg.CT_INVOCATION_TARGET;
                if (e.getCause() != null && e.getCause() instanceof Exception) {
                    ex = (Exception) e.getCause();
                } else {
                    ex = e;
                }
            } finally {
                if (msgID != null) {
                    Object[] msgArgs = { this.getClass(), "checkSave", ex };
                    Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
                    MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                    throw new G9ClientFrameworkException(ex, msg);
                }
            }
        } else {
            cst.result = cst.checkSave();
        }

        return cst.result;

    }

    @Override
    public boolean checkChanged() {
        return nodeCheckChange(false);
    }

    @SuppressWarnings("rawtypes")
    private boolean nodeCheckChange(boolean checkRelation) {
        if (getController().ignoreCheckChanged()) {
            return true;
        }

        ECheckResult rs = getController().callHookCheckChange(getRoleName());
        boolean ok = true;

        if (rs == ECheckResult.CHANGED) {
            return false;
        } else if (rs == ECheckResult.UNCHANGED) {
            return true;
        } else if (rs == ECheckResult.DEFAULT) {

            if (getRoleObject().isUpRelated()) {
                ok = getChangedNoKeyFields().size() == 0;
            } else {
                ok = getChangedFields().size() == 0;
            }
            if (!ok) {
                return false;
            } else if (checkRelation) {
                ok = relationIntact();
            }
        }
        // Default or continue.
        List related = getChildren();
        Iterator it = related.iterator();
        while (it.hasNext() && ok) {
            AbstractNode child = (AbstractNode) it.next();
            ok = child.nodeCheckChange(true);
        }

        return ok;
    }

    /**
     * Recursivly calls check change and then checks the integrity of the
     * relation of this target and the parent target if the target is
     * up-related.
     *
     * @return <code>true</code> if fields and relations are intact.
     */
    public boolean checkChangedAsChild() {
        return nodeCheckChange(true);
    }

    /**
     * Returns the role object associated with object selection target.
     *
     * @return the role object of this target.
     */
    @SuppressWarnings("rawtypes")
    private RoleObject getRoleObject() {
        Object domainObject = getProxy().getObject();
        List roles = getController().getRoleObjects(domainObject);
        Iterator it = roles.iterator();
        RoleObject role = null;
        while (it.hasNext() && role == null) {
            RoleObject cur = (RoleObject) it.next();
            if (cur.getRoleName().equals(getRoleName())) {
                role = cur;
            }
        }

        return role;
    }

    /**
     * Check if the relation between this target and the parent target is
     * unchanged. This only done if this target is an up-related target. The
     * calculation is roughly performed as follows:
     * <ul>
     * <li>Get the child object as seen from the parent target
     * <li>Obtain this target (the performed obtain is only temporly)
     * <li>Compare the two objects using hashCode.
     * </ul>
     * If the hashCode differ, the relation is considered as changed, and this
     * method returns <code>false</code>
     *
     * @return <code>true</code> if the relation is intact.
     */
    @SuppressWarnings("rawtypes")
    private boolean relationIntact() {
        RoleObject currentRole = getRoleObject();

        // Only check if up-related.
        if (!currentRole.isUpRelated()) {
            return true;
        }

        // Get original hash code on object as seen from parent (if
        // object sees "null", then check if "this" target is clean.:
        AbstractNode parentNode = (AbstractNode) getParentNode();
        if (parentNode == null || parentNode == this
                || parentNode.getState() == ObjectSelectionNode.CLEARED
                || parentNode.getState() == ObjectSelectionNode.EDITED) {
            return true;
        }

        RoleObject parentRole = parentNode.getRoleObject();
        Object parentObject = parentNode.getProxy().getObject();
        RoleObject.AssociationAccess associationAccess = parentRole
                .getAssociationAccess(currentRole);
        Object asumedChild = associationAccess
                .getAssociationOnType(parentObject);
        int originalHash = 0;
        if (asumedChild != null) {
            originalHash = asumedChild.hashCode();
        } else {
            return getState() == ObjectSelectionNode.CLEARED;
        }

        // Obtain, calculate new hash and reset proxy
        obtainAsObject();
        int displayedHash = getProxy().getObject().hashCode();
        getProxy().reset();
        return originalHash == displayedHash;
    }

    /**
     * Returns an empty instance of the domain object represented by this object
     * selection target.
     *
     * @return an empty instance of the domain object
     */
    public abstract Object getEmptyDomainObject();

    /**
     * Returns a shallow copy of the domain object, where attributes are read
     * from the dialog.
     *
     * @return an object where attributes are read from the dialog
     */
    @SuppressWarnings("rawtypes")
    @Override
    public Object peekAtObject() {
        DomainObjectProxy tmpProxy = getController().getNewProxy(
                getEmptyDomainObject(), getRoleName());
        DomainObjectProxy oldProxy = objectProxy;
        objectProxy = tmpProxy;
        Object retVal = obtainAsObject();
        objectProxy = oldProxy;

        return retVal;
    }

    /**
     * Retuns the domain object proxy for this target. If no such proxy exist, a
     * new is created.
     *
     * @return the domain object proxy
     */
    @SuppressWarnings("rawtypes")
    public DomainObjectProxy getProxy() {
        if (objectProxy == null) {
            objectProxy = getController().getNewProxy(getEmptyDomainObject(),
                    getRoleName());
        }
        return objectProxy;
    }

    /**
     * Displays the specified attribute value. Displayed value might be
     * considered changed if the <code>markIfChanged</code> parameter is
     * <code>true</code>.
     *
     * @param attributeName
     *            the name of the attribute
     * @param value
     *            the value to display
     */
    public abstract void display(String attributeName, Object value);

    // /////////////////////// INNER CLASSES USED TO PERFORM CHECKS
    // ////////////////
    // /////////////////////// Should be started from EDT!!
    // ////////////////////////

    /**
     * The thread that performs check find.
     */
    private class CheckFindTask implements Runnable {

        /**
         * The result of this task.
         */
        private boolean result;

        /**
         * Default constructor.
         */
        CheckFindTask() {
            super();
        }

        /**
         * Performs check find
         *
         * @return <code>true</code> if it is ok to proceed with action find.
         */
        @SuppressWarnings("rawtypes")
        private boolean checkFind() {
            boolean ok = true;
            ECheckResult cr = getController().callHookCheckFind(getRoleName());

            if (cr == ECheckResult.CHANGED) {
                ok = false;
            } else if (cr == ECheckResult.DEFAULT) {
                ok = getChangedNoKeyFields().size() == 0;
            }

            // Default and continue
            if (ok && cr != ECheckResult.UNCHANGED) {
                List related = getChildren();
                Iterator it = related.iterator();
                while (it.hasNext() && ok) {
                    AbstractNode child = (AbstractNode) it.next();
                    ok = child.checkChangedAsChild();
                }
            }

            if (!ok) { // interact
                Message message = MessageSystem.getMessageFactory().getMessage(
                        CRuntimeMsg.CC_CHECK_FIND_MSG);
                MessageReply reply = Application.getMessageDispatcher(
                        getController().getWindow(),
                        getController().getApplication()).dispatch(message);

                ok = MessageReplyType.REPLY_OK.equals(reply);
            }
            return ok;
        }

        @Override
        public void run() {
            result = checkFind();
        }
    }

    /**
     * The runnable that performs check print
     */
    private class CheckPrintTask implements Runnable {

        /** The result of this task */
        private boolean result;

        @SuppressWarnings("rawtypes")
        private boolean checkPrint() {
            ECheckResult checkResult = getController().callHookCheckPrint(
                    getRoleName());

            boolean ok = true;
            if (checkResult == ECheckResult.CHANGED) {
                ok = false;
            } else if (checkResult == ECheckResult.DEFAULT) {
                ok = checkChanged();
            }

            if (ok && checkResult != ECheckResult.UNCHANGED) {
                List children = getChildren();
                Iterator it = children.iterator();
                while (it.hasNext() && ok) {
                    ok = ((AbstractNode) it.next()).checkChangedAsChild();
                }
            }

            return ok;
        }

        @Override
        public void run() {
            result = checkPrint();
        }
    }

    /**
     * The thread that performs check close
     */
    private class CheckCloseTask implements Runnable {

        /** The result of this task */
        private boolean result;

        /**
         * Default constructor.
         */
        CheckCloseTask() {
            super();
        }

        /**
         * Check if it is ok to proceed with action close.
         *
         * @return <code>true</code> if ok to proceed with action close.
         */
        @SuppressWarnings("rawtypes")
        private boolean checkClose() {
            if (getController().getWindow() instanceof JInternalFrame) {
                JInternalFrame fView = (JInternalFrame) getController()
                        .getWindow();
                fView.toFront();
            } else if (getController().getWindow() instanceof JDialog) {
                JDialog dView = (JDialog) getController().getWindow();
                dView.toFront();

            }

            ECheckResult cr = getController().callHookCheckClose(getRoleName());

            boolean ok = true;

            if (cr == ECheckResult.CHANGED) {
                ok = false;
            } else if (cr == ECheckResult.DEFAULT) {
                ok = checkChanged();
            }

            if (ok && cr != ECheckResult.UNCHANGED) {
                List children = getChildren();
                Iterator it = children.iterator();
                while (it.hasNext() && ok) {
                    ok = ((AbstractNode) it.next()).checkChangedAsChild();
                }
            }

            return ok;
        }

        @Override
        public void run() {
            result = checkClose();
        }
    }

    /**
     * Common code for check save functionality (used both in checkSave and
     * checkDelete).
     */
    private class CheckSaveCommon {

        /**
         * Default constructor.
         */
        CheckSaveCommon() {
            super();
        }

        @SuppressWarnings("rawtypes")
        protected ECheckResult doCheckSave(List nodesToCheck) {
            ECheckResult cr = ECheckResult.DEFAULT;
            Iterator pIt = nodesToCheck.iterator();
            boolean continueCheck = true;
            while (pIt.hasNext() && continueCheck) {
                AbstractNode n = (AbstractNode) pIt.next();
                cr = getController().callHookCheckSave(n.getRoleName());
                if (cr == ECheckResult.CHANGED || cr == ECheckResult.UNCHANGED) {
                    continueCheck = false;
                }
            }
            return cr;
        }
    }

    /**
     * The thread that performs check delete.
     */
    private class CheckDeleteTask extends CheckSaveCommon implements Runnable {

        /** The result of check delete. */
        private boolean result;

        @SuppressWarnings("rawtypes")
        private AbstractNode target;

        /**
         * Constructs a new CheckDeleteTask.
         *
         * @param node
         *            the node to check
         */
        @SuppressWarnings("rawtypes")
        CheckDeleteTask(AbstractNode node) {
            this.target = node;
        }

        /**
         * Check if it is ok to preoceed with delete.
         *
         * @return <code>true</code> if it is ok to proceed with action delete.
         */
        @SuppressWarnings("rawtypes")
        private boolean checkDelete() {
            boolean ok = true;

            // First, perform checkSave on parent
            List nodesToCheck = new LinkedList();
            AbstractNode parent = (AbstractNode) getParentNode();
            if (parent != null) {
                parent.checkSaveListAboveTarget(nodesToCheck, target, false);
            }
            ECheckResult cr = doCheckSave(nodesToCheck);
            ok = cr != null && cr != ECheckResult.CHANGED;
            if (!ok) {
                return false;
            }

            // Do the checkDelete part
            cr = getController().callHookCheckDelete(getRoleName());

            if (cr == ECheckResult.CHANGED) {
                ok = false;
            } else if (cr == ECheckResult.DEFAULT) {
                ok = checkChanged();
            }

            if (ok && cr != ECheckResult.CONTINUE) {
                Iterator it = getChildren().iterator();
                while (it.hasNext() && ok) {
                    ok = ((AbstractNode) it.next()).checkChangedAsChild();
                }
            }

            if (!ok) {
                Message message = MessageSystem.getMessageFactory().getMessage(
                        CRuntimeMsg.CC_CHECK_DELETE_MSG);
                MessageReply reply = Application.getMessageDispatcher(
                        getController().getWindow(),
                        getController().getApplication()).dispatch(message);

                ok = MessageReplyType.REPLY_OK.equals(reply);
            }
            return ok;
        }

        @Override
        public void run() {
            result = checkDelete();
        }
    }

    /**
     * The thread that performs check clear
     */
    private class CheckClearTask implements Runnable {

        /** The check clear result */
        private boolean result;

        /**
         * Default constructor.
         */
        CheckClearTask() {
            super();
        }

        /**
         * Check if it is ok to proceed with clear.
         *
         * @return <code>true</code> if it is ok to proceed with action clear.
         */
        @SuppressWarnings("rawtypes")
        private boolean checkClear() {
            boolean ok = true;
            ECheckResult cr = getController().callHookCheckClear(getRoleName());

            if (cr == ECheckResult.CHANGED) {
                ok = false;
            } else if (cr == ECheckResult.DEFAULT) {
                ok = checkChanged();
            }

            if (ok && cr != ECheckResult.UNCHANGED) { // default or
                // continue
                List children = getChildren();
                Iterator it = children.iterator();
                while (it.hasNext() && ok) {
                    ok = ((AbstractNode) it.next()).checkChangedAsChild();
                }
            }

            if (!ok) { // interact
                Message message = MessageSystem.getMessageFactory().getMessage(
                        CRuntimeMsg.CC_CHECK_CLEAR_MSG);
                MessageReply reply = Application.getMessageDispatcher(
                        getController().getWindow(),
                        getController().getApplication()).dispatch(message);

                ok = MessageReplyType.REPLY_OK.equals(reply);
            }
            return ok;
        }

        @Override
        public void run() {
            result = checkClear();
        }
    }

    /**
     * Builds the list of check save nodes above target.
     *
     * @param nodesToCheck
     *            list to build
     * @param currentChild
     *            current child
     * @param asUpRelated
     *            current child is up-related
     */
    @SuppressWarnings("rawtypes")
    private void checkSaveListAboveTarget(List nodesToCheck,
            AbstractNode currentChild, boolean asUpRelated) {

        // First - go to parent if not as part of up-related.
        if (!asUpRelated && getParentNode() != null) {
            AbstractNode parent = (AbstractNode) getParentNode();
            parent.checkSaveListAboveTarget(nodesToCheck, this, false);
        }
        // Then - go to up related children
        List children = getChildren();
        Iterator childIt = children.iterator();
        while (childIt.hasNext()) {
            AbstractNode child = (AbstractNode) childIt.next();
            if (child != currentChild // YES - reference equality
                    && child.getRoleObject().isUpRelated()) {
                child.checkSaveListAboveTarget(nodesToCheck, this, true);
            }
        }

        // Finally, add self to list:
        nodesToCheck.add(this);
    }

    /**
     * Builds the list of check save nodes below target.
     *
     * @param nodesToCheck
     *            list to build.
     */
    @SuppressWarnings("rawtypes")
    private void checkSaveListFromTarget(List nodesToCheck) {
        List childList = getChildren();

        // First, up-related children:

        Iterator it = childList.iterator();
        while (it.hasNext()) {
            AbstractNode child = (AbstractNode) it.next();
            if (child.getRoleObject().isUpRelated()) {
                child.checkSaveListFromTarget(nodesToCheck);
            }
        }

        // Then self
        nodesToCheck.add(this);

        // Finally, other children
        it = childList.iterator();
        while (it.hasNext()) {
            AbstractNode child = (AbstractNode) it.next();
            if (!child.getRoleObject().isUpRelated()) {
                child.checkSaveListFromTarget(nodesToCheck);
            }
        }
    }

    /**
     * The thread that performs check save.
     */
    private class CheckSaveTask extends CheckSaveCommon implements Runnable {

        /** The result of check save */
        private boolean result;

        @SuppressWarnings("rawtypes")
        private AbstractNode target;

        /**
         * Constructs a new CheckSaveTask
         *
         * @param node
         *            the node to check save
         */
        @SuppressWarnings("rawtypes")
        CheckSaveTask(AbstractNode node) {
            this.target = node;
        }

        /**
         * Performs check save
         *
         * @return <code>true</code> if ok to proceed with save
         */
        @SuppressWarnings("rawtypes")
        private boolean checkSave() {

            List nodesToCheck = new LinkedList();

            // First add parents (and uprelated from parent)
            AbstractNode parent = (AbstractNode) getParentNode();
            if (parent != null) {
                parent.checkSaveListAboveTarget(nodesToCheck, target, false);
            }

            // Add self and children:
            checkSaveListFromTarget(nodesToCheck);

            // Perform check:
            ECheckResult cr = doCheckSave(nodesToCheck);

            return cr != null && cr != ECheckResult.CHANGED;
        }

        @Override
        public void run() {
            result = checkSave();
        }

    }

    @Override
    @SuppressWarnings("rawtypes")
    public List<AbstractNode> getPathToNode() {
        List<AbstractNode> path;
        if (getParentNode() != null) {
            path = getParentNode().getPathToNode();
        } else {
            path = new LinkedList<AbstractNode>();
        }
        path.add(this);
        return path;
    }

    @Override
    public ObjectSelectionNode getRootNode() {
        if (getParentNode() != null) {
            return getParentNode().getRootNode();
        }
        return this;
    }

    /**
     * Check if this is a root node and if so, return <code>true</code>
     *
     * @return <code>true</code> if this is a root node
     */
    public boolean isRootNode() {
        return getParentNode() == null;
    }

    /**
     * Returns a Set of Strings containing the names of all children
     * (recursively).
     *
     * @return the children's names.
     */
    @SuppressWarnings("rawtypes")
    public Set getAllChildNames() {
        Set childrenNames = new HashSet();
        Iterator childIterator = null;
        if (getChildren() != null) {
            childIterator = getChildren().iterator();
        }
        if (childIterator != null) {
            while (childIterator.hasNext()) {
                AbstractNode child = (AbstractNode) childIterator.next();
                childrenNames.add(child.getRoleName());
                childrenNames.addAll(child.getAllChildNames());
            }
        }
        return childrenNames;
    }

    /**
     * Internal use.
     * <p>
     * Merge the associations from the specified domain object with the stored
     * associations. This is done after return from a server action.
     * <p>
     * Before the object tree is sent to the server, all objects not on path to
     * target (and not up-related) are stripped away and stored in order to
     * reduce network traffic. This method restores the associations, merging
     * the result from the server with the stored associations.
     *
     * @param domainObject
     *            the domain object
     * @param pathToTarget
     *            the path to the target target.
     * @param error
     *            true if rhe merge is after an error situation
     */
    @SuppressWarnings("rawtypes")
    public abstract void mergeAssociations(Object domainObject,
            List pathToTarget, boolean error);

    /**
     * Internal use.
     * <p>
     * Maps role names to an association
     */
    @SuppressWarnings("rawtypes")
    private Map roleNameToAssociation = new HashMap();

    /**
     * Internal use.
     * <p>
     * Saves the association
     *
     * @param roleName
     *            the role name of the domain object's association
     * @param domainObject
     *            the domain object
     * @param association
     *            the association
     * @see #mergeAssociations(Object, List, boolean)
     */
    @SuppressWarnings("rawtypes")
    protected void saveAssociation(String roleName, Object domainObject,
            Object association) {
        if (DomainUtil.isLazy(association)) {
            return;
        }
        ObjectToAssociation newAssociation = new ObjectToAssociation(roleName,
                domainObject, association);
        Map keptAssociations = (Map) roleNameToAssociation.get(roleName);
        if (keptAssociations == null) {
            keptAssociations = new HashMap();
            roleNameToAssociation.put(roleName, keptAssociations);
        }
        keptAssociations.put(newAssociation, newAssociation);
    }


    /**
     * Internal use.
     * <p>
     * Gets the stored association with the specified role name belonging to the
     * specified domain object.
     *
     * @param roleName
     *            the role name of the association
     * @param domainObject
     *            the domain object the association belongs to
     * @return the association
     */
    @SuppressWarnings("rawtypes")
    protected Object getAssociation(String roleName, Object domainObject) {
        ObjectToAssociation key = new ObjectToAssociation(roleName,
                domainObject);
        Object association = null;
        Map keptAssociations = (Map) roleNameToAssociation.get(roleName);
        if (keptAssociations != null) {
            ObjectToAssociation objToAss = (ObjectToAssociation) keptAssociations
                    .get(key);
            if (objToAss != null) {
                association = objToAss.getAssociation();
            }
        }
        return association;
    }



    /**
     * Internal use.
     * Removes the specified role instance from the stored set of
     * instances.
     * @param root the root instance.
     */
    @SuppressWarnings("rawtypes")
    public void removeFromParent(Object root) {

        Object association = getAssociationToSelfFromParent(root);

        Object selfInstance = getSelfInstance(root);

        if (association instanceof Collection) {
            Collection collection = (Collection) association;
            collection.remove(selfInstance);
        }
    }

    /**
     * Experimental. Application programmers should <em>NOT!!!</em>
     * invoke this method.
     * @return map of role names to associations
     */
    // Instrumented especially for TAD experiment.
    @SuppressWarnings("rawtypes")
    public Map getRoleNameToAssociationMap() {
    	return roleNameToAssociation;
    }

    /**
     * Clears all associations for the role.
     */
    protected void clearAllAssociation() {
    	roleNameToAssociation.clear();

    }

	@SuppressWarnings("rawtypes")
    private Object getSelfInstance(Object root) {
		List pathToSelf = getPathToNode();
        Object selfInstance = navigateToTarget(root, pathToSelf);
		return selfInstance;
	}

	@SuppressWarnings("rawtypes")
    private Object getAssociationToSelfFromParent(Object root) {
		AbstractNode parentNode = isRootNode() ? this : (AbstractNode) getParentNode();
        Object parentInstance = navigateToTarget(root, parentNode.getPathToNode());

        Object association = parentNode.getAssociation(getRoleName(), parentInstance);
		return association;
	}


    @SuppressWarnings("rawtypes")
    private static Object navigateToTarget(Object root,
            List<AbstractNode> pathToTarget) {
        Object found = root;
        for (int i = 0; i < pathToTarget.size() - 1; i++) {
            AbstractNode par = pathToTarget.get(i);
            RoleObject parentRole = par.getRoleObject();

            AbstractNode child = pathToTarget.get(i + 1);
            // found = par.getAssociation(child.getRoleName(), found);
            AssociationAccess associationAccess = parentRole
                    .getAssociationAccess(child.getRoleObject());
            found = associationAccess.getAssociationOnType(found);
            if (found instanceof Collection) {
                Collection tmp = (Collection) found;
                if (tmp.size() != 1) {
                    throw new G9ClientException("Can't navigate to target instance.");
                }
                found = tmp.iterator().next();
            }
        }
        return found;

    }

    /**
     * Internal use
     * <p>
     * Adds the specified association to the stored associations
     *
     * @param roleName
     *            the role name of the association
     * @param domainObject
     *            the domain object with the association
     * @param association
     *            the association
     */
    @SuppressWarnings("rawtypes")
    protected void addToAssociation(String roleName, Object domainObject,
            Set association) {
        Set keptAssociation = (Set) getAssociation(roleName, domainObject);
        if (keptAssociation != null) {
            keptAssociation.addAll(association);
        } else {
            saveAssociation(roleName, domainObject, association);
        }
    }

    /**
     * Internal use.
     * <p>
     * Class mapping a domain object to the association
     * @param <T> the type.
     */
    @SuppressWarnings("hiding")
    private class ObjectToAssociation<T> {

        /** The domain object */
        private Object domainObject;

        /** The association */
        private Object association;

        /** The name of the association */
        private String roleName;

        /**
         * Internal use.
         * <p>
         * Constructs a new object to association map object
         *
         * @param roleName
         *            the name of the association
         * @param domainObject
         *            the domain object with the association
         * @param association
         *            the association
         */
        @SuppressWarnings("rawtypes")
        ObjectToAssociation(String roleName, Object domainObject,
                Object association) {
            this.roleName = roleName;
            this.domainObject = domainObject;
            if (association instanceof Set) {
                this.association = TypeTool.getDefaultSet();
                ((Set) this.association).addAll((Set) association);
            } else {
                this.association = association;
            }
        }

        /**
         * Internal use
         * <p>
         * Creates a new object to association map object, where the association
         * is <code>null</code>
         *
         * @param roleName
         *            the name of the association
         * @param domainObject
         *            the domain object with the association
         */
        ObjectToAssociation(String roleName, Object domainObject) {
            this(roleName, domainObject, null);
        }

        /**
         * Internal use
         * <p>
         * Returns the asscoation form this association map
         *
         * @return the association
         */
        private Object getAssociation() {
            return association;
        }

        @SuppressWarnings("rawtypes")
        @Override
        public boolean equals(final Object o) {
            if (o == this) {
                return true;
            }
            if (!(o instanceof ObjectToAssociation)) {
                return false;
            }
            ObjectToAssociation other = (ObjectToAssociation) o;
            return domainObject.equals(other.domainObject);
        }

        @Override
        public int hashCode() {
            return domainObject.hashCode();
        }

        @Override
        public String toString() {
            return "ObjectToAssociation, hashCode(" + hashCode() + "): "
                    + roleName;
        }
    }

    /**
     * Internal use. Restores the associations for a given root target. Used
     * internally after a find action on a sub-branch of the object selection.
     *
     * @param o
     *            the domain object for which to restore associations.
     */
    public void resetAssociations(Object o) {
        // overridden by root-nodes.
    }

    @Override
    public boolean isChanged(String attributeName) throws G9ClientException {
        String qualifiedAttributeName = TypeTool.addRoleNamePrefix(
                attributeName, getRoleName());
        JComponent component = getController().getView().fromNameToComponent(
                qualifiedAttributeName);
        if (component instanceof G9ValueState) {
            G9ValueState stateComponent = (G9ValueState) component;
            return stateComponent.isChanged();
        }
        throw new G9ClientException("No such attribute: " + attributeName);
    }

    /**
     * Returns <code>true</code> if this role is a listblock role.
     *
     * @return <code>true</code> if this is a listblock role.
     */
    @Override
    public boolean inListblock() {
        return false;
    }

    /**
     * Internal use. <strong>Application programmers should not use this method
     * as it might change in incompatible ways between releases or even patches.
     * </strong> Sets an association from <code>self</code> to the
     * <code>association</code> with the role <code>assocationRoleName</code>.
     *
     * @param self
     *            the object to set the association from
     * @param associationRoleName
     *            the association name
     * @param association
     *            the association instance.
     */
    protected abstract void setAssociation(Object self,
            String associationRoleName, Object association);

    /**
     * Internal use. <strong>Application programmers should not use this method
     * as it might change in incompatible ways between releases or even patches.
     * </strong> Check if the role is navigable to parent.
     *
     * @return <code>true</code> if role is navigable to parent.
     */
    public abstract boolean isNavigableToParent();

    /**
     * Internal use. <strong>Application programmers should not use this method
     * as it might change in incompatible ways between releases or even patches.
     * </strong> Obtains the role. If it is in a listblock, all instances are
     * obtained, and a set of domain instances are returned. Otherwise,
     * recursively obtains "self" and children unless the node (and all its
     * children) is in the CLEARED state.
     *
     * @return the obtained instance or instances
     */
    @Override
    @SuppressWarnings("rawtypes")
    public Object greedyObtain() {
        if (inListblock()) {
            return obtainSet(false);
        }

        // Check if this node, and all child nodes of this node, is CLEARED.
        // If so, return null.
        if (getState() == ObjectSelectionNode.CLEARED) {
            Iterator childIterator = getChildren().iterator();
            for (;;) {
                if (!childIterator.hasNext()) {
                    // Means we iterated all children, and all were CLEARED
                    return null;
                }
                AbstractNode child = (AbstractNode) childIterator.next();
                if (child.getState() != ObjectSelectionNode.CLEARED) {
                    // Node contains non-CLEARED child. Break and include
                    break;
                }
            }
        }

        Object self = obtainAsObject();
        List childrenList = getChildren();
        Iterator children = childrenList.iterator();
        while (children.hasNext()) {
            AbstractNode childNode = (AbstractNode) children.next();
            Object child = childNode.inListblock() ? childNode.obtainSet(false)
                    : childNode.greedyObtain();
            setAssociation(self, childNode.getRoleName(), child);
            if (childNode.isNavigableToParent()) {
                if (child instanceof Collection) {
                    Iterator childIterator = ((Collection) child).iterator();
                    while (childIterator.hasNext()) {
                        childNode.setAssociation(childIterator.next(),
                                getRoleName(), self);
                    }
                } else if (child != null) {
                    childNode.setAssociation(child, getRoleName(), self);
                }
            }

        }
        return self;
    }

    /**
     * Performs the runnable on the edt.
     *
     * @param task
     *            the task to perform on the edt
     * @param msgArg
     *            the arguments to use if an exception is caught
     */
    private void edt(Runnable task, String msgArg) {
        String msgID = null;
        Exception ex = null;
        if (!SwingUtilities.isEventDispatchThread()) {
            try {
                SwingUtilities.invokeAndWait(task);
            } catch (InterruptedException e) {
                msgID = CRuntimeMsg.CT_INTERRUPTED;
                ex = e;
            } catch (InvocationTargetException e) {
                msgID = CRuntimeMsg.CT_INVOCATION_TARGET;
                if (e.getCause() != null && e.getCause() instanceof Exception) {
                    ex = (Exception) e.getCause();
                } else {
                    ex = e;
                }
            } finally {
                if (msgID != null) {
                    Object[] msgArgs = { this.getClass(), msgArg, ex };
                    Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
                    MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                    throw new G9ClientFrameworkException(ex, msg);
                }
            }
        } else {
            task.run();
        }
    }

    @Override
    public void accept(Visitor<ObjectSelectionNode> visitor) {
        visitor.visit(this);
    }

    @Override
    public void visitBranch(Visitor<ObjectSelectionNode> visitor) {
        accept(visitor);
        List<ObjectSelectionNode> children = getChildren();
        for (ObjectSelectionNode child : children) {
            child.visitBranch(visitor);
        }
    }

}
