/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.awt.Component;
import java.awt.Container;
import java.beans.PropertyVetoException;
import java.lang.reflect.InvocationTargetException;

import javax.swing.JDialog;
import javax.swing.JInternalFrame;
import javax.swing.SwingUtilities;

import no.esito.util.ServiceLoader;
import no.g9.client.support.message.ClientDispatcherContext;
import no.g9.exception.G9BaseException;
import no.g9.exception.G9ClientFrameworkException;
import no.g9.message.*;
import no.g9.service.JGrapeService;
import no.g9.service.print.ExportService;
import no.g9.service.print.PrintService;
import no.g9.support.ClientContext;
import no.g9.support.xml.XmlConverter;

/**
 * Common super class for all generated applications.
 */
public abstract class Application implements ApplicationMethods {

    /** Reference to the current application */
    protected ApplicationMethods currentApplication;

    /** Reference to the client context */
    private static ClientContext clientContext;

    /** Reference to the service proxy */
    private static JGrapeService serviceProxy;

    /** Reference to the xml converter */
    private static XmlConverter xmlConverter;

    /** Reference to the print service. */
    private static PrintService printService;

    /** Reference to the export service. */
    private static ExportService exportService;

    /** Reference to the last exception */
    private RuntimeException lastException;

    /**
     * If this flag is true, caught trhowable's stack trace
     * is printed to std. err. (Throwable.printStackTrace()).
     */
    private static boolean printStackTrace;



    /**
     * If this flag is true, foreign nodes are checked as part of
     * the checkRowSelect that is run when a listblock line selection changes.
     * Instances of listblock methods might override this value.
     */
    private boolean checkForeignNodes = true;

    /**
     * Default constructor. Instantiates Application and initializes the message
     * central.
     */
    public Application() {
        // Empty
    }

    /**
     * Set lastException.
     *
     * @param e (missing javadoc)
     */
    @Override
    public void setLastException(RuntimeException e) {
        lastException = e;
    }

    /**
     * @return lastException
     */
    @Override
    public RuntimeException getLastException() {
        return lastException;
    }

    /**
     * Application-wide default handler for all Runtime Exceptions. Override in
     * subclass and/or override dialogExceptionHandler in each dialog.
     *
     * @param bad the caught exception
     * @return true to continue error handling, false if the error is handled
     */
    @Override
    public boolean applicationExceptionHandler(final Throwable bad) {
        if (isPrintStackTrace()) {
            bad.printStackTrace();
        }
        Message message;
        if (bad instanceof G9BaseException) {
            message = ((G9BaseException)bad).getErrMsg();
            if (message.getMessageText().length() == 0) {
                message = MessageSystem.getMessageFactory().getMessage(message.getMessageID(), message.getArgs());
                ((G9BaseException) bad).setErrMsg(message);
            }
            message.setException(bad);
            getMessageDispatcher(getCurrentApplication().getApplicationWindow(), this).dispatch(message);
            return false;
        }
        message = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.UNHANDLED_EXCEPTION, bad.toString());
        message.setException(bad);
        getMessageDispatcher(getCurrentApplication().getApplicationWindow(), this).dispatch(message);
        return true;
    }

    /**
     * Returns the service proxy
     *
     * @return the service proxy
     */
    public static JGrapeService getServiceProxy() {
        // lazy loading
        if (Application.serviceProxy == null) {
            Application.setServiceProxy(ServiceLoader
                    .getService(JGrapeService.class));
        }
        return Application.serviceProxy;
    }

    /**
     * Sets the service proxy for the application
     *
     * @param serviceProxy the service proxy to set.
     */
    public static void setServiceProxy(JGrapeService serviceProxy) {
        Application.serviceProxy = serviceProxy;
    }

    /**
     * @return Returns the clientContext
     */
    public static ClientContext getClientContext() {
        if (Application.clientContext == null) {
            Application.clientContext = new ClientContext();
        }
        return Application.clientContext;
    }

    /**
     * @param clientContext the clientContext to set
     */
    public static void setClientContext(ClientContext clientContext) {
        Application.clientContext = clientContext;
    }

    /**
     * @return Returns the currentApplication.
     */
    public ApplicationMethods getCurrentApplication() {
        return currentApplication;
    }

    /**
     * @param currentApplication The currentApplication to set.
     */
    public void setCurrentApplication(ApplicationMethods currentApplication) {
        this.currentApplication = currentApplication;
    }

    /**
     * @return Returns the activeInternalWindow.
     */
    @Override
    public G9DialogController getActiveInternalWindow() {
        JInternalFrame frame[] = getG9DesktopPane().getAllFrames();
        G9DialogFrame activeFrame = null;

        for (int i = 0; i < frame.length && activeFrame == null; i++) {
            if (frame[i].isSelected()) {
                activeFrame = (G9DialogFrame) frame[i];
            }
        }

        return activeFrame != null ? activeFrame.getController() : null;
    }


    /**
     * Returns the xml converter
     *
     * @return the xml converter
     */
    public static XmlConverter getXmlConverter() {
        // Lazy loading
        if (Application.xmlConverter == null) {
            Application.setXmlConverter(ServiceLoader
                    .getService(XmlConverter.class));
        }
        return Application.xmlConverter;
    }

    /**
     * Gets the print service for the application.
     *
     * @return the print service
     */
    public static PrintService getPrintService() {
        // Lazy loading
        if (Application.printService == null) {
            Application.setPrintService(ServiceLoader
                    .getService(PrintService.class));
        }
        return Application.printService;
    }

    /**
     * Gets the export service
     *
     * @return the export service
     */
    public static ExportService getExportService() {
        // Lazy loading
        if (Application.exportService == null) {
            Application.setExportService(ServiceLoader
                    .getService(ExportService.class));
        }
        return Application.exportService;
    }

    /**
     * A routine to close all open dialogs - part of the standard exit action.
     * After all dialogs are closed, the visitor's <code>done()</code> method is
     * invoked.
     *
     * @param visitor A visitor that is called for each dialog. If the visitor
     *            returns <code>true</code>, no close operation is attempted. If
     *            it throws an exception, the entire loop is aborted. After all
     *            dialogs have been visited, the vistors
     * @param isExitTask <code>true</code> to actually exit the application at
     *            the end of the the loop
     */
    public void dialogCloser(final DialogVisitor visitor,
                             final boolean isExitTask) {

        // Task to perform on edt. If the dialog listens on
        // the close WM event, the task will be routed to
        // the G9Worker instance of that dialog.
        class CloseDialogTask implements Runnable {
            G9DialogController gdc;

            /**
             * Constructs a new CloseDialogTask
             *
             * @param gdc the dialog to close
             */
            CloseDialogTask(G9DialogController gdc) {
                this.gdc = gdc;
            }

            @Override
            public void run() {

                if (gdc.getView().isIcon()) {
                    try {
                        gdc.getView().setIcon(false);
                    } catch (PropertyVetoException e) {
                       // Not much to do about it...
                    }
                }
                gdc.getView().doDefaultCloseAction();
            }
        }

        class CloseHiddenTask implements Runnable {
            G9DialogController gdc;

            /**
             * Constructs a new CloseHiddenTask
             *
             * @param gdc the dialog to close
             */
            CloseHiddenTask(G9DialogController gdc) {
                this.gdc = gdc;
            }

            @Override
            public void run() {
                Container container = gdc.getWindow();
                if (container != null && !container.isVisible()) {
                    if (container instanceof JDialog) {
                        JDialog dView = (JDialog) container;
                        dView.dispose();
                    }
                    if (container instanceof JInternalFrame) {
                        JInternalFrame fView = (JInternalFrame) container;
                        fView.dispose();
                    }
                    getAddedWindows().remove(gdc);
                }
            }
        }

        // Task to perform on dedicated thread.
        class WorkerTask implements Runnable {
            @Override
            public void run() {
                try { // finally visitor.done();
                    int rest = 0;
                    Object[] foo = getAddedWindows().toArray();
                    for (int i = 0; i < foo.length; i++) {
                        G9DialogController gdc = (G9DialogController) foo[i];

                        // Visitor hook
                        if ( visitor.visit(getCurrentApplication(), gdc) ) {
                            continue;  // Taken care of
                        }

                        if (!gdc.getWindow().isVisible()) { // dialog is hidden, dispose it.
                            if (gdc.getWindow() instanceof JDialog) {
                                JDialog dView = (JDialog) gdc.getWindow();
                                dView.dispose();
                            }
                            if (gdc.getWindow() instanceof JInternalFrame) {
                                JInternalFrame fView = (JInternalFrame) gdc.getWindow();
                                fView.dispose();
                            }
                            getAddedWindows().remove(gdc);
                            continue;
                        }
                        // Invoke dialog's close action on EDT, wait for return.

                        setLastException(null);
                        CloseDialogTask cdt = new CloseDialogTask(gdc);
                        String msgID = null;
                        Exception ex = null;
                        try {
                            SwingUtilities.invokeAndWait(cdt);
                        } catch (InterruptedException e) {
                            msgID = CRuntimeMsg.CT_INTERRUPTED;
                            ex = e;
                        } catch (InvocationTargetException e) {
                            msgID = CRuntimeMsg.CT_INVOCATION_TARGET;
                            ex = e;
                        } finally {
                            if (msgID != null) {
                                Object[] msgArgs = { this.getClass(),
                                        "actionExit", ex };
                                Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
                                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                                throw new G9ClientFrameworkException(ex,
                                        msg);
                            }
                        }

                        // The above invoked close action is sent to the
                        // G9Worker que on that dialog. Here - we retrive the
                        // same worker, and wait for it to finish.
                        G9Worker gdcWorker = G9Worker.getWorker(gdc);
                        gdcWorker.waitOnQueue();

                        // Do an extra check - closed window might just have been hidden
                        // - now close it!
                        CloseHiddenTask cht = new CloseHiddenTask(gdc);
                        ex = null;
                        try {
                            SwingUtilities.invokeAndWait(cht);
                        } catch (InterruptedException e) {
                            msgID = CRuntimeMsg.CT_INTERRUPTED;
                            ex = e;
                        } catch (InvocationTargetException e) {
                            msgID = CRuntimeMsg.CT_INVOCATION_TARGET;
                            ex = e;
                        } finally {
                            if (msgID != null) {
                                Object[] msgArgs = {this.getClass(), "actionExit", ex};
                                Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
                                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                                throw new G9ClientFrameworkException(ex, msg);
                            }
                        }

                        if (getAddedWindows().contains(gdc)) {
                            if (!gdc.getView().isVisible()) { // dialog is hidden, dispose it.
                                gdc.getView().dispose();
                                getAddedWindows().remove(gdc);
                            } else {
                                if (getLastException() == null)
                                    break;
                                rest++;
                            }
                        }
                    }

                    // The following is performed only for exit tasks
                    if ( !isExitTask ) {
                        return;
                    }
                    if (getAddedWindows().size() == 0) {
                        System.exit(0); // ??
                    }
                    else if (rest >= getAddedWindows().size()) {
                        // Some windows did not close due to unhandled RuntimeException.
                        // Ask if user want to exit application anyway
                        Message message = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.COULD_NOT_CLOSE_ALL);
                        MessageReply reply = getMessageDispatcher(null, Application.this).dispatch(message);

                        if (MessageReplyType.REPLY_YES.equals(reply)) {
                            System.exit(0);
                        }
                    }
                } finally {
                    visitor.done();
                }
            }
        }

        G9Worker.enqueueTask(this, new WorkerTask());
    }

    /** Standard exit application. */
    @Override
    public void actionExit() {
        dialogCloser(new DialogVisitor() {
            @Override
            public boolean visit(final ApplicationMethods pApp,
                                 final G9DialogController pGdc) {
                return false;
            }

            @Override
            public void done() {
                // empty implementation.

            }
        }, true);
    }

    /**
     * Run a visitor on all open dialogs, and then invokes <code>done()</code>
     * on the visitor.
     *
     * @param pVisitor The actual {@link no.g9.client.support.DialogVisitor}
     * @return <code>true</code> if all visited dialogs return <code>true</code>
     *         , otherwise false
     */
    @Override
    public boolean visitAll(final DialogVisitor pVisitor)  {
        boolean result= true;
        final Object dialogs[] = getAddedWindows().toArray();
        for (int i = 0; i < dialogs.length; i++) {
            G9DialogController dialog=  (G9DialogController) dialogs[i];
            result &= pVisitor.visit(this, dialog);
        }
        pVisitor.done();
        return result;
    }

    /**
     * Add a new window.
     *
     * @param controller the window to add
     */
    @Override
    public void addWindow(G9DialogController controller) {
        addWindow(controller, true);
    }

    /**
     * @return the printStackTrace
     */
    public static boolean isPrintStackTrace() {
        return Application.printStackTrace;
    }

    /**
     * If set to true, stack trace is printed in applicationExceptionHandler
     *
     * @param printStackTrace the printStackTrace to set
     */
    public static void setPrintStackTrace(boolean printStackTrace) {
        Application.printStackTrace = printStackTrace;
    }

    @Override
    public boolean checkForeignNodes() {
        return checkForeignNodes;
    }


    @Override
    public void setCheckForeignNodes(boolean doCheck) {
        checkForeignNodes = doCheck;
    }

    /**
     * Sets the xml converter service for the application
     *
     * @param xmlConverter the xml converter service
     */
    public static void setXmlConverter(XmlConverter xmlConverter) {
        Application.xmlConverter = xmlConverter;
    }

    /**
     * Sets the print service for the application.
     *
     * @param printService Print service.
     */
    public static void setPrintService(PrintService printService) {
        Application.printService = printService;
    }

    /**
     * Sets the export service for the application.
     *
     * @param exportService Export service.
     */
    public static void setExportService(ExportService exportService) {
        Application.exportService = exportService;
    }

    /**
     * Return the message dispatcher.
     *
     * @param owner the JFC component.
     * @param application the JFC application
     * @return the message dispatcher.
     */
    public static MessageDispatcher getMessageDispatcher(Component owner,
            final ApplicationMethods application) {
        return MessageSystem.getMessageDispatcher(getDispatcherContext(owner, application));
    }

    /**
     * Return a JFC dispatcher context with the given component.
     *
     * @param component the JFC component
     * @param application the JFC application
     * @return a new JFC dispatcher context
     */
    public static DispatcherContext getDispatcherContext(
            final Component component, final ApplicationMethods application) {
        return new ClientDispatcherContext() {

            @Override
            public Component getOwner() {
               return component;
            }

            @Override
            public ApplicationMethods getApplication() {
                return application;
            }

            @Override
            public String toString() {
                return "ClientDispatcherContext ["
                + component + "]";
            }

        };
    }

}
