/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.text.ParseException;
import java.util.Map;
import java.util.WeakHashMap;

import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JToggleButton;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.JTextComponent;

import no.g9.client.component.G9ComboBox;
import no.g9.client.event.G9ValueState;

/**
 * Utility class for linking two gui components together. The link is a one-way
 * link, so that the state in the from-component is reflected in the
 * to-component.
 * <p>
 * It is possible to create more than one link from or to a component, and set
 * up links both ways between two components, or a graph for that matter.
 * <p>
 * The link between components are created by simply calling the overloaded
 * link-method, e.g: <code>
 * JTextField f1, f2;
 * ComponentLink.link(f1, f2);
 * </code>
 * <p>
 * The possible link types are:
 * <ul>
 * <li>text -&gt; text
 * <li>combo -&gt; text
 * <li>combo -&gt; combo
 * <li>radio button -&gt; radio button
 * <li>radio button -&gt; check box
 * <li>check box -&gt; check box
 * <li>check box -&gt; radio button
 * </ul>
 * This list might be extended in future releases.
 */
@SuppressWarnings({"unchecked", "rawtypes"})
public class ComponentLink {

    /**
     * Map that should contain all components involved in some link. (Part of
     * avoiding infinity loops caused by circular links.)
     */
    private static Map linkGraf = new WeakHashMap();

    /**
     * State class used in assosiation with components involved in some link.
     * This state class contributes to the solution that avoid infinity loops
     * caused by circular links. (Note that links are created by using
     * expressions in g9.)
     */
    private static class Flag {
    	
        /** The flag */
        private boolean flag = false;

    	/**
    	 * Default constructor.
    	 */
    	Flag() {
    		super();
    	}

        /** Sets the flag to raised, that is true */
        public void raiseFlag() {
            flag = true;
        }

        /** Sets the flag to lowered, that is false */
        public void lowerFlag() {
            flag = false;
        }

        /**
         * Returns true if the flag is raised
         * 
         * @return true if flag is raised
         */
        public boolean isRaised() {
            return flag;
        }
    }

    /**
     * Returns true and raises the flag assosiated with <code>from</code> if the
     * <code>to</code> component is writable, otherwise false is returned. Note
     * that each compoent is assosiated with a Flag. If the flag of a component
     * is raised it means that it is in the midle of an update and therefore is
     * not writable (,that is should not be updated again before the started
     * update is finished). If an update event trigers an update on an component
     * in the midle of an update, a link loop is detected, and should be
     * terminated by simply doing nothing in replay to the last event.
     * 
     * @param from the component to read a value from
     * @param to the component to write the from value into
     * @return true if the <code>to</code> component is writable, otherwise
     *         false.
     */
    private static boolean getFlag(final JComponent from,
            final JComponent to) {
        Flag fromFlag = (Flag) linkGraf.get(from);
        Flag toFlag = (Flag) linkGraf.get(to);
        if (toFlag.isRaised()) {
            return false;
        } 
        fromFlag.raiseFlag();
        return true;
    }

    /**
     * Signals that an update is finished. The Flag assosiated with <code>from
     * </code> is lowered.
     * 
     * @param from the component to read a value from
     * @see #getFlag(JComponent, JComponent)
     */
    private static void returnFlag(final JComponent from) {
        Flag fromFlag = (Flag) linkGraf.get(from);
        fromFlag.lowerFlag();
    }

    /**
     * Assigns a new Flag with each of the the components <code>from</code> and
     * <code>to</code> components, if the component is not yet assigned with a
     * Flag.
     * 
     * @param from the component to read a value from
     * @param to the component to write the from value into
     * @see #getFlag(JComponent, JComponent)
     */
    private static void addToGraph(final JComponent from,
            final JComponent to) {
        if (!linkGraf.containsKey(from)) {
            linkGraf.put(from, new Flag());
        }
        if (!linkGraf.containsKey(to)) {
            linkGraf.put(to, new Flag());
        }
    }

    /**
     * Copies the isChanged state from one component to another
     * 
     * @param from the component to copy state from
     * @param to the component to copy state to
     */
    private static void copyState(Object from, Object to) {
        if (from instanceof G9ValueState
                && to instanceof G9ValueState) {
            G9ValueState fs = (G9ValueState) from;
            G9ValueState ts = (G9ValueState) to;

            ts.setInitialValue(fs.getInitialValue());
        }
    }

    /**
     * Links to <code>JTextComponent</code>s such that the text in the <code>
     * toTextComponent</code> is the same as that in <code>fromTextComponent
     * </code>
     * 
     * @param fromTextComponent the source of the link
     * @param toTextComponent the target of the link
     * @param attributeName the name of the attribute
     * @param controller the dialog controller of the attribute
     * @param nodeName the node name of the attribute
     */
    public static void link(final JTextComponent fromTextComponent,
            final JTextComponent toTextComponent,
            final String attributeName,
            final G9DialogController controller, final String nodeName) {

        addToGraph(fromTextComponent, toTextComponent);

        fromTextComponent.getDocument().addDocumentListener(
                new DocumentListener() {
                    @Override
                    public void insertUpdate(DocumentEvent e) {
                        updateComponent();
                    }
                    @Override
                    public void removeUpdate(DocumentEvent e) {
                        updateComponent();
                    }
                    @Override
                    public void changedUpdate(DocumentEvent e) {
                        updateComponent();
                    }
                    private void updateComponent() {
                        if (getFlag(fromTextComponent, toTextComponent)) {
                            copyState(fromTextComponent, toTextComponent);
                            ((G9ValueState) toTextComponent)
                                    .display(((G9FieldValue) fromTextComponent)
                                            .getValue());
                            returnFlag(fromTextComponent);

                        }
                    }
                });
    }

    /**
     * Links a <code>JComboBox</code> to a <code>JTextComponent</code> such that
     * the selected item in the specified combo box is displayed in the text
     * component. The displayed text is acquired by invoking the <code> toString
     * </code> method on the selected item.
     * 
     * @param fromComboBox the source of the link
     * @param toTextComponent the target of the link
     * @param attributeName the name of the attribute
     * @param controller the dialog controller of the attribute
     * @param nodeName the node name of the attribute
     */
    public static void link(final JComboBox fromComboBox,
            final JTextComponent toTextComponent,
            final String attributeName,
            final G9DialogController controller, final String nodeName) {

        addToGraph(fromComboBox, toTextComponent);
        
        fromComboBox.addActionListener(new ActionListener() {
        
            @Override
            public void actionPerformed(ActionEvent e) {
                updateComponent();
            }
        
            private void updateComponent() {
                if (getFlag(fromComboBox, toTextComponent)) {
                    Object initialValue = ((G9ComboBox) fromComboBox).getInitialValue();
                    initialValue = ((G9ComboBox) fromComboBox).format(initialValue);
                    if (initialValue != null) {
                        try {
                            initialValue = ((G9FieldValue) toTextComponent).parse((String) initialValue);
                        } catch (ParseException e1) {
                            // Can't do much about it other than trying to 
                            // set the initial value as is.
                        }
                    }
                    ((G9ValueState) toTextComponent).setInitialValue(initialValue);
                    Object item = null;
                    item = fromComboBox.getItemAt(fromComboBox.getSelectedIndex());
                    item = ((G9ComboBox) fromComboBox).format(item);
                    if (item != null) {
                        try {
                            item = ((G9FieldValue) toTextComponent).parse((String) item);
                        } catch (ParseException e) {
                            // Can't do much about it other than trying to 
                            // display the unparsable item.
                        }
                    }
                    
                    ((G9ValueState) toTextComponent).display(item);
                    returnFlag(fromComboBox);
                }
            }
        });
    }

    /**
     * Links two <code>JComboBox</code>es so that when a row is selected in the
     * source combo box, the corresponding row (the row with the same index)
     * gets selected in the target combo box.
     * 
     * @param fromComboBox the source combo box
     * @param toComboBox the target combo box
     * @param attributeName the name of the attribute
     * @param controller the dialog controller of the attribute
     * @param nodeName the node name of the attribute
     */
    public static void link(final JComboBox fromComboBox,
            final JComboBox toComboBox, final String attributeName,
            final G9DialogController controller, final String nodeName) {

        addToGraph(fromComboBox, toComboBox);

        fromComboBox.addItemListener(new ItemListener() {
            @Override
            public void itemStateChanged(ItemEvent e) {
                updateComponent();
            }
            private void updateComponent() {
                if (getFlag(fromComboBox, toComboBox)) {
                    
                    copyState(fromComboBox, toComboBox);
                    G9ComboBox toC = (G9ComboBox) toComboBox;
                    G9ComboBox fromC = (G9ComboBox) fromComboBox;
                    Object item = null;
                    item = fromC.getItemAt(fromC.getSelectedIndex());
                    if (fromC.isDontFire()) {
                        toC.display(item);
                    } else {
                        toC.setSelectedItem(item);
                    }
                    returnFlag(fromComboBox);
                }
            }
        });
    }

    /**
     * Links two toggle buttons (typically check boxex or radio buttons) so that
     * when the source is selected or un-selected, so is the target.
     * 
     * @param fromButton the soucre of the link
     * @param toButton the target of the link.
     * @param attributeName the name of the attribute
     * @param controller the controller of the attribute
     * @param nodeName the node name of the attribute
     */
    public static void link(final JToggleButton fromButton,
            final JToggleButton toButton, final String attributeName,
            final G9DialogController controller, final String nodeName) {

        addToGraph(fromButton, toButton);

        fromButton.addChangeListener(new ChangeListener() {
            @Override
            public void stateChanged(ChangeEvent e) {
                updateComponent();
            }
            private void updateComponent() {
                if (getFlag(fromButton, toButton)) {
                    // AbstractNode methods =
                    // (AbstractNode)
                    // controller.getObjectSelectionNode(nodeName);
                    // methods.display(attributeName, new
                    // Boolean(fromButton.isSelected()), true);
                    copyState(fromButton, toButton);
                    ((G9ValueState) toButton).display(new Boolean(
                            fromButton.isSelected()));
                    returnFlag(fromButton);
                }
            }
        });
    }

    /**
     * Links a text field to a combo box.
     * <p>
     * <strong>This method is not yet implemented</strong>
     * 
     * @param fromTextComponent the source of the link
     * @param toComboComponent the target component
     * @param attributeName the attribute name
     * @param controller the dialog controller
     * @param nodeName the name of the node.
     */
    public static void link(final JTextComponent fromTextComponent,
            final JComboBox toComboComponent, final String attributeName,
            final G9DialogController controller, final String nodeName) {
        
        addToGraph(fromTextComponent, toComboComponent);

        fromTextComponent.getDocument().addDocumentListener(
                new DocumentListener() {
                    @Override
                    public void insertUpdate(DocumentEvent e) {
                        updateComponent();
                    }
                    @Override
                    public void removeUpdate(DocumentEvent e) {
                        updateComponent();
                    }
                    @Override
                    public void changedUpdate(DocumentEvent e) {
                        updateComponent();
                    }
                    private void updateComponent() {
                        if (getFlag(fromTextComponent, toComboComponent)) {
                            copyState(fromTextComponent, toComboComponent);
                            ((G9ComboBox) toComboComponent)
                                    .display(((G9FieldValue) fromTextComponent)
                                            .getValue());
                            returnFlag(fromTextComponent);
                        }
                    }
                });
        
    }
    
}
