/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.awt.Desktop;
import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;

import no.esito.log.Logger;
import no.g9.exception.G9ClientFrameworkException;
import no.g9.message.*;

/**
 * A desktop implementation of the launcher interface. This implementation uses
 * {@link Desktop} to open a file or to browse an URL.
 */
public class DesktopLauncher implements Launcher {

    private static Logger log = Logger.getLogger(Launcher.class);

    @Override
    public void open(File file) {
        Exception exception = null;
        try {
            if (log.isDebugEnabled()) {
                log.debug("Opening file: " + file);
            }
            Desktop.getDesktop().open(file);
        } catch (IOException e) {
            exception = e;
        } catch (IllegalArgumentException e) {
            exception = e;
        } catch (SecurityException e) {
            exception = e;
        } catch (UnsupportedOperationException e) {
            exception = e;
        }

        if (exception != null) {
            log.warn("Caught exception while opening file." , exception);
            Object[] args = { file!=null ? file.toString():"" , exception.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(
                    CRuntimeMsg.FL_OPEN_LOCAL_FILE_ERROR, args);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9ClientFrameworkException(exception, msg);
        }


    }

    @Override
    public void open(URL url) {
        Exception exception = null;
        try {
            if (log.isDebugEnabled()) {
                log.debug("Opening url: " + url);
            }
            Desktop.getDesktop().browse(url.toURI());
        } catch (UnsupportedOperationException e) {
            exception = e;
        } catch (IOException e) {
            exception = e;
        } catch (SecurityException e) {
            exception = e;
        } catch (URISyntaxException e) {
            exception = e;
        }

        if (exception != null) {
            log.warn("Caught exception while opening url." , exception);
            Object[] args = { url!=null ? url.toString():"" , exception.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(
                    CRuntimeMsg.FL_OPEN_REMOTE_URL_ERROR, args);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9ClientFrameworkException(exception, msg);
        }
    }

}
