/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.io.ObjectStreamException;
import java.io.Serializable;

/**
 * The enumeration of legal return values from hooks. The semantics of the
 * return values are described in the hooks using ECheckResult as return
 * type. Comparison of ECheckResult is done by reference equality.
 * 
 */
public final class ECheckResult implements Serializable {
    
    /** Keep track of next ordinal. */
    private static int nextOrdinal = 0;
    
    /** The name of the value, used by toString */
    private final transient String name;
    
    /** The ordinal value. */
    private final int ordinal = nextOrdinal++;

    /**
     * Private constructor. Creates a new CheckResult instance.
     * @param name the name of the check result.
     */
    private ECheckResult(String name) {
        this.name = name;
    }
    
    ///////////////////////////////////////////////////////////////////////////
    // 
    //  NOTE: In order to ensure serialization, new enumerator values 
    //        must be added last. Also note that new enumerator values
    //        must be added to the PRIVATE_VALUES array in the same
    //        order they are declared.
    //
    ///////////////////////////////////////////////////////////////////////////
    
    
    /** The "default" check result. */ 
    public static final ECheckResult DEFAULT = new ECheckResult("default");
    
    /** The "unchanged" check result */
    public static final ECheckResult UNCHANGED = new ECheckResult("unchanged");
    
    /** The "continue" check result */
    public static final ECheckResult CONTINUE = new ECheckResult("continue");
    
    /** The "changed" check result */
    public static final ECheckResult CHANGED = new ECheckResult("changed");
    
    ///////////////////////////////////////////////////////////////////////////
    //
    // Add new enumerator values here. Remember to also update the 
    // PRIVATE_VALUES array.
    //
    ///////////////////////////////////////////////////////////////////////////
    
    
    @Override
    public String toString() {
        return name;
    }
    
    
    /** Used in de-serialization. */
    private static final ECheckResult[] PRIVATE_VALUES = 
        {DEFAULT, UNCHANGED, CONTINUE, CHANGED};
    
    /**
     * Used in de-serialization to ensure that only a single instance of a
     * value exists (in order for comparison by reference to work).
     * 
     * @return the CheckResult with the specified ordinal value.
     * @throws ObjectStreamException
     *             if an exception occurred while de-serializing (should not
     *             happen).
     */
    private Object readResolve() throws ObjectStreamException {
        return PRIVATE_VALUES[ordinal]; 
    }
    
}
