/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.awt.Component;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.MouseEvent;
import java.io.Serializable;
import java.util.EventObject;
import java.util.HashSet;

import javax.swing.JComponent;
import javax.swing.JTable;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.event.CellEditorListener;
import javax.swing.event.ChangeEvent;
import javax.swing.event.EventListenerList;
import javax.swing.table.TableCellEditor;
import javax.swing.tree.TreeCellEditor;

import no.g9.client.component.G9ComboBox;

/** Internal class. */
@SuppressWarnings({"unchecked", "rawtypes"})
public class JComponentCellEditor implements TableCellEditor, TreeCellEditor, Serializable {

    /** List of listeners */
    protected EventListenerList listenerList = new EventListenerList();

    /**
     * The change event
     */
    transient protected ChangeEvent changeEvent = null;

    /** The editor component */
    protected JComponent editorComponent = null;

    /** The container using this editor */
    protected JComponent container = null; /* Can be tree or table */

    /**
     * Internal class.
     * 
     * @return Internal class.
     */
    public Component getComponent() {
        return editorComponent;
    }

    @Override
    public Object getCellEditorValue() {
        return editorComponent;
    }

    @Override
    public boolean isCellEditable(EventObject anEvent) {
        return true;
    }

    
    @Override
    public boolean shouldSelectCell(EventObject anEvent) {
        if (editorComponent != null
                && anEvent instanceof MouseEvent
                && ((MouseEvent) anEvent).getID() == MouseEvent.MOUSE_PRESSED) {
            Component dispatchComponent = SwingUtilities
                    .getDeepestComponentAt(editorComponent, 3, 3);
            MouseEvent e = (MouseEvent) anEvent;
            MouseEvent e2 = new MouseEvent(dispatchComponent,
                    MouseEvent.MOUSE_RELEASED, e.getWhen() + 100000, e
                            .getModifiers(), 3, 3, e.getClickCount(), e
                            .isPopupTrigger());
            dispatchComponent.dispatchEvent(e2);
            e2 = new MouseEvent(dispatchComponent,
                    MouseEvent.MOUSE_CLICKED, e.getWhen() + 100001, e
                            .getModifiers(), 3, 3, 1, e.isPopupTrigger());
            dispatchComponent.dispatchEvent(e2);
        }
        return false;
    }

    
    @Override
    public boolean stopCellEditing() {
        fireEditingStopped();
        return true;
    }

    
    @Override
    public void cancelCellEditing() {
        fireEditingCanceled();
    }

    
    @Override
    public void addCellEditorListener(CellEditorListener l) {
        listenerList.add(CellEditorListener.class, l);
    }

    
    @Override
    public void removeCellEditorListener(CellEditorListener l) {
        listenerList.remove(CellEditorListener.class, l);
    }

    /**
     * Fires the editor stopped event
     */
    protected void fireEditingStopped() {
        Object[] listeners = listenerList.getListenerList();
        /*
         * Process the listeners last to first, notifying those that are
         * interested in this event
         */
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == CellEditorListener.class) {
                // Lazily create the event:
                if (changeEvent == null)
                    changeEvent = new ChangeEvent(this);
                ((CellEditorListener) listeners[i + 1])
                        .editingStopped(changeEvent);
            }
        }
    }

    /**
     * Fire the editing canceled event
     */
    protected void fireEditingCanceled() {
        /* Guaranteed to return a non-null array */
        Object[] listeners = listenerList.getListenerList();
        /*
         * Process the listeners last to first, notifying those that are
         * interested in this event
         */
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == CellEditorListener.class) {
                /* Lazily create the event: */
                if (changeEvent == null)
                    changeEvent = new ChangeEvent(this);
                ((CellEditorListener) listeners[i + 1])
                        .editingCanceled(changeEvent);
            }
        }
    }

    @Override
    public Component getTreeCellEditorComponent(JTree tree, Object value,
            boolean isSelected, boolean expanded, boolean leaf, int row) {
        tree.convertValueToText(value, isSelected, expanded, leaf, row,
                false);
        editorComponent = (JComponent) value;
        container = tree;
        if (editorComponent instanceof G9ComboBox) {
            ((G9ComboBox) editorComponent).editingAboutToBegin();
        }
        return editorComponent;
    }

    /** (missing javadoc)*/
    HashSet addedFocusListener = new HashSet();

    
    @Override
    public Component getTableCellEditorComponent(JTable table,
            Object value, boolean isSelected, int row, int column) {
        editorComponent = (JComponent) value;
        container = table;
        if (editorComponent instanceof G9ComboBox) {
            ((G9ComboBox) editorComponent).editingAboutToBegin();
        }
        if (!addedFocusListener.contains(editorComponent)) {
            addedFocusListener.add(editorComponent);
            editorComponent.addFocusListener(new FocusAdapter() {
                @Override public void focusLost(FocusEvent e) {
                    stopCellEditing();
                }
            });
        }
        return editorComponent;
    }
}
