/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.math.BigDecimal;
import java.text.ParseException;

import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;

import no.esito.log.Logger;
import no.g9.support.FormatHelper;
import no.g9.support.G9Consts;
import no.g9.support.Numeric;

/**
 * <code>Document</code> model for fields with number formatting.
 */
public class NumberDocument extends G9Document {

    private boolean blankWhenZero;
    private int sign; // FormatHelper.SIGN_NONE, SIGN_MINUS, SIGN_FLOATING_MINUS, SIGN_PLUS, SIGN_FLOATING_PLUS
    private int intLength; // Length of integer part
    private int decimalLength; // Length of scale part
    private boolean overflow;
    private boolean isG9Numeric;

    /**
     * Creates a new NumberDocument
     *
     * @param datatype the type of data
     * @param inputFormat the input format
     * @param outputFormat the output format
     * @param maxLength the max length
     * @param sign the sign
     * @param blankWhenZero the blank when zero property
     * @roseuid 44325EED024A
     */
    public NumberDocument(int datatype, String inputFormat, String outputFormat, int maxLength, int sign, boolean blankWhenZero, boolean isG9Numeric) {
        super(datatype, inputFormat, outputFormat, maxLength);

        intLength = inputFormat.indexOf('.');
        overflow = false;
        if (intLength == -1) {
            intLength = inputFormat.length();
            decimalLength = 0;
        }
        else {
            decimalLength = Math.max(inputFormat.length() - intLength - 1,0);
        }
        if (sign != FormatHelper.SIGN_NONE) {
            intLength--;
        }
        setInputFormat(inputFormat);
        setOutputFormat(outputFormat);
        this.sign = sign;
        this.blankWhenZero = blankWhenZero;
        this.isG9Numeric = isG9Numeric;
    }

    /**
     * Determines if the blankWhenZero property is true.
     *
     * @return <code>true</code> if the blankWhenZero property is true
     */
    public synchronized boolean getBlankWhenZero() {
        return blankWhenZero;
    }

    /**
     * Sets the value of the blankWhenZero property.
     *
     * @param aBlankWhenZero the new value of the blankWhenZero property
     */
    public synchronized void setBlankWhenZero(boolean aBlankWhenZero) {
        blankWhenZero = aBlankWhenZero;
    }

    @Override
    public void insertString(int offset, String str, AttributeSet attr) throws BadLocationException {
        if (!getInputMode()) {
            super.insertString(offset,str,attr);
        }
        else if (str != null) {
            String currentText = getText(0,getLength());
            if (currentText == null) {
                currentText = "";
            }
            if (offset < 0) {
                offset = 0;
            }
            if (offset > currentText.length()) {
                offset = currentText.length();
            }
            StringBuffer buffer = (offset > 0) ? new StringBuffer(currentText.substring(0,offset)) : new StringBuffer();
            buffer.append(str);
            buffer.append(currentText.substring(offset));
            if (overflow || checkInputString(getInputFormat(),buffer.toString())) {
                super.insertString(offset, str, attr);
            }
        }
    }

    private boolean isZero(Object value) {
        boolean result = true;
        if (value != null) {
            if (value instanceof Numeric) {
                result = ((Numeric)value).compareTo(new Numeric(0,((Numeric)value).getScale())) == 0;
            }
            else if (value instanceof BigDecimal) {
            	result = ((BigDecimal)value).compareTo(BigDecimal.ZERO.setScale(((BigDecimal)value).scale())) == 0;
            }
            else if (value instanceof Short) {
                result = ((Short)value).compareTo(Short.valueOf((short)0)) == 0;
            }
            else if (value instanceof Integer) {
                result = ((Integer)value).compareTo(Integer.valueOf(0)) == 0;
            }
            else if (value instanceof Long) {
                result = ((Long)value).compareTo(Long.valueOf(0)) == 0;
            }
            else if (value instanceof Float) {
                result = ((Float)value).compareTo(new Float(0.0)) == 0;
            }
            else if (value instanceof Double) {
                result = ((Double)value).compareTo(new Double(0.0)) == 0;
            }
        }
        return result;
    }

    @Override
    public synchronized void setValue(Object value) {
        super.setValue(value);
        String strValue;
        if (blankWhenZero && isZero(value)) {
            strValue = "";
        } else {
            strValue = format(getInputMode() ? getInputFormat() : getOutputFormat(),value,false);
        }
        try {
            replace(0,getLength(),strValue,null);
        } catch (BadLocationException e) {
            Logger.getLogger(NumberDocument.class).warn("Failed to set value", e);
        }
    }

    @Override
    public synchronized String getOutputText() {
        String strValue = null;
        Object value = super.getValue();
        if (value != null) {
            if (blankWhenZero && isZero(value)) {
                strValue = "";
            }
            else {
                strValue = format(getOutputFormat(),value,false);
            }
        }
        return strValue;
    }

    @Override
    public synchronized Object getValue() {
        Object result = super.getValue();
        if (getInputMode()) {
            String strValue = null;
            try {
                strValue = getText(0,getLength());
            } catch (BadLocationException e) {
                Logger.getLogger(NumberDocument.class).warn("Failed to get value", e);
            }
            if (strValue != null) {
                if (!(strValue.length() == 0 && result == null)) {
                    try {
                        result = parse(strValue,false);
                    } catch (ParseException e) {
                        Logger.getLogger(NumberDocument.class).warn("Failed to parse value", e);
                    }
                }
            }
        }
        return result;
    }

    private Object stringToValue(String str) {
        Object retVal;
        try {
            retVal = parse(str, false);
        } catch (ParseException e) {
            retVal = null;
        }
        return retVal;
    }

    @Override
    public synchronized Object transform(Object o) {
        return stringToValue(valueToString(o));
    }


    /**
     * Parse the number string according to the internal format.
     *
     * @param str the string to parse
     * @return number object according to genova datatype
     * @throws ParseException (missing javadoc)
     */
    @Override
    public synchronized Object parse(String str) throws ParseException {
        return parse(str,true);
    }

    /**
     * Parse the number string.
     *
     * @param str the string to parse
     * @param internalFormat internal format or not.
     * @return number object according to genova datatype
     * @throws ParseException (missing javadoc)
     */
    private Object parse(String str,boolean internalFormat) throws ParseException {
        Object retVal = null;
        if (str != null && str.length() != 0) {
            // Convert string so default parsing can be used
            // I.e. remove all but digits and minus and
            // replace local decimal separator with .
            char decimalSeparator = internalFormat ? '.' : FormatHelper.getDecimalSeparator();
            StringBuffer numBuffer = new StringBuffer();
            StringBuffer decimalBuffer = numBuffer;
            int i;
            for (i = 0; i < str.length(); i++) {
                char c = str.charAt(i);
                if (c == '-' && internalFormat && i != 0) {
                    throw new ParseException("Illegal character",i);
                }
                if (Character.isDigit(c) || c == '-') {
                    decimalBuffer.append(c);
                }
                else if (c == decimalSeparator) {
                    if (getDatatype() == G9Consts.DT_SHORTINT || getDatatype() == G9Consts.DT_LONGINT || getDatatype() == G9Consts.DT_LONGLONG) {
                        if (internalFormat) {
                            throw new ParseException("Illegal character",i);
                        }
                        decimalBuffer = new StringBuffer();
                    }
                    else {
                        decimalBuffer.append('.');
                    }
                }
                else if (internalFormat) {
                    throw new ParseException("Illegal character",i);
                }
            }
            int round = 0;
            if (decimalBuffer != numBuffer && decimalBuffer.length() > 0) {
                char c = decimalBuffer.charAt(0);
                if (Character.isDigit(c) && Character.digit(c,10) >= 5) {
                    round = (numBuffer.length() > 0 && numBuffer.charAt(0) == '-') ? -1 : 1;
                }
            }
            try {
                switch (getDatatype()) {
                    case G9Consts.DT_SHORTINT :
                        retVal = Short.valueOf(numBuffer.toString());
                        if (round != 0) {
                            short sh = ((Short)retVal).shortValue();
                            retVal = Short.valueOf((short)(sh + round));
                        }
                        break;
                    case G9Consts.DT_LONGINT :
                        retVal = Integer.valueOf(numBuffer.toString());
                        if (round != 0) {
                            int in = ((Integer)retVal).intValue();
                            retVal = Integer.valueOf(in + round);
                        }
                        break;
                    case G9Consts.DT_LONGLONG :
                        retVal = Long.valueOf(numBuffer.toString());
                        if (round != 0) {
                            long lo = ((Long)retVal).longValue();
                            retVal = Long.valueOf(lo + round);
                        }
                        break;
                    case G9Consts.DT_NUMERIC :
                        if (isG9Numeric) {
                        	retVal = new Numeric(numBuffer.toString(),decimalLength);
                        }
                        else {
                        	retVal = new BigDecimal(numBuffer.toString()).setScale(decimalLength);
                        }
                        break;
                    case G9Consts.DT_REAL :
                        retVal = new Float(numBuffer.toString());
                        break;
                    case G9Consts.DT_DOUBLE :
                        retVal = new Double(numBuffer.toString());
                        break;
                    default:
                        break;
                }
            }
            catch (NumberFormatException e) {
                throw new ParseException(e.getMessage(),0);
            }
        }
        return retVal;
    }

    public Object getZeroValue() {
        switch (getDatatype()) {
            case G9Consts.DT_SHORTINT :
                return Short.valueOf((short) 0);
            case G9Consts.DT_LONGINT :
                return Integer.valueOf(0);
            case G9Consts.DT_LONGLONG :
                return Long.valueOf(0);
            case G9Consts.DT_NUMERIC :
            	if (isG9Numeric) {
            		return new Numeric(0, decimalLength);
            	}
            	else {
            		return BigDecimal.ZERO.setScale(decimalLength);
            	}
            case G9Consts.DT_REAL :
                return Float.valueOf(0);
            case G9Consts.DT_DOUBLE :
                return Double.valueOf(0);
        }
        return null;
    }

    /**
     * @see no.g9.client.support.G9FieldValue#format()
     * @return string representaion of number on internal format
     */
    @Override
    public synchronized String format() {
        return format(getInputFormat(),getValue(),true);
    }

    private String valueToString(Object value) {
        return format(getInputFormat(), value, false);
    }

    /**
     * Format numeric object according to format <br>
     * When internal format the format should always be the input format, i.e.
     * with the signpos to the left and none non digit characters except decimal
     * point.
     *
     * @param format (missing javadoc)
     * @param number (missing javadoc)
     * @param internalFormat (missing javadoc)
     * @return String (missing javadoc)
     */
    private String format(String format, Object number,boolean internalFormat) {
        int sign1 = this.sign;
        if (format.equals(getInputFormat())) {
            if (sign1 == FormatHelper.SIGN_MINUS) sign1= FormatHelper.SIGN_FLOATING_MINUS;
            if (sign1 == FormatHelper.SIGN_PLUS) sign1= FormatHelper.SIGN_FLOATING_PLUS;
        }
        StringBuffer res = new StringBuffer();
        if (number != null) {
            String bStr = number.toString();
            int intPos = 0;
            boolean negative = false;
            if (bStr.length() > 0) {
                if (bStr.length() > 0 && bStr.charAt(intPos) == '-') {
                    negative = true;
                    intPos++;
                }
                char dot = '.';
                int noDigits = bStr.indexOf(dot);
                if (noDigits < 0) noDigits = bStr.length();
                noDigits = noDigits - intPos;
                if (noDigits > intLength || (negative && sign1 == FormatHelper.SIGN_NONE)) {
                    for (noDigits = 0; noDigits < getMaxLength(); noDigits++) {
                        res.append('#');
                    }
                    overflow = true;
                    return res.toString();
                }
                overflow = false;
                int dotPos = format.indexOf('.');
                int signPos = addInt(dotPos < 0 ? format : format.substring(0,dotPos+1),sign1,internalFormat,noDigits < 0 ? "0" : bStr.substring(intPos,noDigits+intPos),intLength,res);
                if (dotPos >= 0) {
                    intPos = bStr.indexOf(dot) + 1;
                    if (intPos > 0 && intPos <= bStr.length()) {
                        noDigits = bStr.length() - intPos;
                        bStr = bStr.substring(intPos);
                        while (decimalLength > noDigits) {
                            bStr = bStr.concat("0");
                            noDigits++;
                        }
                    }
                    else {
                        bStr = "";
                    }
                    int tmp = addInt(format.substring(dotPos+1),sign1,internalFormat, bStr,decimalLength,res);
                    if (signPos == -1) signPos = tmp;
                }
                if (signPos >=0 && sign1 != FormatHelper.SIGN_NONE) {
                    if (negative) {
                        res.insert(signPos,'-');
                    }
                    else if (sign1== FormatHelper.SIGN_PLUS || sign1== FormatHelper.SIGN_FLOATING_PLUS) {
                        res.insert(signPos,'+');
                    }
                    else if (sign1== FormatHelper.SIGN_MINUS ) {
                        res.insert(signPos,' ');
                    }
                }
                if (format.equals(getInputFormat()) || format.charAt(0)== '#' || (format.charAt(0) == '+' && sign1!=FormatHelper.SIGN_MINUS && sign1!=FormatHelper.SIGN_PLUS)) {
                    int i = 0;
                    while (i < res.length() && res.charAt(i)== ' ') i++;
                    if (i > 0) {
                        res.delete(0,i);
                    }
                    i = res.length() - 1;
                    while (i >= 0  && res.charAt(i)== ' ') i--;
                    i++;
                    if (i >= 0) {
                        res.delete(i,res.length());
                    }
                }
            }
        }
        return res.toString();
    }

    /**
     * Add integer to a stringbuffer according to format
     *
     * @param format the string describing the format
     * @param sign1 sign to use
     * @param internalFormat internal format or not
     * @param number the number to format
     * @param maxLength the max length of the number
     * @param buffer the buffer to add the formatted text to.
     * @return the new pos.
     */
    private int addInt(String format,int sign1, boolean internalFormat, String number,int maxLength, StringBuffer buffer) {
        int signPos = -1;
        int resPos = 0;
        int intPos = 0;
        int formatPos = -1;
        int noDigits = number.length();
        for (formatPos = 0; formatPos < format.length(); formatPos++) {
            char c = format.charAt(formatPos);
            switch (c) {
                case '+':
                    signPos = resPos;
                    break;
                case '#':
                case '0':
                    if (noDigits < maxLength) {
                        noDigits++;
                        if (c == '0') {
                            buffer.append('0');
                        } else {
                            buffer.append(' ');
                            if (signPos >= 0 && (sign1 == FormatHelper.SIGN_FLOATING_MINUS || sign1 == FormatHelper.SIGN_FLOATING_PLUS)) {
                                signPos=resPos;
                            }
                        }
                    }
                    else {
                        buffer.append((number.length() == 1 && c == '#' && number.charAt(0) == '0') ? ' ' : number.charAt(intPos));
                        intPos++;
                    }
                    break;
                case ',':
                    buffer.append(FormatHelper.getGroupingSeparator());
                    break;
                case '.':
                    buffer.append(internalFormat ? '.' : FormatHelper.getDecimalSeparator());
                    break;
                default:
                    buffer.append(c);
                    break;
            }
            resPos++;
        }
        return signPos;
    }

    /**
     * Check if input is according to format
     *
     * @param format string describing the format
     * @param input the non-null string to check
     * @return <code>true</code> if input is according to format.
     */
    private boolean checkInputString(String format, String input) {
        assert(input != null);
        // Check maximum length
        if (getMaxLength() > 0 && input.length() > getMaxLength()) {
            return false;
        }
        // Check decimal point
        char dot = FormatHelper.getDecimalSeparator();
        int indexOfFirstDot = input.indexOf(dot);
        if (format.indexOf('.') == -1 && indexOfFirstDot != -1) {
            return false; // No decimal point allowed
        }
        if (indexOfFirstDot != input.lastIndexOf(dot)) {
            return false; // Two decimal points
        }
        // Check sign
        int indexOfMinus = input.indexOf('-');
        int indexOfPlus = input.indexOf('+');
        if ((sign == FormatHelper.SIGN_NONE && (indexOfMinus >= 0 || indexOfPlus >= 0)) ||  // No sign allowed
            ((sign == FormatHelper.SIGN_MINUS || sign == FormatHelper.SIGN_FLOATING_MINUS) && indexOfPlus >= 0) ||  // Plus not allowed
            (indexOfMinus >= 0 && indexOfPlus >= 0) || // Not both signs
            (Math.max(indexOfMinus,indexOfPlus) != Math.max(input.lastIndexOf('-'),input.lastIndexOf('+')))) { // Not two of same sign
            return false;
        }
//        if (indexOfMinus > 0 || indexOfPlus > 0) {
//            return false;
//        }
        // Check legal characters and number of digits
        int noDig = 0;
        int noAllowedDig = intLength;
        for (int i = 0; i < input.length();i++) {
            if (Character.isDigit(input.charAt(i))) {
                noDig++;
                if (noDig > noAllowedDig) {
                    return false;
                }
            }
            else if (input.charAt(i) == dot) {
                noDig = 0;
                noAllowedDig = decimalLength;
            }
            else if ( input.charAt(i) != '-' && input.charAt(i) != '+') {
                return false;
            }
        }
        return true;
    }

}
