/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.awt.Component;
import java.awt.ComponentOrientation;
import java.awt.Font;
import java.awt.LayoutManager;
import java.awt.Toolkit;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import javax.swing.border.Border;

/**
 * Class used for displaying messages or components. Although this class is
 * primarily intended as a status bar (typically at the bottom of the
 * application window), it is a JPanel, so you might find other use for it.
 * <p>
 * The status bar consists of a left and right side, which can be specified when
 * adding components (or text) to the status bar. Components added to the left
 * side, is oriented right-to-left, meaning that the last element added will be
 * the left-most side of the status bar. The right side is oriented
 * left-to-right, thus the last element added will be at the right-most side of
 * the status bar.
 */
public class StatusBar extends JPanel {
    
    /** Constant denoting the left side of the status bar */
    public final static int LEFT = 0;
    
    /** Constant denoting the right side of the status bar */
    public final static int RIGHT = 1;

    /** The left side of the status bar */
    private JPanel leftPane = new JPanel();

    /** The right side of the status bar */
    private JPanel rightPane = new JPanel();

    /** The font used to display text messages */
    private Font msgFont = null;

    /**
     * Default constructor. Creates a new status bar.
     */
    public StatusBar() {
        LayoutManager rightToLeft = new BoxLayout(leftPane, BoxLayout.LINE_AXIS);
        leftPane.setComponentOrientation(ComponentOrientation.RIGHT_TO_LEFT);
        leftPane.setLayout(rightToLeft);
        LayoutManager leftToRight = new BoxLayout(rightPane, BoxLayout.LINE_AXIS);
        rightPane.setComponentOrientation(ComponentOrientation.LEFT_TO_RIGHT);
        rightPane.setLayout(leftToRight);
        
        LayoutManager sLayout = new BoxLayout(this, BoxLayout.LINE_AXIS);
        setLayout(sLayout);
        
        Border border = BorderFactory.createRaisedBevelBorder();
        setBorder(border);
        
        super.add(leftPane);
        super.add(Box.createHorizontalGlue());
        super.add(new JLabel(" "));
        super.add(Box.createHorizontalGlue());
        super.add(rightPane);
    }

    /**
     * Adds a component to the left side of the status bar. This has the same
     * effect as invoking
     * <code>addComponent(component, StatusBar.LEFT, 0, false)</code>
     * 
     * @param component the component to add
     * @return the added component
     */
    @Override
    public Component add(Component component) {
        addComponent(component, LEFT, 0, false);
        
        return component;
    }

    /**
     * Adds the specified text to the left side of the status bar.
     * 
     * @param msg (missing javadoc)
     * @return the added component
     */
    public Component add(String msg) {
        return addComponent(getLabel(msg), LEFT, 0, false);
    }

    /**
     * Add a component to the status bar.The alignment must be either
     * StatusBar.LEFT or StatusBar.RIGHT.
     * 
     * @param component the component to add
     * @param ALIGNMENT constant specifying the alignment, either LEFT or RIGHT
     * @return the added component
     * @see #add(Component)
     */
    public Component addComponent(Component component, final int ALIGNMENT) {
        return addComponent(component, ALIGNMENT, 0, false);
    }

    /**
     * Adds a component to the status bar. All other add-methods are invokes
     * this.
     * 
     * @param component the component to add
     * @param ALIGNMENT constant specifying the alignment, either LEFT or RIGHT
     * @param timer if greater than zero, the amount in milliseconds the
     *            componet shall be on the status bar.
     * @param bell if <code>true</code> the system beeps when the component is
     *            displayed
     * @return the added component
     * @see #add(Component)
     */
    public Component addComponent(final Component component, final int ALIGNMENT, final int timer, boolean bell) {
        switch(ALIGNMENT) {
            case LEFT : leftPane.add(component);
            break;
            case RIGHT : rightPane.add(component);
            break;
            default : throw new IllegalArgumentException("Alignment must be either LEFT or RIGHT");
        }
        
        refreshStatusBar(bell);
        
        if (timer > 0) {
            new Thread(new Runnable() {
               @Override
            public void run() {
                   try {
                     Thread.sleep(timer);
                 } catch (InterruptedException e) {
                     // interrupted :)
                 }
                 removeComponent(component);
               }
            }).start();
        }
        return component;
    }


    /**
     * Adds a text to the left of the status bar.
     * 
     * @param msg the text to display on the left side of the status bar.
     * @return the added component
     */
    public Component addTextLeft(String msg) {
        
        return addComponent(getLabel(msg), LEFT, 0, false);
    }

    /**
     * Adds a text to the right of the status bar.
     * 
     * @param msg the text to display on the right side of the status bar.
     * @return the added component
     */
    public Component addTextRight(String msg) {
        return addComponent(getLabel(msg), RIGHT, 0, false);
    }

    /**
     * Adds a text to the left side of the status bar. If
     * <code>timer &gt; 0 </code>, the text will stay visible for
     * <code>timer</code>milliseconds. If <code>bell</code> <code>true</code>
     * the system beeps when the message is displayed.
     * <p>
     * The displayed text is displayed on a JLabel. If no font is specified
     * (using the <code>setFont</code> method, the default font is used.
     * 
     * @param msg the text to display on the left side of the status bar
     * @param timer if positive, the amount of milliseconds the message will
     *            stay visible
     * @param bell if true the system beeps when the message is displayed.
     * @return the added component
     */
    public Component addTextLeft(String msg, final int timer, boolean bell) {
        return addComponent(getLabel(msg), LEFT, timer, bell);
     }

    /**
     * Adds a text to the right side of the status bar. If
     * <code>timer &gt; 0 </code>, the text will stay visible for
     * <code>timer</code>milliseconds. If <code>bell</code> <code>true</code>
     * the system beeps when the message is displayed.
     * <p>
     * The displayed text is displayed on a JLabel. If no font is specified
     * (using the <code>setFont</code> method, the default font is used.
     * 
     * @param msg the text to display on the right side of the status bar
     * @param timer if positive, the amount of milliseconds the message will
     *            stay visible
     * @param bell if true the system beeps when the message is displayed.
     * @return the added component
     */
    public Component addTextRight(String msg, final int timer, boolean bell) {
        return addComponent(getLabel(msg), RIGHT, timer, bell);
    }
    
    /**
     * Sets the font used by the message bar when displaying text messages. If
     * the specified font is <code>null</code> the default look-and-feel font is
     * used.
     * 
     * @param font the <code>Font</code> used to display text messages.
     */
    public void setMessageFont(Font font) {
        msgFont = font;
    }

    /**
     * Removes the specified component from the status bar.
     * 
     * @param component the component that shall be removed.
     */
    public void removeComponent(Component component) {
        leftPane.remove(component);
        rightPane.remove(component);
        refreshStatusBar(false);
    }

    /**
     * Removes the specified text message from the status bar. The status bar is
     * searched left - to - right, and the first match is removed. If no match,
     * no action is taken.
     * 
     * @param msg the text to remove.
     */
    public void removeMessage(String msg) {
        JLabel l = identifyLabel(msg);
        if (l != null) {
            removeComponent(l);
        }
    }
    
    /**
     * Refreshes the status bar
     * 
     * @param bell flag signaling if the bell should be "belled"
     */
    private void refreshStatusBar(final boolean bell) {
        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
                invalidate();
                validate();
                repaint();
                if (bell) {
                    Toolkit.getDefaultToolkit().beep();
                }
            }
        });
        
    }
    
    /////////////////////// Private helper methods ///////////////////////

    /**
     * Returns a JLabel with the specifyed text. If a message font is specified,
     * the message font is used, otherwise the look-and-feel font is used.
     * 
     * @param msg the text that is put on the JLabel
     * @return a new label
     */
    private JLabel getLabel(String msg) {
        JLabel label = new JLabel(msg);
        if (msgFont != null) {
            label.setFont(msgFont);
        }
        return label;
    }

    /**
     * Finds and returns the first occurence of a contained label with the
     * specified text. The left panel is searched before the right panel.
     * 
     * @param msg the text to search for
     * @return a JLabel with the exact same text as msg, or null if no such
     *         label was found.
     */
    private JLabel identifyLabel(String msg) {
        int leftCount = leftPane.getComponentCount();
        for (int i = 0; i < leftCount; i++) {
            Component c = leftPane.getComponent(i);
            if (c instanceof JLabel) {
                JLabel label = (JLabel) c;
                if (label.getText().equals(msg)) {
                    return label;
                }
                
            }
        }
        
        int rightCount = rightPane.getComponentCount();
        for (int i = 0; i < rightCount; i++) {
            Component c = rightPane.getComponent(i);
            if (c instanceof JLabel) {
                JLabel label = (JLabel) c;
                if (label.getText().equals(msg)) {
                    return label;
                }
                
            }
        }
        
        return null;
    }
    
}
