/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.component;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Vector;

import javax.swing.DefaultComboBoxModel;

import no.g9.support.G9Enumerator;
import no.g9.support.TypeTool;

/**
 * The combo box model class, keeping track of the values in the combo box.
 */
@SuppressWarnings("rawtypes")
public class G9ComboBoxModel extends DefaultComboBoxModel implements Comparator<Object> {

    /** The is sorted property */
    private boolean isSorted;

    /**
     * The list of objects kept by this model. Unfortunatly the super's object
     * list is default protected (whithout a way of getting the contents),
     * meaning I have no access from this package (sigh).
     */
    private List<Object> objects;

    /** Reference to the combo box */
    private G9ComboBox comboBox;

    /**
     * Constructs an empty G9ComboBoxModel object.
     */
    public G9ComboBoxModel() {
        super();
        objects = new LinkedList<Object>();
    }

    /**
     * Constructs a G9ComboBoxModel object initialized with an array of
     * objects.
     *
     * @param items an array of Object objects
     */
    @SuppressWarnings("unchecked")
    public G9ComboBoxModel(Object[] items) {
        super(items);
        objects = Arrays.asList(items);
    }

    /**
     * Constructs a G9ComboBoxModel object initialized with a vector.
     *
     * @param v a Vector object
     */
    @SuppressWarnings("unchecked")
    public G9ComboBoxModel(Vector<?> v) {
        super(v);
        objects = (Vector<Object>) v;
    }

    /**
     * {@inheritDoc}
     * <p>
     * If this is a sorted combo box, and the added object is not null, and the
     * added object is first in the sorted sequence, the added item is selected.
     * If the specified object equals an object already present in the list, it
     * is put just before the element in the list.
     */
    @SuppressWarnings("unchecked")
    @Override
    public void addElement(Object anObject) {

        if (!isSorted()) {
            super.addElement(anObject);
            objects.add(anObject);
        } else {
            super.removeAllElements();
            int pos = 0;
            // find position to insert element at.
            while (pos < objects.size()) {
                Object elem = objects.get(pos);
                if (compare(elem, anObject) >= 0) {
                    break;
                }
                pos++;
            }
            objects.add(pos, anObject);
            for (int i = 0; i < objects.size(); i++) {
                super.addElement(objects.get(i));
            }
        }
    }

    /**
     * Adds a collection of elements to the combo box model.
     *
     * @param items the collection of items to add.
     */
    @SuppressWarnings("unchecked")
    public void addElements(Collection<? extends Object> items) {
        if (!isSorted()) {
            Iterator<? extends Object> it = items.iterator();
            while (it.hasNext()) {
                Object elem = it.next();
                super.addElement(elem);
                objects.add(elem);
            }
        } else {
            super.removeAllElements();
            objects.addAll(items);
            Collections.sort(objects, this);
            Iterator<Object> it = objects.iterator();
            while (it.hasNext()) {
                super.addElement(it.next());
            }
        }
    }

    /**
     * Returns the isSorted property
     *
     * @return <code>true</code> if elements in this combo are sorted.
     */
    public boolean isSorted() {
        return isSorted;
    }

    /**
     * Sets the isSorted property. If set to <code>true</code> the elements in
     * this combo is sorted.
     *
     * @param isSorted the isSorted property
     * @see #addElement(Object)
     */
    public void setSorted(boolean isSorted) {
        this.isSorted = isSorted;
    }

    /**
     * Compares its two arguments for order. Returns a negative integer, zero,
     * or a positive integer as the first argument is less than, equal to, or
     * greater than the second.
     * <p>
     * <em>Note</em> that <code>null</code> is considered the smalles possible
     * value. The comparison is done by
     * {@link TypeTool#compareTo(Object, Object)}, ensuring a locale dependent
     * comparison of Strings. Enumerations are compared using the title.
     *
     * @param o1 the first object to be compared
     * @param o2 the second object to be compared
     * @return a negative integer, zero, or a positive integer as the first
     *         argument is less than, equal to, or greater than the second.
     */
    @Override
    public int compare(Object o1, Object o2) {
        if (o1 == null && o2 == null) {
            return 0;
        } else if (o1 == null && o2 != null) {
            return -1;
        } else if (o1 != null && o2 == null) {
            return 1;
        } else if (o1 instanceof G9Enumerator && o2 instanceof G9Enumerator) {
                String o1S = comboBox.format(o1);
                String o2S = comboBox.format(o2);
                return TypeTool.compareTo(o1S, o2S);
        } else if (o1 instanceof Comparable<?> && o1.getClass() == o2.getClass()) {
            return TypeTool.compareTo(o1, o2);
        } else {
            o1 = comboBox.getDisplayValue(o1);
            o2 = comboBox.getDisplayValue(o2);
            if (o1 == null && o2 == null) {
                return 0;
            } else if (o1 == null && o2 != null) {
                return -1;
            } else if (o1 != null && o2 == null) {
                return 1;
            } else if (o1 instanceof Comparable<?>) {
                return TypeTool.compareTo(o1, o2);
            } else {
                return 0;
            }
        }
    }

    /**
     * Sets the combo box that uses this model.
     *
     * @param comboBox the combo box using this model.
     */
    void setComboBox(G9ComboBox comboBox) {
        this.comboBox = comboBox;
    }

    @Override
    public void removeAllElements() {
        if (objects.size() > 0) {
            objects.clear();
        }
        super.removeAllElements();
    }

    @Override
    public void removeElement(Object anObject) {
        objects.remove(anObject);
        super.removeElement(anObject);
    }

    @Override
    public void removeElementAt(int index) {
        objects.remove(index);
        super.removeElementAt(index);
    }

}
