/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.component;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.beans.PropertyVetoException;
import java.util.Stack;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ActionMap;
import javax.swing.DefaultDesktopManager;
import javax.swing.ImageIcon;
import javax.swing.InputMap;
import javax.swing.JComponent;
import javax.swing.JDesktopPane;
import javax.swing.JFrame;
import javax.swing.JInternalFrame;
import javax.swing.JScrollPane;
import javax.swing.JViewport;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;

import no.g9.client.component.menu.G9MenuBar;
import no.g9.client.support.DialogBlocker;
import no.g9.client.support.G9DialogFrame;

/**
 * g9 desktop pane extends the JDesktopPane, providing actions and methods
 * for tiling and cascading internal frames.
 */
public class G9DesktopPane extends JDesktopPane {

    /** The stack of frames, used when cycling forwards or backwards */
    private Stack<JInternalFrame> frameStack = new Stack<JInternalFrame>();

    /** Offset used to position frames */
    private static final int FRAME_OFFSET = 20;

    /** The name of the cascade action, used as key in the action map. */
    public static final String cascade = "Cascade";

    /** The name of the tile action, used as key in the actino map. */
    public static final String tile = "Tile";

    /**
     * The menu bar of this desktop pane. Menu bars contained in dialogs
     * added to this desktop pane are merged with the application menu bar.
     */
    protected G9MenuBar desktopMenuBar = null;

    /**
     * The tool bar of this desktop pane. Tool bars contained in dialogs added
     * to this desktop pane are merged with the application tool bar.
     */
    protected G9ToolBar desktopToolBar = null;

    /**
     * The desktop manager
     */
    private G9DesktopManager desktopManager;

    /**
     * Creates a new <code>JDesktopPane</code>.
     */
    public G9DesktopPane() {
        super();
        desktopManager = new G9DesktopManager(this);
        setDesktopManager(desktopManager);
        setDragMode(JDesktopPane.OUTLINE_DRAG_MODE);

        // Initializes cascade and tile action.
        ActionMap map = getActionMap();

        Action cascadeAction = new CascadeAction(cascade, null, null, null);
        cascadeAction.setEnabled(false);
        map.put(cascadeAction.getValue(Action.NAME), cascadeAction);

        Action tileAction = new TileAction(tile, null, null, null);
        tileAction.setEnabled(false);
        map.put(tileAction.getValue(Action.NAME), tileAction);

        // Register ctrl+W as a short-cut to close an internal frame.
        // The key for the close action is hard-wiered in
        // BasicDesktopPaneUI,
        // and I can't find a constant for it!
        Object closeActionKey = "close";
        InputMap inputMapWhenAncestor = getInputMap(WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
        KeyStroke controlAndW = KeyStroke.getKeyStroke(KeyEvent.VK_W,
                InputEvent.CTRL_DOWN_MASK);
        inputMapWhenAncestor.put(controlAndW, closeActionKey);

        // FIX ctrl+f6 bug, see comment where the action "g9NextAction" is declared.
        String nextKey = "g9NavigateNext";
        KeyStroke ctrlF6 = KeyStroke.getKeyStroke(KeyEvent.VK_F6, InputEvent.CTRL_DOWN_MASK);
        getActionMap().put(nextKey, g9NextAction);
        inputMapWhenAncestor.put(ctrlF6, nextKey);

        // Add ctrl+shift+f6 (navigate previous)
        String previousKey = "g9NavigatePrevious";
        int ctrlShiftMask = InputEvent.CTRL_DOWN_MASK | InputEvent.SHIFT_DOWN_MASK;
        KeyStroke ctrlShiftF6 = KeyStroke.getKeyStroke(KeyEvent.VK_F6, ctrlShiftMask);
        getActionMap().put(previousKey, g9PreviousAction);
        inputMapWhenAncestor.put(ctrlShiftF6, previousKey);


    }

    /**
     * Sets the desktop's menu bar
     *
     * @param menuBar the menu bar
     */
    public void setMenuBar(G9MenuBar menuBar) {
        desktopMenuBar = menuBar;
    }

    /**
     * Gets the desktop menu bar
     *
     * @return the desktop menu bar
     */
    public G9MenuBar getMenuBar() {
        return desktopMenuBar;
    }

    /**
     * Sets the desktop tool bar
     *
     * @param toolBar the desktop tool bar
     */
    public void setToolBar(G9ToolBar toolBar) {
        desktopToolBar = toolBar;
    }

    /**
     * Gets the desktop tool bar
     *
     * @return the desktop tool bar
     */
    public G9ToolBar getToolBar() {
        return desktopToolBar;
    }

    /**
     * Invoked when a frame is shown by an action
     *
     * @param shownFrame the shown frame
     */
    public void frameShown(JInternalFrame shownFrame) {
        frameStack.remove(shownFrame);
        frameStack.push(shownFrame);
        desktopManager.activateFrame(shownFrame);
        desktopManager.updateCascadeAndTileAction();
    }

    /**
     * Invoked when a frame is hidden by an action.
     *
     * @param hiddenFrame the hidden frame
     */
    public void frameHidden(JInternalFrame hiddenFrame) {
        frameStack.remove(hiddenFrame);
        desktopManager.updateCascadeAndTileAction();
        desktopManager.deactivateFrame(hiddenFrame);
        selectNext(hiddenFrame);
    }

    /**
     * Invoked when a frame is added to the desktop
     *
     * @param addedFrame the added frame
     */
    public void frameAdded(JInternalFrame addedFrame) {
        frameStack.remove(addedFrame);
        frameStack.push(addedFrame);
    }


    @Override
    public void setBounds(int x, int y, int w, int h) {
        super.setBounds(x, y, w, h);
        checkDesktopSize();
       }

    @Override
    public void remove(Component c) {
        super.remove(c);
        checkDesktopSize();
    }

    /**
     * Sets all component size properties ( maximum, minimum, preferred) to the
     * given dimension.
     * @param d the dimension
     */
    public void setAllSize(Dimension d) {
        setMinimumSize(d);
        setMaximumSize(d);
        setPreferredSize(d);
    }

    /**
     * Sets all component size properties ( maximum, minimum, preferred) to the
     * given width and height.
     * @param width the width
     * @param height the height
     */
    public void setAllSize(int width, int height) {
        setAllSize(new Dimension(width, height));
    }

    /**
     * Cascades all internal frames.
     */
    public void cascade() {
        int xPos = 0;
        int yPos = 0;

        JInternalFrame allFrames[] = getAllFrames();

        int frameHeight = getBounds().height
                - ((allFrames.length - 1) * FRAME_OFFSET);
        int frameWidth = getBounds().width
                - ((allFrames.length - 1) * FRAME_OFFSET);

        for (int i = allFrames.length - 1; i >= 0; i--) {
            if (allFrames[i].isIcon()) {
                getDesktopManager().deiconifyFrame(allFrames[i]);
            }
            allFrames[i].setBounds(xPos, yPos, frameWidth, frameHeight);
            xPos += FRAME_OFFSET;
            yPos += FRAME_OFFSET;
        }
    }

    /**
     * Tiles all internal frames horizontally.
     */
    public void tileHorizontal() {
        JInternalFrame allFrames[] = getAllFrames();

        int frameHeight = getBounds().height / allFrames.length;
        int frameWidth = getBounds().width;
        int xPos = 0;
        int yPos = 0;

        for (int i = 0; i < allFrames.length; i++) {
            allFrames[i].setBounds(xPos, yPos, frameWidth, frameHeight);
            yPos += frameHeight;
        }
    }

    /**
     * Tiles all internal frames vertically.
     */
    public void tileVertical() {
        JInternalFrame allFrames[] = getAllFrames();

        int frameHeight = getBounds().height;
        int frameWidth = getBounds().width / allFrames.length;
        int xPos = 0;
        int yPos = 0;

        for (int i = 0; i < allFrames.length; i++) {
            allFrames[i].setBounds(xPos, yPos, frameWidth, frameHeight);
            xPos += frameWidth;
        }

    }

    /**
     * Tiles all internal frames
     */
    public void tile() {
        JInternalFrame allFrames[] = getAllFrames();
        int rows = 1;
        int cols = 1;

        // calculate number of row and columns - minimize rows
        while (rows * cols < allFrames.length) {
            if (cols <= rows) {
                cols++;
            } else {
                rows++;
            }
        }

        // calculate number or rows in each column
        int[] rowsInCol = new int[cols];
        int firstColNumber = allFrames.length % rows;
        if (firstColNumber == 0) {
            firstColNumber = rows;
        }
        rowsInCol[0] = firstColNumber;

        for (int i = 1; i < rowsInCol.length; i++) {
            rowsInCol[i] = rows;
        }

        // position frames in columns and rows.
        int frameCount = allFrames.length - 1;
        int frameWidth = getBounds().width / cols;
        int frameHeigth = 0;
        int xPos = 0;
        int yPos = 0;
        for (int colNum = 0; colNum < cols; colNum++) {
            xPos = frameWidth * colNum;
            for (int row = 0; row < rowsInCol[colNum]; row++) {
                frameHeigth = getBounds().height / rowsInCol[colNum];
                yPos = row * frameHeigth;
                if (allFrames[frameCount].isIcon()) {
                    getDesktopManager().deiconifyFrame(
                            allFrames[frameCount]);
                }
                allFrames[frameCount--].setBounds(xPos, yPos, frameWidth,
                        frameHeigth);
            }
        }

    }

    private void checkDesktopSize() {
        if (getParent() != null && isVisible())
            desktopManager.resizeDesktop();
    }

    /**
     * Some management is needed in order to merge menus...
     */
    private class G9DesktopManager extends DefaultDesktopManager {

        /** The desktop pane.*/
        private G9DesktopPane desktop;

        /**
         * Activates the argument frame, updates the desktop menu and toolbar,
         * and updates the tile and cascade actions.<br>
         *
         * @param f the frame to activate
         */
        @Override
        public void activateFrame(JInternalFrame f) {
            super.activateFrame(f);

            if (f instanceof G9DialogFrame)
                updateDesktopMenuAndToolBar((G9DialogFrame) f);
            resizeDesktop();
            updateCascadeAndTileAction();
        }

        /**
         * Deactivates the argument frame and updates the desktop menu and
         * toolbar. <br>
         * If another frame is selected then the activation of the newly
         * selected frame (See {@link #activateFrame(JInternalFrame)}) already
         * has updated the desktop menu and toolbar. Therefore nothing has to be
         * done regarding the menu and toolbar. <br>
         * If no other frame is active then only application menu and toolbar
         * elements should be visible. Elements on the menu and toolbar
         * belonging to the deactivated frame is therefore removed by calling
         * {@link #updateDesktopMenuAndToolBar(G9DialogFrame)} with
         * <code>null</code> argument.<br>
         * The eventually updating of the desktop menu and toolbar is done after
         * all other pending AWT events have been processed. This ensures that
         * the eventually frame to become activated really is activated and
         * hence that the test for an active frame works correctly.
         *
         * @param f the frame to deactivate
         * @see javax.swing.DefaultDesktopManager#deactivateFrame(javax.swing.JInternalFrame)
         */
        @Override
        public void deactivateFrame(final JInternalFrame f) {
			super.deactivateFrame(f);

			SwingUtilities.invokeLater(new Runnable() {
				@Override
                public void run() {
					if (getSelectedFrame() == null) {
					    updateDesktopMenuAndToolBar(null);
                        if (f instanceof G9DialogFrame) {
                            G9DialogFrame gdf = (G9DialogFrame) f;
                            JFrame appWindow = gdf.getController()
                                    .getApplication()
                                    .getApplicationWindow();
                            appWindow.getContentPane()
                                    .requestFocusInWindow();
                        }
                    }
				}
			});
		}

		@Override
        public void closeFrame(JInternalFrame f) {
            super.closeFrame(f);
            frameStack.remove(f);
            updateCascadeAndTileAction();
            selectNext(f);
        }

        /**
         * Updates the desktop menu- and toolbar.
         *
         * @param gdm the menu- and toolbar of <code>gdm<code> are respectivelly
		 *        merged with this desktop menu- and toolbar.
         */
		private void updateDesktopMenuAndToolBar(final G9DialogFrame gdm) {

			if (desktopMenuBar != null) {
				desktopMenuBar.mergeMenues(gdm);
			}
			if (desktopToolBar != null)
				desktopToolBar.mergeToolBars(gdm);

            toggleEditMenu(gdm);
		}

        /**
		 * Updates the enabled state of the cascade and tile actions.
		 */
        protected void updateCascadeAndTileAction() {
            getActionMap().get(cascade).setEnabled(
                    getAllFrames().length != 0);
            getActionMap().get(tile)
                    .setEnabled(getAllFrames().length != 0);
        }

        /**
         * Constructor.
         *
         * @param desktop
         *            the desktop
         */
        public G9DesktopManager(G9DesktopPane desktop) {
            this.desktop = desktop;
        }

        @Override
        public void endResizingFrame(JComponent f) {
            super.endResizingFrame(f);
            resizeDesktop();
        }

        @Override
        public void endDraggingFrame(JComponent f) {
            super.endDraggingFrame(f);
            resizeDesktop();
        }

        private Insets getScrollPaneInsets() {
            JScrollPane scrollPane = getScrollPane();
            if (scrollPane == null) {
                return new Insets(0, 0, 0, 0);
            }
            return getScrollPane().getBorder().getBorderInsets(scrollPane);
        }

        private JScrollPane getScrollPane() {
            if (desktop.getParent() instanceof JViewport) {
                JViewport viewPort = (JViewport) desktop.getParent();
                if (viewPort.getParent() instanceof JScrollPane)
                    return (JScrollPane) viewPort.getParent();
            }
            return null;
        }

    /**
         * Resize the desktop due to internal frame size changes.
         */
        protected void resizeDesktop() {
            int x = 0;
            int y = 0;
            JScrollPane scrollPane = getScrollPane();
            Insets scrollInsets = getScrollPaneInsets();

            if (scrollPane != null) {
                JInternalFrame allFrames[] = desktop.getAllFrames();
                for (int i = 0; i < allFrames.length; i++) {
                    if (allFrames[i].getX() + allFrames[i].getWidth() > x) {
                        x = allFrames[i].getX() + allFrames[i].getWidth();
                    }
                    if (allFrames[i].getY() + allFrames[i].getHeight() > y) {
                        y = allFrames[i].getY() + allFrames[i].getHeight();
                    }
                }
                Dimension d = scrollPane.getVisibleRect().getSize();
                if (scrollPane.getBorder() != null) {
                    d.setSize(
                            d.getWidth() - scrollInsets.left - scrollInsets.right,
                            d.getHeight() - scrollInsets.top - scrollInsets.bottom);
                }

                if (x <= d.getWidth())
                    x = ((int) d.getWidth()) - 20;
                if (y <= d.getHeight())
                    y = ((int) d.getHeight()) - 20;
                desktop.setAllSize(x, y);
                scrollPane.invalidate();
                scrollPane.validate();
            }
        }

    }

    /**
     *
     * Internal use.
     *
     * Toggles the edit menu actions (cut, copy, paste, select all).
     * If the dialog is blocked, edit actions are disabled, otherwise enabled.
     *
     * @param dialog the currently active dialog
     */
    public synchronized void toggleEditMenu(final G9DialogFrame dialog) {
        boolean enable = !DialogBlocker.isBlocked(dialog);
        G9MenuBar menuBar = getMenuBar();
        if (menuBar == null) {
            return;
        }
        menuBar.toggleEditMenu(enable);
    }

    /**
     * Inner class representing the cascade action. If no description is
     * provided, a default short description text is provided. If no mnemonic
     * key is specified, a default mnemonic key is provided.
     *
     * <p>
     * Normally it should be sufficient to use the action map to get an instance
     * of this action (since you normally only want one instance in your app).
     */
    public class CascadeAction extends AbstractAction {

        /** The default short description text */
        public static final String defaultDesc = "Cascades all dialog windows";

        /** The default menmonic key - 'C' */
        public final Integer defaultMnemonic = new Integer(KeyEvent.VK_C);

        /**
         * Constructor - creates a new instance of CascadeAction.
         *
         * @param text the text displayed on buttons etc.
         * @param icon the icon displayed on buttons etc.
         * @param desc the short description of this action - used for tooltip
         *            functions.
         * @param mnemonic the mnemonic key for this action.
         */
        public CascadeAction(String text, ImageIcon icon, String desc, Integer mnemonic) {
            super(text, icon);
            if (desc == null) {
                desc = defaultDesc;
            }
            if (mnemonic == null) {
                mnemonic = defaultMnemonic;
            }
            putValue(SHORT_DESCRIPTION, desc);
            putValue(MNEMONIC_KEY, mnemonic);
        }

        @Override
        public void actionPerformed(ActionEvent e) {
            cascade();
        }

    }

    /**
     * Inner class representing the tile action. If no description is specified,
     * a default, short description text is provided. If no mnemonic key is
     * specified, a default mnemonic key is provided.
     * <p> Normally it should be sufficient to use the action map to get an
     * instance of this action (since you normally only want one instance in
     * your app).
     */
    public class TileAction extends AbstractAction {

        /** The default short description text. */
        public static final String defaultDesc = "Tiles all dialog windows";

        /** The default mnemonic key ('V'). */
        public final Integer defaultMnemonic = new Integer(KeyEvent.VK_V);

        /**
         * Constructor - creates a new instance of TileAction.
         *
         * @param text the text displayed on buttons etc.
         * @param icon the icon displayed on buttons etc.
         * @param desc the short description of this action - used for tooltip
         *            functions.
         * @param mnemonic the mnemonic key for this action.
         */
        public TileAction(String text, ImageIcon icon, String desc,
                Integer mnemonic) {
            super(text, icon);

            if (desc == null) {
                desc = defaultDesc;
            }

            if (mnemonic == null) {
                mnemonic = defaultMnemonic;
            }

            putValue(SHORT_DESCRIPTION, desc);
            putValue(MNEMONIC_KEY, mnemonic);
        }

        @Override
        public void actionPerformed(ActionEvent e) {
            tile();
        }

    }


    /**
     * Selects the next frame in the z-order.
     *
     * @param currentFrame the frame that is selected now.
     */
    private void selectNext(JInternalFrame currentFrame) {
        JInternalFrame nextFrame = null;

        for (int i = 0; i < getComponentCount(); i++) {
            if (getComponent(i) instanceof JInternalFrame) {
                nextFrame = (JInternalFrame) getComponent(i);
                if (nextFrame != currentFrame && nextFrame.isVisible()) {
                    try {
                        nextFrame.setSelected(true);
                        break;
                    } catch (PropertyVetoException e) {
                        // Could not select .... ok.
                    }
                }
            }
        }
    }

    /**
     * The action that selects the next open internal frame.
     *
     * <p>FIX of CTRL+F6 bug, see
     * http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=4773378
     * This fix should be removed if running on Java 1.5 or higer.
     * If the fix is removed, remeber to also remove it from the
     * constructor!
     */
    private Action g9NextAction = new AbstractAction() {

        @Override
        public void actionPerformed(ActionEvent e) {
            // If no components are present or no component is visible - return!
            boolean visible = false;
            for (int i = 1; i < getComponentCount() && !visible; i++) {
                visible = visible || getComponent(i).isVisible();
            }
            if (!visible) {
                return;
            }

            // Current focus owner has index zero.
            Component cur = getComponent(0);
            remove(cur);
            add(cur);
            // Select frame with lowest possible index.
            for (int i = 0; i < getComponentCount(); i++) {
                Component next = getComponent(i);
                if (next.isVisible()) {
                    JInternalFrame frame = null;
                    if (next instanceof JInternalFrame) {
                        frame = (JInternalFrame) next;

                    } else if (next instanceof JInternalFrame.JDesktopIcon) {
                        // Previous line might be an error in future
                        // releases of Java but I can't figure out how to
                        // select the icon without it.
                        frame = ((JInternalFrame.JDesktopIcon) next).getInternalFrame();

                    }
                    if (frame != null) {
                        try {
                            frame.setSelected(true);
                            if (frame.isIcon()) {
                                frame.getDesktopIcon().requestFocus();
                            }
                            break;
                        } catch (PropertyVetoException pve) {
                            // Could not select - try next
                        }
                    } // frame != null
                } // next.isVisbible()
            } // for (...)
        } // public void actionPerformed(...)
    }; // private Action gen...

    /**
     * The action that selects the previous, open internal frame
     */
    private Action g9PreviousAction = new AbstractAction() {
        @Override
        public void actionPerformed(ActionEvent e) {
            // If no components are present or no component is visible - return!
            boolean visible = false;
            for (int i = 0; i < getComponentCount() && !visible; i++) {
                visible = visible || getComponent(i).isVisible();
            }
            if (!visible) {
                return;
            }

            // Select frame with highest possible index.
            for (int i = getComponentCount() - 1; i >= 0; i--) {
                Component previous = getComponent(i);
                if (previous.isVisible()) {
                    JInternalFrame frame = null;
                    if (previous instanceof JInternalFrame) {
                        frame = (JInternalFrame) previous;
                    } else if (previous instanceof JInternalFrame.JDesktopIcon) {
                        // Previous line might be an error in future
                        // releases of Java but I can't figure out how to
                        // select the icon without it.
                        frame = ((JInternalFrame.JDesktopIcon) previous).getInternalFrame();
                    }
                    if (frame != null) {
                        try {
                            Component[] foo = getComponents();
                            frame.setSelected(true);
                            if (frame.isIcon()) {
                                for (int j = 0; j < i; j++) {
                                    Component tmp = foo[j];
                                    remove(tmp);
                                    add(tmp);
                                }
                                frame.getDesktopIcon().requestFocus();
                            }
                            break;
                        } catch (PropertyVetoException pve) {
                            // Could not select - try next.
                       }
                    } // frame != null
                } // previous.isVisible()
            } // for (...)
        } // public void actionPerformed(...)
    }; // private Action gen...

}
