/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.component;

import java.awt.Component;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.text.ParseException;

import javax.swing.JPasswordField;
import javax.swing.JRootPane;
import javax.swing.SwingConstants;
import javax.swing.event.EventListenerList;
import javax.swing.text.Document;

import no.g9.client.event.G9ValueChangedEvent;
import no.g9.client.event.G9ValueChangedListener;
import no.g9.client.event.G9ValueState;
import no.g9.client.support.EnumeratorDocument;
import no.g9.client.support.G9Document;
import no.g9.client.support.G9FieldValue;
import no.g9.exception.G9ClientFrameworkException;
import no.g9.message.*;
import no.g9.support.TypeTool;

/**
 * g9 Text Field is basicly a fix on a bug in Java 1.4, which does not set
 * the background color when the text field is disabled. In addition, the g9
 * Text Field holds a state
 */
public class G9PasswordField extends JPasswordField implements
        G9ValueState, G9FieldValue {

    /** The old value, used to trigger value changed */
    private Object oldVal;

    /** The initial value, used to determine if the component is changed. */
    private Object initialVal;

    /** Flag indicating the editable state of this component. */
    private boolean editable = true;

    /** Flag indicating the enabled state of this component */
    private boolean enabled = true;

    /** Flag indicating if bug fix code should run. */
    private boolean bugFix;

    /**
     * The output justification of this component
     */
    private int outputJustification = SwingConstants.LEFT;

    /** The input justifiaction of this component */
    private int inputJustification = SwingConstants.LEFT;

    /** Flag indicating if select all on focus is on */
    private boolean selectAllOnFocus = false;

    /** List of value changed listeners */
    protected EventListenerList vcListeners = new EventListenerList();

    /**
     * Constructs a new <code>G9TextField</code>. A default model is
     * created, the initial string is <code>null</code>, and the number of
     * columns is set to 0.
     */
    public G9PasswordField() {
        this(null, null, 0);
    }

    /**
     * Constructs a new <code>G9TextField</code> initialized with the
     * specified text. A default model is created and the number of columns is
     * 0.
     *
     * @param text the text to be displayed, or <code>null</code>
     */
    public G9PasswordField(String text) {
        this(null, text, 0);
    }

    /**
     * Constructs a new empty <code>G9TextField</code> with the specified
     * number of columns. A default model is created and the initial string is
     * set to <code>null</code>.
     *
     * @param columns the number of columns to use to calculate the preferred
     *            width; if columns is set to zero, the preferred width will be
     *            whatever naturally results from the component implementation
     */
    public G9PasswordField(int columns) {
        this(null, null, columns);
    }

    /**
     * Constructs a new <code>G9TextField</code> initialized with the
     * specified text and columns. A default model is created.
     *
     * @param text the text to be displayed, or <code>null</code>
     * @param columns the number of columns to use to calculate the preferred
     *            width; if columns is set to zero, the preferred width will be
     *            whatever naturally results from the component implementation
     */
    public G9PasswordField(String text, int columns) {
        this(null, text, columns);
    }

    /**
     * Constructs a new <code>G9TextField</code> that uses the given text
     * storage model and the given number of columns. This is the constructor
     * through which the other constructors feed. If the document is
     * <code>null</code>, a default model is created.
     *
     * @param doc the text storage to use; if this is <code>null</code>, a
     *            default will be provided by calling the
     *            <code>createDefaultModel</code> method
     * @param text the initial string to display, or <code>null</code>
     * @param columns the number of columns to use to calculate the preferred
     *            width &gt;= 0; if <code>columns</code> is set to zero, the
     *            preferred width will be whatever naturally results from the
     *            component implementation
     * @exception IllegalArgumentException if <code>columns</code> &lt; 0
     */
    public G9PasswordField(Document doc, String text, int columns) {
        super(doc, text, columns);
        // sets up the focus listener for this component.
        addFocusListener(new FocusAdapter() {

            @Override public void focusGained(FocusEvent e) {
                ((G9Document) getDocument()).setInputMode(true);
                if (menuOrToolbar(e.getOppositeComponent())) {
                    return;
                }

                oldVal = getValue();
                setHorizontalAlignment(inputJustification);
                if (selectAllOnFocus) {
                    selectAll();
                }
            }

            @Override public void focusLost(FocusEvent e) {
                if (menuOrToolbar(e.getOppositeComponent())) {
                    return;
                }
                Object value = getValue();
                if ((oldVal != null ^ value != null)
                        || oldVal != null && !oldVal.equals(value)) {
                    fireG9ValueChangedEvent(
                            G9ValueChangedEvent.VALUE_CHANGED, oldVal,
                            value);
                }
                ((G9Document) getDocument()).setInputMode(false);
                setHorizontalAlignment(outputJustification);
                setCaretPosition(0);
            }

            /**
             * Check if the componet that lost focus is a menu or toolbar
             * component
             * @param opposite the component that lost focus
             * @return <code>true</code> if the component is a menu or toolbar component
             */
            private boolean menuOrToolbar(Component opposite) {
                return opposite == null
                        || opposite instanceof JRootPane
                        || (opposite instanceof G9Button && ((G9Button) opposite)
                                .isInToolbar());

            }


        });

        super.setEditable(true);
        super.setEnabled(true);
        editable = true;
        enabled = true;
        bugFix = TypeTool.isVmVersion1_4();
    }

    /**
     * Forces the currently displayed text to be considered the old text.
     */
    @Override
    public void resetState() {
        oldVal = getValue();
        initialVal = oldVal;
    }

    @Override
    public void setEnabled(boolean enable) {
        super.setEnabled(enable);
        if (!bugFix) {
            return;
        }
        enabled = enable;
        if (!enable || editable) {
            super.setEditable(enable);
        }

    }

    @Override
    public void setEditable(boolean editable) {
        this.editable = editable;
        super.setEditable(editable);
        setEnabled(enabled);
    }

    @Override
    public void addValueChangedListener(G9ValueChangedListener listener) {
        vcListeners.add(G9ValueChangedListener.class, listener);
    }

    /**
     * Fires a g9 value changed event.
     *
     * @param id the type of event
     * @param oldValue the old value
     * @param newValue the new value
     */
    protected void fireG9ValueChangedEvent(int id, Object oldValue,
            Object newValue) {
        Object[] listeners = vcListeners.getListenerList();
        G9ValueChangedEvent e = null;

        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (e == null) {
                e = new G9ValueChangedEvent(this, id, oldValue,
                        newValue);
            }
            if (listeners[i] == G9ValueChangedListener.class) {
                switch (e.getID()) {
                    case G9ValueChangedEvent.VALUE_CHANGED:
                        ((G9ValueChangedListener) listeners[i + 1]).valueChanged(e);
                        break;
                    default:
                        break;
                }
            }
        }
    }

    @Override
    public boolean isChanged() {
        if (initialVal == null) {
            return getValue() != null;
        }
        if (getValue() == null) {
            return true;
        }
        // neither is null.
        return !getValue().equals(initialVal);
    }

    @Override
    public void display(Object value) {
        Document doc = getDocument();
        if (doc instanceof EnumeratorDocument && value instanceof String) {
            value = ((EnumeratorDocument) doc).toValue(value);
        }
        setValue(value);
    }

    @Override
    public void setText(String s) {
        Document doc = getDocument();
        if (doc instanceof EnumeratorDocument) {
            setValue(((EnumeratorDocument) doc).toValue(s));
        } else {
            super.setText(s);
        }
    }

    @Override
    public Object getInitialValue() {
        return initialVal;
    }

    @Override
    public void setInitialValue(Object value) {
        Document doc = getDocument();
        if (doc instanceof EnumeratorDocument && (value instanceof String || value instanceof Integer)) {
            value = ((EnumeratorDocument) doc).toValue(value);

        }
        initialVal = value;
        oldVal = initialVal;
    }

    /**
     * Sets the input justification of this component
     *
     * @param justification the input justification
     */
    public void setInputJustification(int justification) {
        inputJustification = justification;
    }

    /**
     * Sets the output justifiaction of this component
     *
     * @param justification the output justification
     */
    public void setOutputJustification(int justification) {
        outputJustification = justification;
    }

    /**
     * Set the documents new value. The document is responsible for formatting
     * the value, and inserting it.
     *
     * @param value (missing javadoc)
     */
    @Override
    public void setValue(Object value) {
        Document doc = getDocument();
        if (doc instanceof G9Document) {
            ((G9Document) doc).setValue(value);
        } else { // Fallback
            setText(value.toString());
        }
        setCaretPosition(0);
    }

    /**
     * Try creating a new value by parsing the displayed text according to the
     * output format. If unsuccessful the old value is used.
     *
     * @return the new value
     */
    @Override
    public Object getValue() {
        Object value = null;
        Document doc = getDocument();
        if (doc instanceof G9Document) {
            G9Document gDoc = (G9Document) doc;
            value = gDoc.getValue();
        } else {
            value = getPassword(); // Fallback
        }
        return value;
    }

    @Override
    public String format() {
        String str = "";
        Document doc = getDocument();
        if (doc instanceof G9Document) {
            G9Document gDoc = (G9Document) doc;
            str = gDoc.format();
        }
        return  str;
    }

    @Override
    public Object parse(String formattedString) {
        Object o = null;
        Document doc = getDocument();
        if (doc instanceof G9Document) {
            G9Document gDoc = (G9Document) doc;
            try {
                o = gDoc.parse(formattedString);
            }
            catch (ParseException e) {
                Object[] args = { getClass(), getName(), formattedString };
                Message msg = MessageSystem.getMessageFactory().getMessage(
                        CRuntimeMsg.CF_DOCUMENT_PARSEEXCEPTION, args);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9ClientFrameworkException(e, msg);
            }
        }
        return  o;
    }

    /**
     * Parses the specified object
     *
     * @param o the object to parse
     * @return the parsed object
     */
    public Object parse(Object o) {
        return o;
    }

    /**
     * If the selectAllOnFocus property is <code>true</code> the displayed text
     * is selected each time this component gains focus.
     *
     * @return Returns the selectAllOnFocus.
     */
    public boolean isSelectAllOnFocus() {
        return selectAllOnFocus;
    }

    /**
     * If the selectAllOnFocus property is <code>true</code> the displayed text
     * is selected each time this component gains focus.
     *
     * @param selectAllOnFocus The selectAllOnFocus property.
     */
    public void setSelectAllOnFocus(boolean selectAllOnFocus) {
        this.selectAllOnFocus = selectAllOnFocus;
    }

}
