/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.component;

import javax.swing.DefaultCellEditor;
import javax.swing.event.EventListenerList;

import no.g9.client.event.G9ValueChangedEvent;
import no.g9.client.event.G9ValueChangedListener;
import no.g9.support.TypeTool;

/**
 * The default g9 cell editor, used in listblocks. Checkbox and combo box
 * uses the default implementation of DefaultCellEditor. The text component version
 * differs - it uses the actual value object and not the valueobject's toString
 * when setting the editor value, and it returns the value object (not the value's
 * toString).
 * 
 * In addition, this class also takes over the responsibility to fire value changed 
 * events from text field components.
 * 
 */
public class GDefaultCellEditor extends DefaultCellEditor {

    /*
     * The purpose of this class is to return a value object and not a string 
     * object as the super implementation does.
     */

    /** List of event listeners */
    protected EventListenerList vcListeners = new EventListenerList();
    
    /**
     * Constructs a <code>GDefaultCellEditor</code> object that uses a check box.
     * This is the same as invoking the DefaultCellEditor's constructor.
     *
     * @param checkBox  a <code>G9CheckBox</code> object
     */
    public GDefaultCellEditor(G9CheckBox checkBox) {
        super(checkBox);
    }

    
    /**
     * Constructs a <code>GDefaultCellEditor</code> object that uses a
     * combo box.
     *
     * @param comboBox  a <code>G9ComboBox</code> object
     */
    public GDefaultCellEditor(G9ComboBox comboBox) {
        super(comboBox);
    }

    /**
     * Constructs a <code>GDefaultCellEditor</code> that uses a text field.
     *
     * @param textField  a <code>G9TextField</code> object
     */
    public GDefaultCellEditor(final G9TextField textField) {
        super(textField);
        delegate = new EditorDelegate() {
            
            @Override
            public Object getCellEditorValue() {
                return textField.getValue();
            }

            @Override
            public void setValue(Object value) {
                textField.setValue(value);
                super.setValue(value);
            }

            @Override
            public boolean stopCellEditing() {
                Object currentValue = textField.getValue();
                Object oldValue = textField.getInitialValue();
                if (TypeTool.differ(currentValue, oldValue)) {
                    textField.resetState();
                    textField.fireG9ValueChangedEvent(
                            G9ValueChangedEvent.VALUE_CHANGED, oldValue,
                            currentValue);
                }
                return super.stopCellEditing();
            }
            
        };
        textField.addActionListener(delegate);
        
        // Hijack vc listeners.
        //        vcListeners = textField.vcListeners;
        //        textField.vcListeners = null;
    }
    
    
    /**
     * Fires a g9 value changed event.
     * @param id the type of event
     * @param oldValue the old value
     * @param newValue the new value
     */
    protected void fireG9ValueChangedEvent(int id, Object oldValue,
            Object newValue) {
        Object[] listeners = vcListeners.getListenerList();
        G9ValueChangedEvent e = null;

        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (e == null) {
                e = new G9ValueChangedEvent(this, id, oldValue,
                        newValue);
            }
            if (listeners[i] == G9ValueChangedListener.class) {
                switch (e.getID()) {
                    case G9ValueChangedEvent.VALUE_CHANGED:
                        ((G9ValueChangedListener) listeners[i + 1])
                                .valueChanged(e);
                        break;
                    default:
                        break;
                }
            }
        }
    }
    
}
