/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.component.menu;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.JSeparator;
import javax.swing.JTextField;
import javax.swing.text.DefaultEditorKit;
import javax.swing.text.JTextComponent;

import no.g9.client.component.EnableManager;
import no.g9.message.CRuntimeMsg;

/**
 * The edit menu class.
 *
 */
public class G9EditMenu extends G9Menu {

    /** The default edit menu items */
    private static List<JComponent> defaultEditMenuItems;
    
    /** Manager used to enable and disable menues. */
    private EnableManager enableMenuManager = new EnableManager();

    /** The copy menu item */
    private static G9MenuItem copyMenuItem;
    
    /** The cut menu item */
    private static G9MenuItem cutMenuItem;
    
    /** The paste menu item */
    private static G9MenuItem pasteMenuItem;
    
    /** The select all menu item */
    private static G9MenuItem selectAllMenuItem;
    
    
    /**
     * Constructs a new <code>JMenu</code> with no text.
     */
    public G9EditMenu() {
        this("");
    }

    /**
     * Constructs a new <code>JMenu</code> whose properties are taken from the
     * <code>Action</code> supplied.
     * 
     * @param a the <code>Action</code>
     * @since 1.3
     */
    public G9EditMenu(Action a) {
        this("");
        setAction(a);
    }

    /**
     * Constructs a new <code>JMenu</code> with the supplied string as its text
     * and specified as a tear-off menu or not.
     * 
     * @param s the text for the menu label
     * @param b can the menu be torn off (not yet implemented)
     */
    public G9EditMenu(String s, boolean b) {
        this(s);
    }

    /**
     * Constructs a new <code>JMenu</code> with the supplied string as its text.
     * 
     * @param s the text for the menu label
     */
    public G9EditMenu(String s) {
        super(s, G9Menu.EDIT_MENU);
        initDefaultItems();
    }

    /**
     * Constructs the default menu items and adds them to the menu. Current
     * default menu items are: Cut, Copy, Paste and Select all.
     */
    private synchronized void initDefaultItems() {
        if (defaultEditMenuItems == null) {
            defaultEditMenuItems = new Vector<JComponent>();

            cutMenuItem = createMenuItem(
                    actionByName(DefaultEditorKit.cutAction), "Ctrl + x",
                    CRuntimeMsg.CM_TEXT_CUT);
            defaultEditMenuItems.add(cutMenuItem);

            copyMenuItem = createMenuItem(
                    actionByName(DefaultEditorKit.copyAction), "Ctrl + c",
                    CRuntimeMsg.CM_TEXT_COPY);
            defaultEditMenuItems.add(copyMenuItem);

            pasteMenuItem = createMenuItem(
                    actionByName(DefaultEditorKit.pasteAction),
                    "Ctrl + v", CRuntimeMsg.CM_TEXT_PASTE);
            defaultEditMenuItems.add(pasteMenuItem);
            defaultEditMenuItems.add(new JSeparator());

            selectAllMenuItem = createMenuItem(
                    actionByName(DefaultEditorKit.selectAllAction),
                    "Ctrl + a", CRuntimeMsg.CM_TEXT_SELECTALL);
            defaultEditMenuItems.add(selectAllMenuItem);

        }
        setDefaultItem(defaultEditMenuItems);
    }

    /**
     * The map of actions. Key is action name, value is action.
     */
    private static Map<Object, Action> actionMap;
   
    /**
     * Initializes the action map
     */
    private static synchronized void initActionMap() {
        
        JTextComponent foo = new JTextField();
        actionMap = new HashMap<Object, Action>();
        Action[] actions = foo.getActions();
        for (int i = 0; i < actions.length; i++) {
            Action a = actions[i];
            actionMap.put(a.getValue(Action.NAME), a);
        }
        
    }
    
    /**
     * Gets the action associated with the specified name
     * 
     * @param name the name of the action
     * @return the action with the specified name
     */
    private static Action actionByName(String name) {
        if (actionMap == null) {
            initActionMap();
        }
        return actionMap.get(name);
    }
    
    /**
     * Internal use. 
     * Toggles the edit menu (cut, copy, paste, select all).
     * Depending on the parameter, these are either enabled or disabled. This is
     * used as part of blocking/unblocking a dialog.
     * 
     * @param enable if <code>true</code> actions are disabled.
     */
    protected synchronized void toggleEditComponents(boolean enable) {
        
        enableMenuManager.setEnabled(cutMenuItem, enable);
        
        enableMenuManager.setEnabled(copyMenuItem, enable);
        
        enableMenuManager.setEnabled(pasteMenuItem, enable);
        
        enableMenuManager.setEnabled(selectAllMenuItem, enable);
        
     }

}
