/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.util.List;

import javax.swing.JFrame;
import javax.swing.JInternalFrame;
import javax.swing.JOptionPane;

import no.g9.client.component.G9DesktopPane;
import no.g9.client.component.G9ToolBar;
import no.g9.client.component.menu.G9MenuBar;
import no.g9.client.event.G9DataEvent;
import no.g9.domain.G9Enums;

/** Interface for application */
@SuppressWarnings("rawtypes")
public interface ApplicationMethods  {

    /** Constant for yes */
    public final int YES = JOptionPane.YES_OPTION;

    /** Constant for NO */
    public final int NO = JOptionPane.NO_OPTION;

    /** Constant for CANCEL */
    public final int CANCEL = JOptionPane.CANCEL_OPTION;

    /** Constant for if user closed window */
    public final int CLOSED_WINDOW = JOptionPane.CLOSED_OPTION;

    /**
     * Application-wide default handler for alle Runtime Exceptions.
     *
     * @param bad the caught exception
     * @return true to continue error handling, false if the error is handled
     */
    public boolean applicationExceptionHandler(Throwable bad) ;

    /**
     * Get name of running application.
     *
     * @return the application name.
     */
    public String getApplicationName();

    /**
     * Returns the JFrame MDI appclication window.
     *
     * @return The outermoust JFrame.
     */
    public JFrame getApplicationWindow();

    /**
     * Access method for the dynamic enumeration helper. It is probably better
     * to access the generated enumeration file directly in dialog code as it
     * contains more spesific methods for working with the relevant
     * enumerations.- this one is used by the runtime-system.
     *
     * @return The G9Enums object.
     */

    public G9Enums getEnums();

    /**
     * Gives a message to a status bar, which will be removed after 5 seconds.
     * If bell is flagged, an audio signal (alert/bell) will be sent to the
     * user. Note that the users audio settings might mute the signal.
     *
     * @param message The message to display to the user.
     * @param bell If true bell will be sent.
     * @deprecated use the {@link StatusBar} instead.
     */
    @Deprecated
    public void statusBarMessage(String message, boolean bell);

    /**
     * Gives a message to a status bar, which will be removed after 5 seconds.
     * The params[] option is an optional set of arguments which should be
     * substituted in the message. If bell is flagged, an audio signal
     * (alert/bell) will be sent to the user. Note that the users audio settings
     * might * mute the signal.
     *
     * @param message The message to display.
     * @param params The arguments to be substituted, as defined by
     *            java.text.MessageFormat.
     * @param bell If true bell will be sent.
     * @deprecated use the {@link StatusBar} instead.
     */
    @Deprecated
    public void statusBarMessage(String message, Object params[],
            boolean bell);

    /**
     * Gives a message to a status bar, which is removed after a given set of
     * seconds. If bell is flagged, an audio signal (alert/bell) will be sent to
     * the user. Note that the users audio settings might mute the signal.
     *
     * @param message The message to display.
     * @param bell If true bell will be sent.
     * @param seconds The amount of time in seconds the message will be visible.
     * @deprecated use the {@link StatusBar} instead.
     */
    @Deprecated
    public void statusBarMessage(String message, boolean bell, int seconds);

    /** Requests application to be moved to front and given focus */
    public void toFront();

    /**
     * Gives a message to a status bar, which is removed after a given set of
     * seconds. The params[] option is an optional set of arguments which should
     * be substituted in the message. If bell is flagged, an audio signal
     * (alert/bell) will be sent to the user. Note that the users audio settings
     * might mute the signal.
     *
     * @param message The message to s
     * @param params The arguments to be substituted, as defined by
     *            java.text.MessageFormat.
     * @param bell If true bell will be sent.
     * @param seconds The amount of time in seconds the message will be visible.
     * @deprecated use the {@link StatusBar} instead.
     */
    @Deprecated
    public void statusBarMessage(String message, Object params[],
            boolean bell, int seconds);

    /**
     * Gives user a message box with a ok button. This message box is modal.
     *
     * @param message The title of the displayed message box.
     * @param title The title of the message box.
     * @deprecated use the
     *             {@link no.g9.message.MessageDispatcher#dispatch}
     *             instead.
     */
    @Deprecated
    public void okMessageBox(String message, String title);

    /**
     * Gives user a message box with a yes/no button. This message box is modal.
     *
     * @param message The message to display.
     * @param standardValue The default choice of the buttons, as defined by
     *            Application.YES/Application.NO.
     * @param title The title of the displayed message box.
     * @return The value the user chosed, based on Application.YES,
     *         Application.NO or Application.CLOSED_WINDOW.
     * @deprecated use the
     *             {@link no.g9.message.MessageDispatcher#dispatch}
     *             instead.
     */
    @Deprecated
    public int yesNoMessageBox(String message, int standardValue,
            String title);

    /**
     * Gives user a message box with a yes/no/cancel button.
     *
     * @param message The message to display.
     * @param standardValue The default choice of the buttons, as defined by
     *            Application.YES/Application.CANCEL/Application.NO.
     * @param title The title of the displayed message box.
     * @return The value the user chosed, based on Application.YES,
     *         Application.NO, Application.CANCEL or Application.CLOSED_WINDOW
     *         if the user closed the window.
     * @deprecated use the
     *             {@link no.g9.message.MessageDispatcher#dispatch}
     *             instead.
     */
    @Deprecated
    public int yesNoCancelMessageBox(String message, int standardValue,
            String title);

    /**
     * Gives user a message box with a given set of choices(buttons).
     *
     * @param message The message to display.
     * @param choices The choices the user should have. This should be an array
     *            of Strings.
     * @param standardChoice The choice that is the standard choice.
     * @param title The title of the displayed message box.
     * @return The Index position of the choices that the user selected or
     *         Application.CLOSED_WINDOW if the user closed the window.
     * @deprecated use the
     *             {@link no.g9.message.MessageDispatcher#dispatch}
     *             instead.
     */
    @Deprecated
    public int multiMessageBox(String message, Object choices[],
            String standardChoice, String title);

    /**
     * Gives user a message box with a ok button. The params[] option is an
     * optional set of arguments which should be substituted in the message.
     *
     * @param message The message to display.
     * @param params Options to be replaced in the message as defined by
     *            java.text.MessageFormat.
     * @param title The title of the displayed message box.
     * @deprecated use the
     *             {@link no.g9.message.MessageDispatcher#dispatch}
     *             instead.
     */
    @Deprecated
    public void okMessageBox(String message, Object params[], String title);

    /**
     * Gives user a message box with a yes/no button. The params[] option is an
     * optional set of arguments which should be substituted in the message.
     *
     * @param message The message to display.
     * @param params Options to be replaced in the message as defined by
     *            java.text.MessageFormat.
     * @param standardValue The standard value to display, Application.YES or
     *            Application.NO.
     * @param title The title of the displayed message box.
     * @return The value the user chosed, based on Application.YES,
     *         Application.NO or Application.CLOSED_WINDOW if the user closed
     *         the window.
     * @deprecated use the
     *             {@link no.g9.message.MessageDispatcher#dispatch}
     *             instead.
     */
    @Deprecated
    public int yesNoMessageBox(String message, Object params[],
            int standardValue, String title);

    /**
     * Gives user a message box with a yes/no/cancel button. The params[] option
     * is an optional set of arguments which should be substituted in the
     * message.
     *
     * @param message The message to display.
     * @param params Options to be replaced in the message as defined by
     *            java.text.MessageFormat.
     * @param standardValue The standard value to display, Application.YES,
     *            Application.NO or Application.CANCEL.
     * @param title The title of the displayed message box.
     * @return The value the user chosed, based on Application.YES,
     *         Application.NO or Application.CLOSED_WINDOW if the user closed
     *         the window.
     * @deprecated use the
     *             {@link no.g9.message.MessageDispatcher#dispatch}
     *             instead.
     */
    @Deprecated
    public int yesNoCancelMessageBox(String message, Object params[],
            int standardValue, String title);

    /**
     * Gives user a message box with a given set of choices(buttons). The
     * params[] option is an optional set of arguments which should be
     * substituted in the message.
     *
     * @param message The message to display.
     * @param params Options to be replaced in the message as defined by
     *            java.text.MessageFormat.
     * @param choices Array of strings with the choices for the user.
     * @param standardChoice The String that matches one of the strings sent in
     *            the params list that is to be the default choice of the
     *            message box.
     * @param title The title of the displayed message box.
     * @return The Index position of the choices that the user selected or
     *         Application.CLOSED_WINDOW if the user closed the window.
     * @deprecated use the
     *             {@link no.g9.message.MessageDispatcher#dispatch}
     *             instead.
     */
    @Deprecated
    public int multiMessageBox(String message, Object params[],
            Object choices[], String standardChoice, String title);

    /**
     * Removes a dialog from view. Do not use this method. Use the dispose()
     * method defined in the controller class.
     *
     * @param g The dialog controller to remove.
     */
//    public void removeWindow(G9DialogController g);
    /**
     * Checks if there exist a dialog with the given name.
     *
     * @param name As defined in g9 (dmdialog).
     * @return <code>true</code> if the dialog exists.
     */
    public boolean isWindowAdded(String name);

    /**
     * Creates a dialog with the given name.
     *
     * @param name The dialog name as defined in g9.
     * @return The controller class of the newly created dialog.
     */
    public G9DialogController createDialog(String name);

    /**
     * Creates a dialog with the given name. If the second parameter is
     * <code>true</code> the window is added to the application and made
     * visible.
     * <p>
     * If the created dialog is a dialog box, and the dialog box is made visible
     * (the second parameter is <code>true</code> the method will not return
     * until the dialog is closed or made invisible.
     *
     * @param name The dialog name as defined in g9
     * @param add if <code>true</code> the dialog is added and shown.
     * @return The controller class of the newly created dialog.
     */
    public G9DialogController createDialog(String name, boolean add);

    /**
     * Searches through the added dialogs and returns that one if there exist a
     * dialog with the given name. If not it creates a new one.
     * <p>
     * If the dialog is a new, modal dialog box, this method will not return
     * until the dialog is closed or hidden.
     *
     * @param name The dialog name as defined in g9.
     * @return The controller class of the pre-existing or newly created dialog.
     */
    public G9DialogController getOldDialogOrCreateNew(String name);

    /**
     * Internal use. Adds a window to the list of application windows,
     * initializes it and shows it.
     *
     * @param dialogController (missing javadoc)
     */
    public void addWindow(G9DialogController dialogController);

    /**
     * Internal use. Adds a window to the list of application windows. Depending
     * on the second parameter the window is either initialized or not.
     *
     * @param dialogController (missing javadoc)
     * @param doInit (missing javadoc)
     */
    public void addWindow(G9DialogController dialogController, boolean doInit);

    /**
     * Searches through the added dialogs and returns that one if there exist a
     * dialog with the given name. If not found it returns null.
     *
     * @param name The dialog name as defined in g9.
     * @return The requested dialog controller.
     */
    public G9DialogController getDialog(String name);

    /**
     * Calls g9DataEvent in all open window controllers with eventObject as
     * parameter. All dialogs will receive the event before new G9DataEvents
     * are pumped to the dialogs.
     *
     * @param eventObject The data object that is to be sent. The message is
     *            user defined.
     */
    public void sendG9DataEvent(G9DataEvent eventObject);

    /**
     * Makes the window controlled by this controller modal. See doc on
     * G9DialogController.
     *
     * @param g The controller to set.
     * @param isModal True makes it modal, false makes it unmodal.
     */
    public void setModal(G9DialogController g, boolean isModal);

    /**
     * Returns the controller window that is modal if any.
     *
     * @return The selected controller window if any.
     */
    public G9DialogController getModal();

    /**
     * This method is called each time a window is created in an MDI setting.
     * This method is suitable for changing behaviour of internal frames, for
     * instance setting resizeable, iconifiable and icon for window. Example:
     *
     * <pre>
     *
     * public void commonViewStartup(JInternalFrame frame, G9DialogController gc) {
     *     frame.setResizable(false);
     *     frame.setIconifiable(true);
     *     frame.setFrameIcon(getMyLogo());
     * }
     * </pre>
     *
     * @param frame The internal frame just activated.
     * @param gc Its matching controller object.
     */
    public void commonViewStartup(JInternalFrame frame,
            G9DialogController gc);

    /**
     * This method is called each time a window is created outside of a MDI
     * context. See doc for commonViewStartup(JInternalFrame,
     * G9DialogController) for example of usage.
     *
     * @param frame The frame just activated
     * @param gc Its matching controller object.
     */
    public void commonViewStartup(JFrame frame, G9DialogController gc);

    /**
     * Returns the status bar, used for displaying messages.
     *
     * @return the status bar for the application.
     */
    public StatusBar getStatusBar();

    /**
     * Returns the windows present in the application.
     *
     * @return a List of opened windows.
     */
    public List getAddedWindows();


    /**
     * @return Returns the activeInternalWindow.
     */
    public G9DialogController getActiveInternalWindow();

    /**
     * @return Returns the previousActiveInternalWindow.
     */
//    public G9DialogController getPreviousActiveInternalWindow();

    /**
     * Returns the desktop pane of this application window.
     *
     * @return the <code>G9DesktopPane</code>
     */
    public G9DesktopPane getG9DesktopPane();

    /**
     * Returns the menu bar of the application.
     *
     * @return the <code>G9MenuBar</code>
     */
    public G9MenuBar getApplicationMenuBar();

    /**
     * Internal use!
     * <p>
     * Exit the application. Open dialogs that have a WMClose event defined are
     * closed gracefully. If all open dialogs where closed, the application
     * exits (invokes System.exit(0)).
     */
    public void actionExit();


    /**
     * Run a visitor on all open dialogs.
     *
     * @param pVisitor The actual {@link no.g9.client.support.DialogVisitor}
     * @return <code>true</code> if all visited dialogs returns
     *         <code>true</code>, otherwise false
     */
    public boolean visitAll(DialogVisitor pVisitor);

    /**
     * Returns the the application tool bar.
     *
     * @return the <code>G9ToolBar</code>
     */
    public G9ToolBar getApplicationToolBar();

    /**
     * Sets the last exception that occured.
     *
     * @param e the last exception
     */
    public void setLastException(RuntimeException e);

    /**
     * Gets the last exception that occured.
     *
     * @return the last exception
     */
    public RuntimeException getLastException();


    /**
     * If the checkForeginNodes flag is <code>true</code> (the default value),
     * foreign nodes are checked as part of the checkRowSelect when a selected
     * line changes in a listblock. The listblock might override this value.
     *
     * @return the checkForeignNodes flag
     */
    public boolean checkForeignNodes();

    /**
     * Sets the checkForeignNodes flag. If this flag is <code>true</code> (the
     * default value) then all nodes that are visible in a listblock are checked
     * for modifications when a listblock line select change occurs. If this
     * flag is <code>false</code> the listblock role's node is checked among
     * with its children (the nodes in the listblock role's object selection
     * tree).
     * <p>
     * This value can be overridden by a listblock instance, in which case the
     * overridden value will be used to determine the behaviour of the
     * checkRowSelect method.
     *
     * @param doCheck the boolean value of the checkForeingNodesFlag
     */
    public void setCheckForeignNodes(boolean doCheck);

}
