/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.awt.Component;
import java.awt.Container;
import java.awt.FocusTraversalPolicy;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.swing.ButtonModel;
import javax.swing.JComboBox;

import no.g9.client.component.G9ButtonGroup;
import no.g9.client.component.G9RadioButton;

/**
 * The custom traversal policy for a dialog box or a document window.
 */
@SuppressWarnings({"unchecked", "rawtypes"})
public class CustomTraversalPolicy extends FocusTraversalPolicy {

	/** Maps a component to the focus cycle index */
    private Map frameComponentToIndex;

	/** Array of componets ordered by focus sequence */
    private Component[] componentSequence;
    
    /** The last component to receive focus */
    private Component lastFocusedComponent = null;

    /**
     * Creates a new CustomTraversalPolicy instance.
     * 
     * @param controller the controller for the container where the components
     *            reside
     * @param componentOrder an array specifying the name and sequence of the
     *            focusable components.
     */
    public CustomTraversalPolicy(G9DialogController controller,
            String[] componentOrder) {
        frameComponentToIndex = initFocusIndex(controller, componentOrder);
        componentSequence = initComponentArray(frameComponentToIndex);
        addFocusListeners();
    }

    @Override
    public Component getComponentAfter(Container focusCycleRoot, Component aComponent) {
    	Component next = null;
    	Integer curIndex = (Integer) frameComponentToIndex.get(aComponent);
    	if (curIndex != null) {
    		for (int i = curIndex.intValue() + 1; i < componentSequence.length; i++) {
    			next = componentSequence[i];
    			if (acceptFocus(next)) {
    				return next;
    			}
    		}
    	}
    	return getFirstComponent(focusCycleRoot);
    }

    

    @Override
    public Component getComponentBefore(Container focusCycleRoot,
            Component aComponent) {
        
    	Component previous = null;
    	
    	Integer curIndex = (Integer) frameComponentToIndex.get(aComponent);
    	if (curIndex != null) {
    		for (int i = curIndex.intValue() - 1; i >= 0; i--) {
    			previous = componentSequence[i];
    			if (acceptFocus(previous)) {
    				return previous;
    			}
    		}
    	}
    	
    	return getLastComponent(focusCycleRoot);
    }

    
    @Override
    public Component getDefaultComponent(Container focusCycleRoot) {
    	return acceptFocus(lastFocusedComponent) ? 
    			lastFocusedComponent : getFirstComponent(focusCycleRoot);
    }

    @Override
    public Component getFirstComponent(Container focusCycleRoot) {
        Component first = null;
        for (int i = 0; i < componentSequence.length && first == null; i++) {
        	first = componentSequence[i];
        	if (!acceptFocus(first)) {
        		first = null;
        	}
        }
        
        return first;

    }

    @Override
    public Component getLastComponent(Container focusCycleRoot) {
        Component last = null;
        for (int i = componentSequence.length - 1; i >= 0 && last == null; i--) {
        	last = componentSequence[i];
        	if (!acceptFocus(last)) {
        		last = null;
        	}
        }
        
        return last;
    }

    /**
     * Check if <code>component</code> can receive focus. Only components that
     * are showing, accepting focus and is enabled can receive focus.
     * 
     * @param component the Component to check.
     * @return <code>true</code> if it is ok to assign focus to the specified
     *         component.
     */
    private boolean acceptFocus(Component component) {
    	
    	boolean accept = component != null && component.isEnabled()
				&& component.isFocusable() && component.isShowing();
    	
    	// Only accept selected radiogroupbutton if one is selected
    	if(component instanceof G9RadioButton) {
    		G9RadioButton rGrpButton= (G9RadioButton) component;
    		ButtonModel model = rGrpButton.getGroup().getSelection();
    		accept= accept && (rGrpButton.getGroup().isClear() || 
    						   model == rGrpButton.getModel());
    	}    	
        return accept;
    }


    /**
     * Sets up the componet array. Given an index, this array is used to
     * determine which component that is before and after the specified index.
     * 
     * @param componentToIndex the map from component to tab-sequence index.
     * @return the array of components
     */
    private Component[] initComponentArray(Map componentToIndex) {
        Component[] components = new Component[componentToIndex.size()];
        Iterator it = componentToIndex.keySet().iterator();
        while (it.hasNext()) {
            Component comp = (Component) it.next();
            int index = ((Integer) componentToIndex.get(comp)).intValue();
            components[index] = comp;
        }

        return components;
    }

    /**
     * Sets up the index map. Each component has a focus index, which makes it
     * possible to determine the component before and after in the focus cycle.
     * 
     * @param controller the G9DialogController where the components reside
     * @param componentOrder a (String) array of component names, the index
     *            gives the focus cycle sequence.
     * @return the map from component to tab sequence index.
     */
    private Map initFocusIndex(G9DialogController controller,
            String[] componentOrder) {
        Map componentToIndex = new HashMap();
        int radioGroupOffset = 0;
        for (int i = 0; i < componentOrder.length; i++) {
            Component comp = controller.getView().fromNameToComponent(
                    componentOrder[i]);
            comp = comboBoxCheck(comp);

            if (comp instanceof G9ButtonGroup.ComponentGroup) {
                G9ButtonGroup buttonComp = ((G9ButtonGroup.ComponentGroup) comp)
                        .getGroup();
                Enumeration e = buttonComp.getElements();
                // skip first element which is a hidden button used to
                // clear selection.
                if (e.hasMoreElements()) {
                    e.nextElement();
                }
                while (e.hasMoreElements()) {
                    comp = (Component) e.nextElement();
                    componentToIndex.put(comp, new Integer(i
                            + radioGroupOffset));
                    radioGroupOffset++;
                }

                radioGroupOffset--; // incremented one time to many in
                // while-loop.

            } else {
                componentToIndex.put(comp, new Integer(i
                        + radioGroupOffset));
            }

        }

        return componentToIndex;
    }

    /**
     * Editable combos needs custom handeling, since the component loosing focus
     * is not the combo but the editor component.
     * 
     * @param comp the component - might be an editable combo box
     * @return if <code>comp</code> is an editable combo, the editor component
     *         is returned. If not, <code>comp</code> is returned.
     */
    private Component comboBoxCheck(Component comp) {
        Component retVal = comp;
        if (retVal instanceof JComboBox) {
            JComboBox combo = (JComboBox) retVal;
            if (combo.isEditable()) {
                retVal = combo.getEditor().getEditorComponent();
            }
        }

        return retVal;
    }

    /**
     * Adds a focus listener to each component in order to determine the last
     * focused component.
     */
    private void addFocusListeners() {
    	
    	FocusListener listener = new FocusAdapter() {
			@Override public void focusGained(FocusEvent e) {
				lastFocusedComponent = (Component) e.getSource();
			}
		};
    	
    	for (int i = 0; i < componentSequence.length; i++) {
    		Component component = componentSequence[i];
    		component.addFocusListener(listener);
    	}
    }

}