/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;

import no.esito.log.Logger;
import no.g9.support.CustomDate;

import org.joda.time.DateMidnight;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

/**
 * <code>Document</code> model for fields with date formatting. To be used with the Joda DateMidnight type.
 */
@SuppressWarnings("deprecation")
public class DateMidnightDocument extends G9Document {

    /**
     * @param datatype
     *            the Genova datatype
     * @param inputFormat
     *            the input format to use when parsing text input
     * @param outputFormat
     *            the output format to use
     * @param maxLength
     *            maximum length in number of input characters
     */
    public DateMidnightDocument(int datatype, String inputFormat, String outputFormat, int maxLength) {
        super(datatype, inputFormat, outputFormat, maxLength);
    }

    @Override
    public void insertString(int offset, String str, AttributeSet attr) throws BadLocationException {
        if (str == null || str.length() == 0) {
            return;
        }
        if (!getInputMode()) {
            super.insertString(offset, str, attr);
            return;
        }
        String format = getInputFormat();
        if (str.length() + getLength() > format.length()) {
            return;
        }

        char aChar = str.charAt(0);
        char fChar = format.charAt(offset);
        if (Character.isDigit(aChar) == Character.isLetter(fChar)) {
            if (Character.isLetterOrDigit(fChar) || !Character.isLetterOrDigit(aChar)) {
                super.insertString(offset, str, attr);
                offset++;
                if (offset < format.length()) {
                    char bChar = format.charAt(offset);
                    // if the next char in the format is not a digit append it
                    if (!Character.isLetterOrDigit(bChar) && (getLength() == offset || getText(offset, 1).charAt(0) != bChar)) {
                        super.insertString(offset, "" + bChar, attr);

                    }
                }
            }
        }
    }

	@Override
    public synchronized Object transform(Object o) {
        return stringToDate(dateToString((DateMidnight) o));
    }

    @Override
    public synchronized void setValue(Object value) {
        String buf = null;
        super.setValue(value);
        if (value != null) {
            DateMidnight dt = (DateMidnight) value;
            buf = dt.toString(getFormat());
        }
        try {
            replace(0, getLength(), buf, null);
        } catch (BadLocationException e) {
            Logger.getLogger(DateMidnightDocument.class).warn("Failed to set value.", e);
        }
    }

    @Override
    public synchronized Object getValue() {
        Object result = super.getValue();
        if (getInputMode()) {
            String strValue = null;
            try {
                strValue = getText(0, getLength());
            } catch (BadLocationException e) {
                Logger.getLogger(DateMidnightDocument.class).warn("Failed to get value.", e);
            }
            if (strValue != null) {
                if (!(strValue.length() == 0 && result == null)) {
                    result = getFormatter().parseDateTime(strValue).toDateMidnight();
                }
            }
        }
        return result;
    }

    @Override
    public synchronized String format() {
        String str = "";
        DateMidnight dt = (DateMidnight) getValue();
        if (dt != null) {
            str = dt.toString(getInternalFormat());
        }
        return str;
    }

    @Override
    public synchronized Object parse(String str) {
        Object value = null;
        if (str != null && str.length() != 0) {
            value = getInternalFormatter().parseDateTime(str).toDateMidnight();
        }
        return value;
    }

    @Override
    public synchronized String getOutputText() {
        return getValue() != null ? ((DateMidnight) getValue()).toString(getOutputFormat()) : null;
    }

    private String dateToString(DateMidnight date) {
        if (date != null) {
            return date.toString(getFormat());
        }
        return "";
    }

    private DateMidnight stringToDate(String str) {
        if (str != null && !str.isEmpty()) {
            return getFormatter().parseDateTime(str).toDateMidnight();
        }
        return null;
    }

    private DateTimeFormatter getFormatter() {
        return DateTimeFormat.forPattern(getFormat());
    }

    private String getFormat() {
        return getInputMode() ? getInputFormat() : getOutputFormat();
    }

    private DateTimeFormatter getInternalFormatter() {
        return DateTimeFormat.forPattern(getInternalFormat());
    }

    private String getInternalFormat() {
        return CustomDate.getDatetimeInternalFormat(getDatatype());
    }

}
