/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;

import javax.swing.JDialog;
import javax.swing.SwingUtilities;

/**
 * Internal use.
 * <p>
 * This class is a utility class for opening a modal dialog, and still be able
 * to return before the modal dialog is disposed or hidden.
 */
public abstract class DialogBoxTool {

    /**
     * Internal use.
     * <p>
     * Shows the specified dialog and returns after the dialog is shown(even if
     * the dialog is modal) and activated.
     * 
     * @param dialog (missing javadoc)
     */
    public static void showDialog(final JDialog dialog) {
        if (dialog.isVisible()) {
            dialog.toFront();
            return;
        }

        DialogAdapter adapter = getAdapter(dialog);

        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
                dialog.setVisible(true);
                
            }
        });

        adapter.waitForWindowToOpen();
        adapter.waitForWindowToShow();
        return;

    }

    /**
     * Returns a dialog adapter for the specified dialog
     * 
     * @param dialog (missing javadoc)
     * @return the dialog adapter.
     */
    private static DialogAdapter getAdapter(JDialog dialog) {
        DialogAdapter adapter = null;

        WindowListener[] listeners = dialog.getWindowListeners();
        for (int i = 0; i < listeners.length; i++) {
            WindowListener foo = listeners[i];
            if (foo instanceof DialogAdapter) {
                adapter = (DialogAdapter) foo;
                break;
            }
        }

        if (adapter == null) {
            adapter = new DialogAdapter();
            dialog.addWindowListener(adapter);
        }

        return adapter;
    }

    /**
     * The dialog adapter extends WindowAdapter and is used to wait for the
     * dialog to open. This is needed because the thread opening a modal dialog
     * is stopped until the dialog closes.
     */
    private static class DialogAdapter extends WindowAdapter {
        /**
         * The window opened flag 
         */
        private boolean windowOpened;
        
        /**
         * The window shown flag
         */
        private boolean windowShown;
        
        
        /**
         * Waits for the dialog to open befor returning.
         */
        private synchronized void waitForWindowToOpen() {
            boolean interrupted = false;
            try {
                while (!windowOpened) {
                    try {
                        wait();
                    } catch (InterruptedException e) {
                        interrupted = true;
                    }
                }
            } finally {
                if (interrupted) {
                    Thread.currentThread().interrupt();
                }
            }
        }

        /**
         * Waits for the dialog to show before returning.
         */
        private synchronized void waitForWindowToShow() {
            boolean interrupted = false;
            try {
                while (!windowShown) {
                    try {
                        wait();
                    } catch (InterruptedException e) {
                        interrupted = true;
                    }
                }
            } finally {
                if (interrupted) {
                    Thread.currentThread().interrupt();
                }
            }
        }

        @Override
        public void windowOpened(WindowEvent e) {
            synchronized (this) {
                windowOpened = true;
                notify();
            }
        }

        @Override
        public void windowClosed(WindowEvent e) {
            synchronized (this) {
                windowOpened = false;
            }
        }

        @Override
        public void windowActivated(WindowEvent e) {
            synchronized (this) {
                windowShown = true;
                notify();
            }
        }

        @Override
        public void windowDeactivated(WindowEvent e) {
            synchronized (this) {
                if (!((JDialog) e.getSource()).isVisible()) {
                    windowShown = false;
                    notify();
                }
            }
        }
    }
    
}
