/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.text.ParseException;

import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;

import no.esito.log.Logger;
import no.esito.util.EnumUtil;
import no.g9.domain.G9Enums;
import no.g9.support.G9Enumerator;
import no.g9.support.ObjectFactory;

/**
 * The class controlling how enumerators are displayed in text components.
 */
@SuppressWarnings("rawtypes")
public class EnumeratorDocument extends G9Document {

    /** The g9 enumerations */
    private G9Enums enumerations;

    /** The g9 enumerator value */
    private Object enumValue;

    /** The type of the enumeration */
    private Class enumClass;

    /** The type name of the enumeration */
    String enumTypeName;

    /**
     * Constructs a new EnumeratorDocument
     *
     * @param datatype the datatype
     * @param inputFormat the input format
     * @param outputFormat the output format
     * @param maxLength the max length
     * @param enumerations the enumerations
     * @param enumClass the type of the enumeration
     */
    public EnumeratorDocument(int datatype, String inputFormat, String outputFormat,
            int maxLength, G9Enums enumerations, Class enumClass) {
        super(datatype, inputFormat, outputFormat, maxLength);
        this.enumClass = enumClass;
        this.enumerations = enumerations;
        enumTypeName = enumClass.getName();
        String[] foo = enumTypeName.split("\\.");
        enumTypeName = foo[foo.length - 1];
    }

    @Override
    public void insertString(int offset, String str, AttributeSet attr) throws BadLocationException {
        int maxLength = getMaxLength();
        if (str != null && maxLength > 0 && offset + str.length() > getMaxLength()) {
            int newlen = getMaxLength() - offset;
            if (newlen <= 0) {
                return;
            }
            str = str.substring(0,newlen);
        }

        super.insertString(offset, str, attr);
    }

    @Override
    public synchronized void setValue(Object value) {
        super.setValue(value);
        String textValue = "";
        if (value != null && (value instanceof G9Enumerator || value.getClass().isEnum())) {
        	enumValue = value;
        	textValue = EnumUtil.getEnumLiteral(enumValue);
        }
        try {
            replace(0, getLength(), textValue, null);
        } catch (BadLocationException e) {
            Logger.getLogger(EnumeratorDocument.class).warn("Failed to set value", e);
        }
    }

    @Override
    public synchronized Object getValue() {
        Object result = super.getValue();
        if (getInputMode()) {
            String strValue = null;
            try {
                strValue = getText(0, getLength());
            } catch (BadLocationException e) {
                Logger.getLogger(EnumeratorDocument.class).warn("Failed to get value", e);
            }
            if (strValue != null) {
                result = strValue;
            }
        }
        return toValue(result);
    }

    /**
     * Sets the enumerator value
     *
     * @param obj the value
     * @return the enumeration with the value set.
     */
    public Object toValue(Object obj) {
        if (obj == null || obj instanceof G9Enumerator || obj.getClass().isEnum()) {
            return obj;
        }
        int curVal = - 1;
        if (obj instanceof String) {
        	String text = (String) obj;
        	curVal = enumerations.toValue(enumTypeName, text);
        } else if (obj instanceof Integer) {
        	Integer curValObj = (Integer) obj;
        	curVal = curValObj.intValue();
        }
        if (curVal == G9Enums.NO_VALUE) {
        	return null;
        }
        if (enumClass.isEnum()) {
        	return EnumUtil.getEnumObject(enumClass.getName(), curVal);
        }
        else {
        	G9Enumerator enumerator = (G9Enumerator) ObjectFactory.newObject(enumClass.getName());
        	enumerator.setCurrentValue(curVal);
        	return enumerator;
        }
    }

    @Override
    public synchronized String format() {
        return enumToString(getValue());
    }

    private String enumToString(Object enumerator) {
        if (enumerator != null) {
            return EnumUtil.getEnumLiteral(enumerator);
        }
        return "";
    }

    @Override
    public synchronized Object transform(Object o) {
        Object retVal;
        try {
            retVal = parse(enumToString(o));
        } catch (ParseException e) {
            retVal = null;
        }

        return retVal;
    }

    @Override
    public synchronized Object parse(String str) throws ParseException {
        Object result = null;
        if (str != null && str.length() != 0) {
        	result = EnumUtil.getEnumObject(enumClass.getName(), str);
        }
        return result;
    }

}
