/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.awt.Component;
import java.awt.Container;
import java.awt.event.KeyEvent;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import no.g9.client.event.G9DataEvent;
import no.g9.domain.DomainObjectProxy;
import no.g9.exception.G9ClientException;

/** The general controller class for dialogs. Use this if need */
@SuppressWarnings({"rawtypes"})
public interface G9DialogController {

    /**
     * Returns the window used by this controller.
     *
     * @return The internal frame.
     */
    public Container getWindow();

    /**
     * Called upon startup of the dialog - used by the runtime system.
     * Preferably use a window open event defined in dmdialog to add code to
     * startup.
     */
    public void init();

    /**
     * Default handler for alle Runtime Exceptions.
     *
     * @param bad the caught exception
     * @return true to continue error handling, false if the error is handled
     */
    public boolean dialogExceptionHandler(Throwable bad);

    /**
     * The list of toolbar elements to be added to the dynamic list of menu
     * elements for this dialog as it recieves focus. The default method is
     * generated by g9.
     *
     * @deprecated use
     *             {@link no.g9.client.support.G9DialogFrame#getG9ToolBar()}
     * @return List of toolbar elements.
     */
    @Deprecated
    public Vector getToolBarElements();

    /**
     * Is called when someone sends a data event to the application.
     *
     * @param eventData The data object - user defined.
     */
    public void g9DataEvent(G9DataEvent eventData);

    /**
     * Destroys this dialog.
     */
    public void disposeDialog();

    /**
     * Returns the title of the main window as defined in the WindowBlock in
     * g9.
     *
     * @return The initial title for the dialog's window.
     */
    public String getWindowTitle();

    /**
     * Set the title of the dialog window.
     *
     * @param title the new title.
     */
    public void setWindowTitle(String title);


    /**
     * Returns the name of the server application for the dialogs object
     * selection
     *
     * @return The server application name
     */
    public String getServerApplicationName();

    /**
     * Sets the server application name for the dialogs object selection
     *
     * @param serverApplicationName (missing javadoc)
     */
    public void setServerApplicationName(String serverApplicationName);


    /**
     * Returns the dialog name of the dialog as defined in g9.
     *
     * @return The dialog name
     */
    public String getDialogName();

    /**
     * Performs obtain on the given object selection attribute found in this
     * controller.
     *
     * @param objSelAttr The object selection to obtain value from.
     * @return A String with the formatted value.
     */
    public String obtain(String objSelAttr);

    /**
     * Performs display on the given object selection with the given value found
     * in this controller.
     *
     * @param objSelAttr The object selection to display value to.
     * @param value The value to display.
     */
    public void display(String objSelAttr, Object value);

    /**
     * Updates the specified field from the object selection, and displays the
     * new value.
     *
     * @param objSelAttr the name of the attribute (as in the object selection)
     * @param value the (domain) object with the acutal value to display
     * @param resetState Set to false to avoid resetting the attribute state.
     */
    public void display(String objSelAttr, Object value, boolean resetState);

    /** Sends this dialog to the back of the MDI. */
    public void toBack();

    /** Sends this dialog to the front of the MDI. */
    public void toFront();

    /**
     * Recursivly enables or disables a component and all its children.
     *
     * @param component the component to disable
     * @param enabled if <code>true</code> the component is enabled.
     * @since version 8.1
     */
    public void setEnabledComponent(Component component, boolean enabled);

    /**
     * Internal use. Application developers should invoke
     * setEnabledComponent(Component, boolean).
     *
     * @param component (missing javadoc)
     * @param enabled (missing javadoc)
     * @param delay (missing javadoc)
     */
    public void setEnabledComponent(Component component, boolean enabled, boolean delay);

    /**
     * Recursivly enables or disables a component and all its children.
     *
     * @param name the (logical) name of the component to disable
     * @param enable if <code>true</code> the component is enabled.
     * @since version 8.1
     */
    public void setEnabledComponent(String name, boolean enable);

    /**
     * Recursivly hides or shows a component and all its children.
     *
     * @param component the component to show or hide
     * @param visible if <code>true</code> the component is shown.
     * @since version 8.1
     */
    public void setVisibleComponent(Component component, boolean visible);

    /**
     * Internal use. Application delvelopers should invoke
     * setVisibleComponent(component, visible). Recursivly hides or shows a
     * component and all its children.
     *
     * @param component the component to show or hide
     * @param visible if <code>true</code> the component is made visible,
     *            otherwise hidden
     * @param delay if <code>true</code> the visible task is delayed allowing
     *            other tasks on the EDT to finish.
     */
    public void setVisibleComponent(Component component, boolean visible, boolean delay);

    /**
     * Recursivly hides or shows a component and all its children.
     *
     * @param name the (locical) name the component to show or hide
     * @param visible if <code>true</code> the component is shown.
     * @since version 8.1
     */
    public void setVisibleComponent(String name, boolean visible);

    /**
     * Changes cursor for this dialog into a wait cursor. It is advicable that
     * the setNormalCursor is called before a new setWaitCursor call is made,
     * even though the code attempts to clean up after multiple call to
     * setWaitCursor prior to the call to setNormalCursor.
     */
    public void setWaitCursor();

    /**
     * Make this dialog modal. It does this by calling blockDialog() in all
     * dialogs open except the one that gets this call. pen dialogs exception
     * the one receving this call to It remains modal until the dialog until the
     * setModal(false) is performed. If a new dialog is created using the
     * createDialog then it will not be part of the blocked windows. If the
     * modal dialog is closed the blocked dialogs will reopen again.
     *
     * @param isModal True if modal, false to remove.
     */
    public void setModal(boolean isModal);

    /** Blocks dialog from receving any events whatsoever */
    public void blockDialog();

    /** UnBlocks dialog so it may again receive events. */
    public void unblockDialog();

    /**
     * Returns true if the dialog is blocked.
     *
     * @return Returns blocked status.
     */
    public boolean isBlocked();

    /** Changes cursos for this dialog into a normal cursor. */
    public void setNormalCursor();

    /**
     * Override this one if you want to control the next component to get
     * focus when using TAB.
     *
     * @return The component to have focus.
     * @param f
     *            The component which currently has focus.
     *            @deprecated
     */
//    public Component nextTabElement(Component f);

    /**
     * Override this one if you want to control the next component to get
     * focus when using -TAB.
     *
     * @return The component to have focus.
     * @param f
     *            The component which currently has focus.
     *            @deprecated
     */
//    public Component previousTabElement(Component f);

    /**
     * Callback from keyevents pass through this method. Used internally, do not
     * use unless certain of effec.
     *
     * @param event The key event.
     * @return True if the event was processed.
     */
    public boolean g9Accelerator(KeyEvent event);

    /**
     * Makes sure the dialog is considered clean.
     *
     */

    /**
     * Gets the dialog window representing this dialog.
     *
     * @return the dialog window.
     */
    public G9DialogFrame getView();

    /**
     * Perfoms checkClear on the specified root role
     *
     * @param roleName the name of the role
     * @return true if it's ok to clear the role (and all children)
     */
    public boolean checkClear(String roleName);

    /**
     * Performs checkClose on this dialog.
     *
     * @return true if it's ok to close the dialog.
     */
    public boolean checkClose();

    /**
     * Handles the return from another dialog
     *
     * @param returnObject (missing javadoc)
     */
    public void returnToCaller(Object returnObject);

    /**
     * Handles initialization of a dialog from another dialog
     *
     * @param returnObject (missing javadoc)
     * @param startObject (missing javadoc)
     */
    public void startInit(Object returnObject, Object startObject);

    /**
     * Clears all window blocks in this dialog.
     */
    public void clearDialog();

    /**
     * Gets the role name method pointer, e.g if the roleName is "Order" it
     * returns the field <code>theOrder</code>
     *
     * @param roleName the name of the role
     * @return the <code>ObjectSelectionNode</code> associated with the
     *         <code>roleName</code>.
     */
    public ObjectSelectionNode getObjectSelectionNode(String roleName);

    /**
     * Returns the controller for the application
     *
     * @return the application controller.
     */
    public ApplicationMethods getApplication();

    /**
     * Returns the a list of role objects contained in this dialog representing
     * the specified domain object.
     *
     * @param anObject the domain object
     * @return a list of role objects.
     */
    public List getRoleObjects(Object anObject);

    /**
     * Internal use, returns a new DomainObjectProxy for the specified object and role name
     * @param o the object
     * @param roleName the role name
     * @return a object proxy
     */
    public DomainObjectProxy getNewProxy(Object o, String roleName);

    /**
     * This method is invoked <em>after</em> the dialog is fully realized and
     * drawn on screen, so that you migth modify fields, set values etc.
     * <p>
     * <em>Note</em>This method is called from the EDT, thus if you need to do
     * some time consuming task, you should set up a dedicated thread and let
     * the gui carry on (you might want to block the dialog though...).
     * <p>
     * The pattern below show how to both set up a dedicated thread for the
     * (possibly) time consuming task, and how to get back to the EDT for
     * updating the dialog window.: <code>
     * // SET UP NEW THREAD CODE
     * Runnable longTask = new Runnable() {
     *          public void run() {
     *              final Object someObject = LINES WITH TIME CONSUMING CODE...
     *
     *              // G9_INTERNAL_DISPLAY CODE - GET BACK TO EDT
     *              SwingUtilities.invokeLater(new Runnable() {
     *                  public void run() {
     *                      // GUI CODE
     *                      display(someObject);
     *                  }
     *              });
     *          }
     *     };
     *
     * // INIT THREAD
     * Thread longTimeThread = new Thread(longTask, "thread name");
     * longTimeThread.setPriority(Thread.NORM_PRIORITY);
     *
     * // START THREAD
     * longTimeThread.start();
     *
     * </code>
     */
    public void setAccessPolicy();

    /**
     * Returns the ignoreCheckChanged property.
     *
     * @return the ignoreCheckChanged property
     * @see #setIgnoreCheckChanged(boolean)
     */
    public boolean ignoreCheckChanged();

    /**
     * Sets whether check change is to be executed or not. If the
     * ignoreCheckChange property is set to <code>true</code>, check change will
     * not be executed on nodes in the dialog.
     *
     * @param ignoreCheckChange the value of the checkChanged property.
     */
    public void setIgnoreCheckChanged(boolean ignoreCheckChange);

    /**
     * Recursivly clears all nodes from the object selection.
     */
    public void clearObjectSelection();

    /**
     * Returns a map of the role objects.
     *
     * @return a map of role objects.
     */
    public Map getRoleObjects();

    /**
     * Check if the specified role attribute is changed. If the specified role
     * attribute is not found a G9ClientException is thrown.
     *
     * @param osAttributeName the name of the attribute
     * @return <code>true</code> if the field is changed, otherwise
     *         <code>false</code>
     * @throws G9ClientException if the specified field is not found.
     * @see ObjectSelectionNode#isChanged(String)
     */
    public boolean isChanged(String osAttributeName)
            throws G9ClientException;

    /**
     * Internal use. Forwards the call to the specified hook method.
     *
     * @param roleName the name of the node to check
     * @return the result of the check
     */
    public abstract ECheckResult callHookCheckSave(String roleName);

    /**
     * Internal use. Forwards the call to the specified hook method.
     *
     * @param roleName the name of the node to check
     * @return the result of the check
     */
    public abstract ECheckResult callHookCheckDelete(String roleName);

    /**
     * Internal use. Forwards the call to the specified hook method.
     *
     * @param roleName the name of the node to check
     * @return the result of the check
     */
    public abstract ECheckResult callHookCheckClose(String roleName);

    /**
     * Internal use. Forwards the call to the specified hook method.
     *
     * @param roleName the name of the node to check
     * @return the result of the check
     */
    public abstract ECheckResult callHookCheckClear(String roleName);

    /**
     * Internal use. Forwards the call to the specified hook method.
     *
     * @param roleName the name of the node to check
     * @return the result of the check
     */
    public abstract ECheckResult callHookCheckFind(String roleName);

    /**
     * Internal use. Forwards the call to the specified hook method.
     *
     * @param roleName the name of the node to check
     * @return the result of the check
     */
    public abstract ECheckResult callHookCheckChange(String roleName);

    /**
     * Internal use. Forwards the call to the specified hook method.
     *
     * @param roleName the name of the node to check
     * @param curSel the current selection
     * @param newSel the new selection
     * @return the result of the check
     */
    public abstract ECheckResult callHookCheckRowSelect(String roleName, int curSel, int newSel);

    /**
     * Internal use. Forwards the call to the specified hook method.
     *
     * @param roleName the name of the node to check
     * @return the result of the check
     */
    public abstract ECheckResult callHookCheckPrint(String roleName);

}


