/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.awt.Component;
import java.awt.Container;
import java.awt.event.HierarchyEvent;
import java.awt.event.HierarchyListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.beans.PropertyVetoException;
import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JInternalFrame;
import javax.swing.JTabbedPane;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;
import javax.swing.WindowConstants;
import javax.swing.event.EventListenerList;

import no.g9.client.component.G9ComboBox;
import no.g9.client.component.G9DesktopPane;
import no.g9.client.component.G9TabPaneManager;
import no.g9.client.component.G9ToolBar;
import no.g9.client.component.menu.G9MenuBar;
import no.g9.client.event.G9WindowEvent;
import no.g9.client.event.G9WindowListener;
import no.g9.exception.G9ClientFrameworkException;

/**
 * General class representing a dialog window. In addition to a JInternalFrame,
 * this class has the capability to:
 * <ul>
 * <li>Add G9WindowEvent listeners, (using the the addInternalFrame method.
 * After this window is made visible (by calling on the setVisible method) it
 * fires a G9WindowEvent, signaling that this window is indeed visible. This
 * is usefull for listeners interessted in customizing the GUI, and is mostly
 * for internal use.
 * <li>- Enable and disable components by the method
 * <code>setEnabledComponent</code>
 * <p>
 * It works as follows:
 * <ul>
 * <li>All subcomponents of a disabled component is disabled
 * <li>Enabelig a disabled component only enables subcomponents that either have
 * explicitly been made enabled through the <code>setEnabledComponent</code>
 * method or was enabled at the time this component was disabled.
 * </ul>
 * </ul>
 * <p>
 * A <code>G9MenuBar</code> might be added to the dialog window. If so, the
 * G9DesktopPane, will manage the dialog windows menu bar, and merge it with
 * the application bar when ever the dialog window has focus, and clean up the
 * application menu bar when needed. Please note that the the menu bar will be
 * temporarily empty while it is merged with the applications menu bar (the
 * components in the menu bar can only be in one container).
 */
@SuppressWarnings({"unchecked", "rawtypes"})
public abstract class G9DialogFrame extends JInternalFrame {

    /**
     * <code>disabledComponents</code> should contain at any time the disabled  
     * components in the dialog. It is the job of the subclass initialise this
     * set with "default" hidden dialog components.
     */
    protected final Set disabledComponents = new HashSet();

    /** The list of event listeners */
    protected EventListenerList g9WindowEventListeners = new EventListenerList();

    /** Map names to components */
    protected Map nameToComponent = new HashMap();

    /** Map components to names */
    protected Map componentToName = new HashMap();

    /** The dialog menu bar */
    private G9MenuBar menuBar;

    /** The dialog tool bar */
    private G9ToolBar toolBar;

    /** The set of registered mnemonics */
    private Set registeredMnemonics;

    /** Map note book tabs to the tab pane's manager */
    private Map<Component, G9TabPaneManager> tabPaneMap = new HashMap<Component, G9TabPaneManager>();

    /**
     * Can't find the proper way to access the modifier, thus we initialize it
     * the hard and ugly way...
     */
    private static int mouselessModifier = InputEvent.ALT_DOWN_MASK;

    /**
     * Default constructor. Creates a new resizable, closable, maximizable and
     * iconifable dialog window with no title.
     */
    public G9DialogFrame() {
        this("", true, true, true, true);
    }

    /**
     * Constructs a new dialog frame.
     * 
     * @param title
     *            the title of the frame (used in title bar)
     * @param resizable
     *            if <code>true</code> the frame is resizable
     * @param closable
     *            if <code>true</code> the frame is closable
     * @param maximizable
     *            if <code>true</code> the frame is maxiizable
     * @param iconifable
     *            if <code>true</code> the frame is iconifable
     * @see JInternalFrame
     */
    public G9DialogFrame(String title, boolean resizable, boolean closable,
            boolean maximizable, boolean iconifable) {
        super(title, resizable, closable, maximizable, iconifable);
        setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
        addHierarchyListener(new HierarchyListener() {

            @Override
            public void hierarchyChanged(HierarchyEvent e) {
                if (HierarchyEvent.SHOWING_CHANGED == (e.getChangeFlags() & HierarchyEvent.SHOWING_CHANGED)) {
                    JInternalFrame internalFrame = (JInternalFrame) e
                            .getSource();
                    if (!internalFrame.isVisible()) {
                        fireG9WindowEvent(G9WindowEvent.G9_DIALOG_HIDDEN);
                    }
                }
            }

        });
    }
    
    /**
     * Registers a tab pane and the corresponding tab pane manager.
     * @param page the managed tab pane.
     * @param manager the tab pane manager.
     */
    protected void addNotePage(Component page, G9TabPaneManager manager) {
        tabPaneMap.put(page, manager);
    }
    
    /**
     * Gets the tab pane manager for the specified tab page.
     * @param page the tab page.
     * @return the tab pane manager of the page.
     */
    protected G9TabPaneManager getTabManager(Component page) {
        return tabPaneMap.get(page);
    }

    /**
     * Fires a g9 window event.
     * 
     * @param id
     *            the type of the event being fired; one of the following:
     *            <ul>
     *            <li><code>G9WindowEvent.G9_DIALOG_VISIBLE</code></li>
     *            <li><code>G9WindowEvent.G9_DIALOG_CREATED</code></li>
     *            <li><code>G9WindowEvent.G9_DIALOG_HIDDEN</code></li>
     *            </ul>
     *            If the event type is not one of the above, nothing happens.
     */
    protected void fireG9WindowEvent(int id) {
        if (g9WindowEventListeners == null) {
            return;
        }
        Object[] listeners = g9WindowEventListeners.getListenerList();
        G9WindowEvent e = null;

        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (e == null) {
                e = new G9WindowEvent(this, id);
            }
            if (listeners[i] == G9WindowListener.class) {
                switch (e.getID()) {
                case G9WindowEvent.G9_DIALOG_VISIBLE:
                    ((G9WindowListener) listeners[i + 1]).dialogVisible(e);
                    break;
                case G9WindowEvent.G9_DIALOG_CREATED:
                    ((G9WindowListener) listeners[i + 1]).dialogCreated(e);
                    break;
                case G9WindowEvent.G9_DIALOG_HIDDEN:
                    ((G9WindowListener) listeners[i + 1]).dialogHidden(e);
                    break;
                default:
                    break;
                }
            }
        }
    }

    /**
     * Adds the specified listener to receive g9 window events from this
     * g9 window.
     * 
     * @param l
     *            the g9 window listener
     */
    public void addG9WindowListener(G9WindowListener l) {
        g9WindowEventListeners.add(G9WindowListener.class, l);
    }

    /**
     * Makes the component visible or invisible. Overrides
     * <code>JInternalFrame.setVisible</code>. Each time a g9 dialog frame
     * is made visible, it fires a G9WindowEvent, after the frame is
     * painted.
     * 
     * @param visible
     *            true to make the component visible; false to make it invisible
     * @beaninfo attribute: visualUpdate true
     */

    @Override
    public void setVisible(boolean visible) {
        setVisible(visible, false);
    }

    /**
     * Sets visible and invokes setAccessPolicy if <code>firsTimeOpened</code>
     * is true.
     */
    class VisibleAndAccessTask implements Runnable {

        /** (missing javadoc) */
        boolean visible;

        /** (missing javadoc) */
        boolean firstTimeOpened;

        /**
         * @param visible
         *            (missing javadoc)
         * @param firstTimeOpened
         *            (missing javadoc)
         */
        VisibleAndAccessTask(boolean visible, boolean firstTimeOpened) {
            this.visible = visible;
            this.firstTimeOpened = firstTimeOpened;
        }

        @Override
        public void run() {
            superSetVisible(visible);
            if (visible && firstTimeOpened) {
                getController().setAccessPolicy();
                fireG9WindowEvent(G9WindowEvent.G9_DIALOG_VISIBLE);
            }
        }
    }

    /**
     * Makes the component visible or invisible. If the
     * <code>callSetAccessPolicy</code> parameter is <code>true</code> the
     * controller's <code>setAccessPolicy</code> method is invoked before the
     * opened event is fired (only applies if the window is made visible). The
     * actual showing/hiding of the internal frame is performed by invoking
     * <code>super.setVisible(boolean)</code>.
     * 
     * @param visible
     *            if <code>true</code> the frame is painted and shown.
     * @param firstTimeOpened
     *            if <code>true</code> and <code>visible</code> is also
     *            <code>true</code> the setAccessPolicy-method is invoked, and
     *            the opened event is fiered
     * @see Component#setVisible(boolean)
     */
    public void setVisible(boolean visible, boolean firstTimeOpened) {

        if (SwingUtilities.isEventDispatchThread()) {
            superSetVisible(visible);
            if (visible && firstTimeOpened) {
                getController().setAccessPolicy();
                fireG9WindowEvent(G9WindowEvent.G9_DIALOG_VISIBLE);

            }
        } else {
            try {

                SwingUtilities.invokeAndWait(new VisibleAndAccessTask(visible,
                        firstTimeOpened));
            } catch (InterruptedException e) {
                throw new G9ClientFrameworkException(e);
            } catch (InvocationTargetException e) {
                throw new G9ClientFrameworkException(e);

            }
        }
    }

    private void superSetVisible(boolean visible) {
        super.setVisible(visible);
    }

    /**
     * Recursivly enables or disables a component and all its children except
     * the components in <code>disableComponents</code>.
     * 
     * @param component
     *            the component to enable or disable
     * @param enable
     *            if <code>true</code> the component is enabled.
     * @since version 8.1
     */
    public void setEnabledComponent(Component component, boolean enable) {

        setEnabledComponent(component, enable, true);
    }

    /**
     * Internal use only.
     * <p>
     * Recursivly enables or disables a component and all its children except
     * the components in <code>disableComponents</code>. The <code>delay</code>
     * parameter controlls wheter the disabling should be delayed or not. If,
     * <code>true</code> other tasks and events (such as focus shifts) on the
     * EDT are allowed to finish before this task.
     * 
     * @param component
     *            the component to enable or disable
     * @param enable
     *            if <code>true</code> the component is enabled
     * @param delay
     *            if <code>true</code> the task is put last on the EDT
     */
    public void setEnabledComponent(Component component, boolean enable,
            boolean delay) {
        EnableTask task = new EnableTask(component, enable);
        if (delay) {
            SwingUtilities.invokeLater(task);
        } else {
            task.run();
        }
    }

    private class EnableTask implements Runnable {
        private Component component;

        private boolean enable;

        private EnableTask(Component component, boolean enable) {
            this.component = component;
            this.enable = enable;
        }

        @Override
        public void run() {
            boolean isAncestorDisabled = isAncestorDisabled(component);

            if (component instanceof G9DialogFrame) {
                getG9MenuBar().setEnabledWholeMenuBar(enable);
                getG9ToolBar().setEnabledWholeToolBar(enable);
            }

            if (enable) {
                if (disabledComponents.remove(component) && !isAncestorDisabled) {
                    enableComponent(component);
                }
            } else {
                if (disabledComponents.add(component) && !isAncestorDisabled)
                    disableComponent(component);
            }
        }
    }

    /**
     * Returns <code>true</code> if the given component is enabled otherwise
     * <code>false</code>.
     * 
     * @param component
     *            a component
     * @return Returns <code>true</code> if the given component is enabled
     *         otherwise <code>false</code>.
     */
    public boolean isEnabledComponent(Component component) {
        return !disabledComponents.contains(component);
    }

    /**
     * Returns <code>true</code> if any ancestor of <code>component</code> is
     * disabled
     * 
     * @param component
     *            (missing javadoc)
     * @return <code>true</code> if there exist an ancestor that is disabled,
     *         otherwise <code>false</code>
     * @since version 8.1
     */
    private boolean isAncestorDisabled(Component component) {
        if (component == null) {
            return false;
        }
        Component parent = component.getParent();
        if (parent != null && !parent.isEnabled()) {
            return true;
        }
        return isAncestorDisabled(parent);
    }

    /**
     * Recursivly enables a component and all its children except the components
     * in <code>disableComponents</code>.
     * 
     * @param component
     *            the component to enable
     */
    private void enableComponent(Component component) {
        if (component instanceof Container) {
            Container container = (Container) component;
            if (!disabledComponents.contains(component)) {
                if (container.getParent() != null
                        && container.getParent() instanceof JTabbedPane) {
                    JTabbedPane pane = (JTabbedPane) container.getParent();
                    pane.setEnabledAt(pane.indexOfComponent(component), true);
                } else {
                    container.setEnabled(true);
                }
                Component[] child = container.getComponents();
                for (int i = 0; i < child.length; i++) {
                    enableComponent(child[i]);
                }
            }
        }
    }

    /**
     * Recursivly disables a component and all its children.
     * 
     * @param component
     *            the component to disable
     * @since version 8.1
     */
    protected void disableComponent(Component component) {
        if (component instanceof Container) {
            Container container = (Container) component;
            if (container.getParent() != null
                    && container.getParent() instanceof JTabbedPane) {
                JTabbedPane pane = (JTabbedPane) container.getParent();
                pane.setEnabledAt(pane.indexOfComponent(component), false);
            } else {
                container.setEnabled(false);
            }
            Component[] child = container.getComponents();
            for (int i = 0; i < child.length; i++) {
                disableComponent(child[i]);
            }
        }
    }

    private class VisibleTask implements Runnable {
        /** The component to show or hide */
        Component component;

        /** The visible flag */
        boolean visible;

        private VisibleTask(Component component, boolean visible) {
            this.component = component;
            this.visible = visible;
        }

        @Override
        public void run() {
            if (component instanceof JInternalFrame) {
                if (component instanceof G9DialogFrame) {
                    G9DialogFrame gFrame = (G9DialogFrame) component;
                    if (gFrame.getController().getWindow() instanceof JDialog) {
                        JDialog dFrame = (JDialog) gFrame.getController()
                                .getWindow();
                        dFrame.setVisible(visible);
                        return;
                    }

                }
                JInternalFrame frame = (JInternalFrame) component;
                if (visible) {
                    frame.show();
                    G9DesktopPane desktop = (G9DesktopPane) frame
                            .getDesktopPane();
                    if (desktop != null) {
                        desktop.frameShown(frame);
                    }
                    try {
                        frame.setSelected(true);
                        frame.getFocusTraversalPolicy().getDefaultComponent(
                                frame).requestFocus();
                    } catch (PropertyVetoException e) {
                        // Could not select or focus. Not much to do...
                    }
                } else {
                    frame.setVisible(false);
                    // frame.hide();
                    G9DesktopPane desktop = (G9DesktopPane) frame
                            .getDesktopPane();
                    if (desktop != null) {
                        desktop.frameHidden(frame);
                    }
                    getDesktopPane().repaint();
                }
            } else {
                G9TabPaneManager tabManager = getTabManager(component);
                if (tabManager != null) {
                    tabManager.setVisible(component, visible);
                } else {
                    component.setVisible(visible);
                }
            }
        }
    }

    /**
     * Shows or hide the specified component.
     * 
     * @param component
     *            the component to show or hide
     * @param visible
     *            flag indicating if component should be shown or hidden,
     *            <code>true</code> indicating show.
     * @param delay
     *            flag indicating if component should be shown or hidden
     *            imidiatly or delayed (allowing other tasks on EDT to finish)
     */
    public void setVisibleComponent(Component component, boolean visible,
            boolean delay) {
        VisibleTask visibleTask = new VisibleTask(component, visible);
        if (delay) {
            SwingUtilities.invokeLater(visibleTask);
        } else {
            visibleTask.run();
        }
    }

    /**
     * Recursivly hides a component and all its children or recursivly shows a
     * component and all its children except explicedly hidden children.
     * 
     * @param component
     *            the component to show or hide
     * @param visible
     *            if <code>true</code> the component is shown.
     * @since version 8.1
     */
    public void setVisibleComponent(Component component, boolean visible) {
        setVisibleComponent(component, visible, true);
    }

    /**
     * Returns the swing component associated with the specified attribute name.
     * 
     * @param name
     *            the name of the attribute, e.g. DomainClass.id
     * @return the swing component used to display the attribute.
     */
    public JComponent fromNameToComponent(String name) {
        return (JComponent) nameToComponent.get(name);
    }

    /**
     * Returns the name of the attribute displayed by the specified component.
     * 
     * @param component
     *            the component that displays an attribute
     * @return the name of the attribute
     */
    public String fromComponentToName(JComponent component) {
        return (String) componentToName.get(component);
    }

    /**
     * Sets this dialog windows menu bar (using the inherited
     * <code>setJMenuBar(JMenuBar)</code> method).
     * <p>
     * If a dialog window has a G9MenuBar, the menu bar will be
     * automatically merged with the application menu bar when needed. See class
     * description for details.
     * 
     * @param menuBar
     *            the menu bar of this dialog window.
     */
    public void setG9MenuBar(G9MenuBar menuBar) {
        this.menuBar = menuBar;
    }

    /**
     * Gets the <code>G9MenuBar</code> of this dialog window, or
     * <code>null</code> if this dialog window does not have a
     * <code>G9MenuBar</code> set.
     * 
     * @return the <code>G9MenuBar</code> of this dialog window.
     */
    public G9MenuBar getG9MenuBar() {
        return menuBar;
    }

    /**
     * Sets this dialog windows tool bar.
     * <p>
     * If a dialog window has a G9ToolBar, the tool bar will be
     * automatically merged with the application menu bar when needed. See class
     * description for details.
     * 
     * @param toolBar
     *            the menu bar of this dialog window.
     */
    public void setG9ToolBar(G9ToolBar toolBar) {
        this.toolBar = toolBar;
    }

    /**
     * Gets the <code>G9ToolBar</code> of this dialog window, or
     * <code>null</code> if this dialog window does not have a
     * <code>G9ToolBar</code> set.
     * 
     * @return the <code>G9ToolBar</code> of this dialog window.
     */
    public G9ToolBar getG9ToolBar() {
        return toolBar;
    }

    /**
     * Returns the controller for this dialog.
     * 
     * @return the dialog controller.
     */
    public abstract G9DialogController getController();

    /**
     * Internal use.
     * 
     * @param code
     *            (missing javadoc)
     * @return the modified code.
     * @deprecated (missing javadoc)
     */
    @Deprecated
    protected int modifyKeyCode(int code) {
        if (code >= 'a' && code <= 'z') {
            return code - ('a' - 'A');
        }
        switch (code) {
        // case 'æ':
        // return 'Æ';
        // case 'ø':
        // return 'Ø';
        // case 'å':
        // return 'Å';
        case 64702:
            return KeyEvent.VK_F1;
        case 64703:
            return KeyEvent.VK_F2;
        case 64704:
            return KeyEvent.VK_F3;
        case 64705:
            return KeyEvent.VK_F4;
        case 64706:
            return KeyEvent.VK_F5;
        case 64707:
            return KeyEvent.VK_F6;
        case 64708:
            return KeyEvent.VK_F7;
        case 64709:
            return KeyEvent.VK_F8;
        case 64710:
            return KeyEvent.VK_F9;
        case 64711:
            return KeyEvent.VK_F10;
        case 64712:
            return KeyEvent.VK_F11;
        case 64713:
            return KeyEvent.VK_F12;
        case 64767:
            return KeyEvent.VK_DELETE;
        case 64539:
            return KeyEvent.VK_ESCAPE;
        }
        return code;
    }

    /**
     * Returns true if this dialog is modal.
     * 
     * @return a boolean value, true indicating that the frame is modal.
     * @deprecated Internal frames support of modality is broken.
     */
    @Deprecated
    public abstract boolean isModal();

    /**
     * Overrides the setClosed method in JInternalFrame. When this method is
     * invoked (as it is when the user presses Ctrl F4, the default close action
     * is invoked.
     * 
     * @param b
     *            not used
     */
    @Override
    public void setClosed(boolean b) {
        doDefaultCloseAction();
    }

    /**
     * <em>Internal use.</em>
     * <p>
     * Used by listblock lines that needs to supply a renderer for a cell
     * holding a null value. If the cell value is <code>null</code> the line
     * invokes this method and will pass the resulting value to the
     * cell-renderer.
     * <ul>
     * <li>If the corresponding edit-field is a check box, a Boolean initialized
     * to <code>false</code> is returned.
     * <li>If the corresponding edit-field is a combo box, the text
     * representation of the combo's first item is returned.
     * <li>Otherwise <code>null</code> is returned.
     * </ul>
     * 
     * @param attribute
     *            the domain attribute
     * @return a properly configured value the cell renderer can use.
     */
    protected Object getNullValueForCell(String attribute) {
        Object retVal = null;
        JComponent editFieldComponent = fromNameToComponent(attribute);
        if (editFieldComponent instanceof JCheckBox) {
            retVal = new Boolean(false);
        } else if (editFieldComponent instanceof G9ComboBox) {
            G9ComboBox combo = (G9ComboBox) editFieldComponent;
            retVal = combo.format(combo.getItemAt(0));
        }

        return retVal;
    }

    /** (missing javadoc) */
    protected Set mnemonicsComponents = new HashSet();

    /**
     * Internal use. Registers a mnemonic
     * 
     * @param mnemonic
     *            the mnemonic
     */
    protected void registerMnemonic(int mnemonic) {
        KeyStroke stroke = KeyStroke.getKeyStroke(mnemonic,
                G9DialogFrame.mouselessModifier);
        if (registeredMnemonics == null) {
            registeredMnemonics = new HashSet();

        }
        registeredMnemonics.add(stroke);
    }

    /**
     * Internal use. Adds the button to the list of buttons with mnemonics.
     * 
     * @param button
     *            (missing javadoc)
     */
    protected void addToMnemonicButtons(JButton button) {
        mnemonicsComponents.add(button);
    }

    /**
     * Internal use. Removes the buttons with mnemonics. This is done in order
     * to avoid a memory leakage when creating a dialog box.
     */
    protected void cleanUpMnemonicButtons() {
        Iterator it = mnemonicsComponents.iterator();
        while (it.hasNext()) {
            Component comp = (Component) it.next();
            remove(comp);
            comp.removeNotify();
        }
    }

    /**
     * Internal use. Check if the specified stroke constitutes a registered
     * mnemonic.
     * 
     * @param stroke
     *            the stroke
     * @return <code>true</code> if the stroke is a registered mnemonic
     */
    public boolean isRegisteredMnemonic(KeyStroke stroke) {
        return registeredMnemonics != null
                && registeredMnemonics.contains(stroke);
    }

}
