/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.awt.Component;
import java.text.ParseException;

import javax.swing.JTable;
import javax.swing.table.DefaultTableCellRenderer;

/**
 * The cell render used to render cell values
 *
 */
public class G9TableCellRenderer extends DefaultTableCellRenderer {
    
    private G9Document document;
    
    /**
     * Creates a new cell render. The specified document is used to format cell values
     * @param document the document that will be used to format cell values
     * @param justification the horizontal justification
     */
    public G9TableCellRenderer(G9Document document, int justification) {
        this.document = document;
        setHorizontalAlignment(justification);
    }
    
    
    /**
     * Sets the <code>String</code> object for the cell being rendered to
     * <code>value</code>.
     * 
     * @param value  the string value for this cell; if value is
     *      <code>null</code> it sets the text value to an empty string
     * @see javax.swing.JLabel#setText
     * 
     */
    @Override
    protected void setValue(Object value) {
        
        if (value instanceof String) {
            try {
                value = document.parse((String) value);
            } catch (ParseException e) {
                value = null;
            }
        }
        
        document.setValue(value);
        
        setText(document.getOutputText());
    }


    @Override
    public Component getTableCellRendererComponent(JTable table, Object value,
			boolean isSelected, boolean hasFocus, int row, int column) {
    	
		Component rendererComponent = super.getTableCellRendererComponent(
				table, value, isSelected, hasFocus, row, column);
		rendererComponent.setEnabled(isEnabled());
		
		if (table instanceof G9Table) {
			G9Table gt = (G9Table) table;
			boolean isChanged = gt.isChanged(row);
			if (isSelected) {
				if (!hasFocus) {
					rendererComponent.setForeground(table.getSelectionForeground());
				} 
			} else if (isChanged){
				rendererComponent.setForeground(gt.getChangedForgroundColor());
			} else {
				rendererComponent.setForeground(table.getForeground());
			}
			
			CellRenderOverride cro = gt.getRenderOverride();
			if (cro != null) {
				rendererComponent = cro.getTableCellRendererComponent(
						rendererComponent, gt.getListblock().getLine(row),
						value, isSelected, hasFocus, row, column);
			}
		}
		
		return rendererComponent;
	}
    
}
