/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.util.Vector;
import java.util.Enumeration;
import java.util.Comparator;
import java.util.Collections;

/**
 * Internal use.
 * <p>
 * Used to keep track of a line of objects in a list.
 */
@SuppressWarnings({"unchecked", "rawtypes"})
public class NodeInfoList {

    private Vector list = new Vector();

    /**
     * Returns an enumeration of the list's elements.
     * 
     * @return the list's elements.
     */
    public Enumeration elements() {
        return list.elements();
    }

    /**
     * Returns the vector holding all lines. Note that this is the same
     * vector that is used internally by this class.
     * 
     * @return a vector with all lines.
     */
    public Vector allLines() {
        return list;
    }

    /**
     * Sorts the list using the supplied comparator as argument to
     * Collections.sort(List, Comparator).
     * 
     * @param c the comparator
     */
    public void sort(Comparator c) {
        Collections.sort(list, c);
    }

    /**
     * Returns the size of the list
     * 
     * @return the size of the list.
     */
    public int getVisualRowCount() {
        return list.size();
    }

    /**
     * Adds a line to the list
     * 
     * @param ob the line
     *            
     */
    public void addLine(Object ob) {
        list.add(ob);
    }

    /**
     * Adds a line to the list at the specified position.
     * 
     * @param ob the line
     * @param index the position of the line
     */
    public void addLine(Object ob, int index) {
        list.insertElementAt(ob, index);
    }

    /**
     * Replaces the line at the specified index with the new line.
     * 
     * @param ob the new line
     * @param index the position of the line
     */
    public void setLine(Object ob, int index) {
        list.setElementAt(ob, index);
    }

    /**
     * Returns the line at the specified index
     * 
     * @param index the position of the line
     * @return a line
     */
    public Object getLine(int index) {
        return list.elementAt(index);
    }

    /**
     * Clears the list of lines.
     */
    public void clearLines() {
        list.clear();
    }

    /**
     * Sorts the list of lines and then removes the specified lines from the
     * list.
     * 
     * @param lines the array of indexes to remove from the list.
     */
    public void clearLines(int lines[]) {
        /*
         * First sort the lines to be sure that we can remove them from the
         * top
         */
        java.util.Arrays.sort(lines);
        int lastDelete = lines[lines.length - 1];
        list.remove(lastDelete);
        int deletePosition = lines.length - 1;
        while (deletePosition >= 0) {
            /*
             * Then only remove the lines, once per time they are asked
             * removed
             */
            if (lastDelete != lines[deletePosition]) {
                lastDelete = lines[deletePosition];
                list.remove(lastDelete);
            }
            deletePosition--;
        }
    }

    /**
     * Moves an element in the list to a new position
     * 
     * @param from the current position
     * @param to the new position
     */
    public void moveElement(int from, int to) {
        Object ob = list.elementAt(from);
        list.insertElementAt(ob, to);
        list.remove(from);
    }

    /**
     * Removes the line at the specified position
     * 
     * @param index the position of the line
     */
    public void clearLine(int index) {
        list.remove(index);
    }

    /**
     * Returns the size of the list.
     * 
     * @return the list's size.
     */
    public int size() {
        return list.size();
    }
    
}
