/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support;

import java.beans.PropertyVetoException;
import java.lang.reflect.InvocationTargetException;
import javax.swing.JDialog;
import javax.swing.JInternalFrame;
import javax.swing.SwingUtilities;
import no.g9.exception.G9ClientException;
import no.g9.exception.G9ClientFrameworkException;
import no.g9.message.*;

/**
 * Class used to get a dialog from the application where input data for the
 * dialog or search criteria intended to reduce the result set may be included.
 */
public class StartDialog {

    /**
     * Object which is used when instantiating the desired dialog.
     */
    private Object inputData;

    /**
     * OpCode is a local constant that may be utilized by the callee to
     * determine further processing when gaining focus.
     */
    private int opCode;

    /**
     * Helper class used for instantiating a dialog on EDT.
     */
    private static class DialogCreator implements Runnable {

        /** Reference to application. */
        ApplicationMethods application;

        /** Name of dialog. */
        String dialogName;

        /** Controller of created dialog. */
        G9DialogController dialogController;

        /**
         * Standard constructor.
         *
         * @param app Reference to application.
         * @param dialogName Name of dialog.
         */
        public DialogCreator(ApplicationMethods app, String dialogName) {
            this.application = app;
            this.dialogName = dialogName;
        }

        @Override
        public void run() {
            // Is the dialog already created?
            boolean isDialogCreated = application.getDialog(dialogName) != null;

            // Create new dialog or get reference to existing dialog.
            dialogController = application.getOldDialogOrCreateNew(dialogName);

            // Take action depending on the dialog type.
            boolean isDialogBox= dialogController.getWindow() instanceof JDialog;
            if (isDialogBox) {
                if (isDialogCreated) {
                    dialogController.toFront();
                }
                // else: getOldDialogOrCreateNew(..) has done everything
                //       needed (showDialog() and setVisible()).

            // Window is a JInternalFrame inside the MDI window.
            } else {
                JInternalFrame fView = dialogController.getView();

                // Show if not visible
                if (!fView.isVisible()) {
                    fView.setVisible(true);
                }

                // De-iconify if iconified
                if (fView.isIcon()) {
                    try {
                        fView.setIcon(false);
                    } catch (PropertyVetoException e) {
                        // could not de-iconify - not much to do about it...
                    }
                }

                // Move to front and select just in case..
                fView.moveToFront();
                try {
                    fView.setSelected(true);
                } catch (PropertyVetoException e) {
                    // could not select - not much to do about it...
                }

                // Let desktopPane know about the shown frame.
                application.getG9DesktopPane().frameShown(fView);

                fView.getFocusTraversalPolicy().getDefaultComponent(fView)
                                        .requestFocus();
            }
        }
    }

    /**
     * Gets inputData
     *
     * @return inputData
     */
    public Object getInputData() {
        return this.inputData;
    }

    /**
     * Sets inputData
     *
     * @param startData Data desirable to instantiate with
     */
    public void setInputData(Object startData) {
        this.inputData = startData;
    }

    /**
     * Gets the <code>opCode</code> for further processing in callee
     *
     * @return opCode
     */
    public int getOpCode() {
        return this.opCode;
    }

    /**
     * Sets the <code>opCode</code> for further processing in callee
     *
     * @param opCode (missing javadoc)
     */
    public void setOpCode(int opCode) {
        this.opCode = opCode;
    }


    /**
     * Instantiates a dialog, where the startInit operation is called on
     * instantiation. Note that this method can not be invoked on the EDT as it
     * needs to block on that thread.
     *
     * @param application Application-reference
     * @param name Name of the callee
     * @param returnObject ReturnToDialog to regain context in caller
     * @return True if OK
     * @throws G9ClientException if this method is invoked on the EDT.
     */
    public boolean instantiateDialog(final ApplicationMethods application,
                                     final String name,
                                     final ReturnToDialog returnObject) {

        if (SwingUtilities.isEventDispatchThread()) {
            throw new G9ClientException("Can't instantiate new dialog from EDT.");
        }

        // Create reference to dialog and open it.
        DialogCreator dialogCreator = new DialogCreator(application, name);
        String msgID = null;
        Exception ex = null;
        try {
            SwingUtilities.invokeAndWait(dialogCreator);
        } catch (InterruptedException e) {
            msgID = CRuntimeMsg.CT_INTERRUPTED;
            ex = e;
        } catch (InvocationTargetException e) {
            msgID = CRuntimeMsg.CT_INVOCATION_TARGET;
            if (e.getCause() != null && e.getCause() instanceof Exception) {
                ex = (Exception) e.getCause();
            } else {
                ex = e;
            }

        } finally {
            if (msgID != null) {
                Object[] msgArgs= { this.getClass(), "instantiateDialog", ex };
                Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9ClientFrameworkException(ex, msg);
            }
        }

        final G9DialogController dialogController = dialogCreator.dialogController;

        if (dialogController != null) {
            final StartDialog outer = this;

            G9Worker.enqueueTask(dialogController, new Runnable() {
                @Override
                public void run() {
                    ApplicationMethods app = dialogController.getApplication();
                    try {
                        dialogController.startInit(returnObject, outer);
                    } catch (Exception e) {
                        app.applicationExceptionHandler(e);
                    }
                }
            });
        }

        return dialogController != null;
    }

}