/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.support.message;

import java.awt.Component;
import java.lang.reflect.InvocationTargetException;
import java.util.Arrays;

import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import no.g9.client.support.ApplicationMethods;
import no.g9.client.support.G9DialogController;
import no.g9.message.DispatcherContext;
import no.g9.message.Message;
import no.g9.message.MessageInteractor;
import no.g9.message.MessageReply;
import no.g9.message.MessageReplyType;
import no.g9.message.MessageType;
import no.g9.message.MessageTypeEnum;
import no.g9.message.ReplySetType;

/**
 * Utility class for displaying a {@link no.g9.message.Message}
 */
public class ClientInteractor implements MessageInteractor {

    /** Parent application */
    private ApplicationMethods application;

    /**
     * Constructs a ClientInteractor object without an application
     */
    public ClientInteractor() {
        // Empty
    }

    /**
     * Constructs a ClientInteractor object
     *
     * @param application The parent application
     */
    public ClientInteractor(ApplicationMethods application) {
        setApplication(application);
    }

    /**
     * Interacts a message. The message reply is set to
     * MessageReplyType.REPLY_NONE if the dialog is closed
     *
     * @param msg The message to interact
     * @param context keeps the gui component that "owns" the message, e.g. the
     *            application frame, a document window or a dialog box.
     * @throws IllegalArgumentException if <code>msg</code> is null or
     *             <code>owner</code> isn't a Component.
     */
    @Override
    public MessageReply interact(final Message msg, final DispatcherContext context) {
        if (msg == null) {
            throw new IllegalArgumentException("Message cannot be a null pointer");
        }
        if (context == null) {
            throw new IllegalArgumentException("Context cannot be null");
        }
        ClientDispatcherContext clientCtx = (ClientDispatcherContext) context;
        if (clientCtx.getOwner() == null) {
            throw new IllegalArgumentException("Owner must be a Component");
        }

        Component owner = clientCtx.getOwner();
        setApplication(clientCtx.getApplication());
    	if (!SwingUtilities.isEventDispatchThread()) {
    		try {
    			SwingUtilities.invokeAndWait(new Runnable() {
    				@Override
                    public void run() {
    					interact(msg, context);
    				}
    			});
    		} catch (InterruptedException e) {
    			Thread.interrupted();
    		} catch (InvocationTargetException e) {
    			if (e.getCause() instanceof IllegalArgumentException) {
    				throw (IllegalArgumentException) e.getCause();
    			}
    			throw new RuntimeException(e.getCause());
    		}
    		return msg.getReply();
    	}

        MessageType msgType = msg.getMsgType();

        /* MSGTYPE_QQ action */
        if (msgType.equals(MessageTypeEnum.QUESTION)) {
            resolveDialogAction(msg, JOptionPane.QUESTION_MESSAGE,
                    MessageReplyType.REPLY_OK, owner);

            /* MSGTYPE_QWARN action */
        } else if (msgType.equals(MessageTypeEnum.QWARN)) {
            resolveDialogAction(msg, JOptionPane.WARNING_MESSAGE,
                    MessageReplyType.REPLY_OK, owner);

            /* MSGTYPE_INFO action */
        } else if (msgType.equals(MessageTypeEnum.INFO)) {
            resolveDialogAction(msg, JOptionPane.INFORMATION_MESSAGE,
                    MessageReplyType.REPLY_OK, ReplySetType.REPLSET_OK, owner);

            /* MSGTYPE_ERROR action */
        } else if (msgType.equals(MessageTypeEnum.ERROR)) {
            resolveDialogAction(msg, JOptionPane.ERROR_MESSAGE,
                    MessageReplyType.REPLY_OK, ReplySetType.REPLSET_OK, owner);

            /* MSGTYPE_STATUS action */
        } else if (msgType.equals(MessageTypeEnum.STATUS)) {
            showOnStatusBar(msg);

            /* MSGTYPE_PROGRESS action */
        } else if (msgType.equals(MessageTypeEnum.PROGRESS)) {
            showOnStatusBar(msg);

            /* Default action for other message types */
        }
        return msg.getReply();
    }

    /**
     * Displays the message text of
     * <code>msg<code> on the application statusbar.
     * The reply type of the message is set to MessageReplyType.REPLY_NONE
     *
     * @param msg The message to display
     */
    private void showOnStatusBar(Message msg) {
        application.getStatusBar().add(msg.getHint());
        msg.setReply(MessageReplyType.REPLY_NONE);
    }

    /**
     * Resolves which kind of dialog to create and display, and calls the
     * belonging "dialog action"-method. The choice is based on the reply set
     * given in <code>msg</code>. Reply and potential input is stored in
     * <code>msg</code>.
     *
     * @param msg The message to create a dialog for.
     * @param msgType the message type
     * @param standardOption the set of reply options
     * @param owner the dialog that "owns" the message
     */
    private void resolveDialogAction(Message msg, int msgType,
            MessageReplyType standardOption, Component owner) {

        if (msg.getValidReplies().equals(ReplySetType.REPLSET_NONE)) {
            return;
        } else if (msg.getValidReplies().equals(
                ReplySetType.REPLSET_ABORTRETRYIGNORE)
                || msg.getValidReplies().equals(ReplySetType.REPLSET_OK)
                || msg.getValidReplies().equals(
                        ReplySetType.REPLSET_OKCANCEL)
                || msg.getValidReplies().equals(
                        ReplySetType.REPLSET_OKINFO)
                || msg.getValidReplies()
                        .equals(ReplySetType.REPLSET_YESNO)
                || msg.getValidReplies().equals(
                        ReplySetType.REPLSET_YESNOCANCEL)) {
            showSimpleDialog(msg, msgType, standardOption, false, owner);
        } else if (msg.getValidReplies().equals(
                ReplySetType.REPLSET_STRING)) {
            showSimpleDialog(msg, msgType, standardOption, true, owner);
        } else if (msg.getValidReplies().equals(ReplySetType.REPLSET_DIR)
                || msg.getValidReplies().equals(
                        ReplySetType.REPLSET_DIR_EXISTS)
                || msg.getValidReplies().equals(ReplySetType.REPLSET_FILE)
                || msg.getValidReplies().equals(
                        ReplySetType.REPLSET_FILE_EXISTS)) {
            showFileChooserDialog(msg);
        }
    }

    /**
     * Same as
     * {@link #resolveDialogAction(Message, int, MessageReplyType, Component)},
     * except for the possibility to override the replySet (the dialog options)
     * of the message.
     *
     * @param msg the message to display
     * @param msgType the type of message
     * @param standardOption the type of reply
     * @param replySet the set of replies
     * @param owner the dialog "owning" the message
     */
    private void resolveDialogAction(Message msg, int msgType,
            MessageReplyType standardOption, ReplySetType replySet, Component owner) {
        ReplySetType originalReplySet = msg.getValidReplies();
        msg.setValidReplies(replySet);
        resolveDialogAction(msg, msgType, standardOption, owner);
        msg.setValidReplies(originalReplySet);
    }

    /**
     * Shows a simple dialog. Note that the application dialog title is used as
     * the simple dialog title if the message does not contain any title
     *
     * @param msg The message to display
     * @param msgType The dialog type
     * @param standardOption The standard option
     * @param wantsInput If true a string inputfield is displayed in the dialog.
     * @param owner the gui component that "owns" the message, e.g. the
     *            application frame, a document window or a dialog box.
     */
    private void showSimpleDialog(Message msg, int msgType,
            MessageReplyType standardOption, boolean wantsInput,
            Component owner) {
        MessageReplyType replySet[] = msg.getValidReplies().toReplySet();
        MessageReplyType defaultReply = (Arrays.asList(replySet)
                .contains(standardOption)) ? standardOption : replySet[0];

        JOptionPane pane = new JOptionPane(msg.getMessageText(),
                msgType, JOptionPane.DEFAULT_OPTION, null, replySet,
                defaultReply);

        JDialog dialog = null;

        if (owner != null) {
            dialog = pane.createDialog(owner, getTitle(msg));
        } else if (application != null){
            dialog = pane.createDialog(application.getApplicationWindow(), getTitle(msg));
        } else {
            dialog = pane.createDialog(null, getTitle(msg));
        }

        pane.setWantsInput(wantsInput);

        dialog.setVisible(true);
        dialog.dispose();

        /* Store reply in message allways */
        Object reply = pane.getValue();

        if (reply == null || !(reply instanceof MessageReplyType)) {
            msg.setReply(MessageReplyType.REPLY_NONE);
        } else {
            msg.setReply((MessageReplyType) reply);
        }

        /* Store inputstring in message, only if wantsInput */
        if (wantsInput) {
            Object replyString = pane.getInputValue();

            if (replyString != JOptionPane.UNINITIALIZED_VALUE) {
                msg.setReplString((String) replyString);
            }
        }
    }

    /**
     * Shows a filechooser dialog. Note that the application dialog title is
     * used as the simple dialog title if the message does not contain any title
     *
     * @param msg The message to display
     */
    private void showFileChooserDialog(Message msg) {

        JFileChooser fileChooser = new JFileChooser();

        fileChooser.setDialogTitle(getTitle(msg));

        // Choose form of JFileChooser based on the message valid replies
        if (msg.getValidReplies().equals(ReplySetType.REPLSET_DIR)) {
            fileChooser
                    .setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
            fileChooser.setDialogType(JFileChooser.SAVE_DIALOG);
        } else if (msg.getValidReplies().equals(
                ReplySetType.REPLSET_DIR_EXISTS)) {
            fileChooser
                    .setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
            fileChooser.setDialogType(JFileChooser.OPEN_DIALOG);
        } else if (msg.getValidReplies().equals(ReplySetType.REPLSET_FILE)) {
            fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
            fileChooser.setDialogType(JFileChooser.SAVE_DIALOG);
        } else if (msg.getValidReplies().equals(
                ReplySetType.REPLSET_FILE_EXISTS)) {
            fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
            fileChooser.setDialogType(JFileChooser.OPEN_DIALOG);
        } else {
            // Log and throw exception
        }

        // Collect and store user reply in message
        String buttonTextPrefix = "Save ";
        String buttonText = (msg.getTitle().equals("")) ? "Save"
                : buttonTextPrefix + msg.getTitle();
        int reply = fileChooser.showDialog(getParentComponent(),
                buttonText);

        if (reply == JFileChooser.ERROR_OPTION) {
            msg.setReply(MessageReplyType.REPLY_NONE);
        } else if (reply == JFileChooser.CANCEL_OPTION) {
            msg.setReply(MessageReplyType.REPLY_CANCEL);
        } else if (reply == JFileChooser.APPROVE_OPTION) {
            msg.setReply(MessageReplyType.REPLY_OK);
            msg.setReplString(fileChooser.getSelectedFile().toString());
            msg.setReplyFile(fileChooser.getSelectedFile());
        } else {
            // Log and throw exception
        }
    }

    /**
     * If the message title is nonempty the message title is returned, else the
     * application windows title is returned.
     *
     * @param msg The message to compute the title for
     * @return Returns a string (, never null);
     */
    private String getTitle(Message msg) {

        String title = msg.getTitle();
        if (title != null && title.trim().length() != 0) {
            return title;
        }
        if (application != null
                && application.getApplicationWindow() != null) {
            return application.getApplicationWindow().getTitle();
        }
        return "";
    }

    /**
     * Gets the parent component.
     *
     * @return Parent component
     */
    private Component getParentComponent() {
        if (application == null) {
            return null;
        }
        G9DialogController gdc = application.getActiveInternalWindow();
        Component activeWindow = gdc == null ? null : gdc.getWindow();

        return (activeWindow == null) ? application.getApplicationWindow()
                : activeWindow;
    }

    /**
     * @return Returns the application.
     */
    public ApplicationMethods getApplication() {
        return application;
    }

    /**
     * @param application The application to set.
     */
    public void setApplication(ApplicationMethods application) {
        this.application = application;
    }
}
