/*
 * Decompiled with CFR 0.152.
 */
package no.nav.common.kafka.consumer;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import no.nav.common.kafka.consumer.ConsumeStatus;
import no.nav.common.kafka.consumer.KafkaConsumerClientConfig;
import no.nav.common.kafka.consumer.TopicConsumer;
import no.nav.common.kafka.consumer.util.ConsumerUtils;
import org.apache.kafka.clients.consumer.ConsumerRebalanceListener;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.consumer.OffsetAndMetadata;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.WakeupException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KafkaConsumerClient<K, V>
implements ConsumerRebalanceListener {
    public static final long DEFAULT_POLL_DURATION_MS = 1000L;
    private static final long POLL_ERROR_TIMEOUT_MS = 5000L;
    private final Logger log = LoggerFactory.getLogger(KafkaConsumerClient.class);
    private final ExecutorService pollExecutor = Executors.newSingleThreadExecutor();
    private final Map<TopicPartition, OffsetAndMetadata> offsetsToCommit = new ConcurrentHashMap<TopicPartition, OffsetAndMetadata>();
    private final Set<TopicPartition> revokedOrFailedPartitions = ConcurrentHashMap.newKeySet();
    private final KafkaConsumerClientConfig<K, V> config;
    private volatile ClientState clientState = ClientState.NOT_RUNNING;
    private volatile CountDownLatch processedRecordsLatch;
    private volatile CountDownLatch shutdownLatch;
    private KafkaConsumer<K, V> consumer;

    public KafkaConsumerClient(KafkaConsumerClientConfig<K, V> config) {
        KafkaConsumerClient.validateConfig(config);
        this.config = config;
        Runtime.getRuntime().addShutdownHook(new Thread(this::stop));
    }

    public void start() {
        if (this.clientState == ClientState.RUNNING) {
            return;
        }
        this.clientState = ClientState.RUNNING;
        this.log.info("Starting kafka consumer client...");
        this.pollExecutor.submit(this::consumeTopics);
    }

    public void stop() {
        if (this.clientState != ClientState.RUNNING) {
            return;
        }
        this.clientState = ClientState.NOT_RUNNING;
        this.log.info("Stopping kafka consumer client...");
        try {
            this.consumer.wakeup();
            this.shutdownLatch.await(10L, TimeUnit.SECONDS);
            this.log.info("Kafka consumer client stopped");
        }
        catch (InterruptedException e) {
            this.log.error("Failed to stop kafka consumer client gracefully", (Throwable)e);
        }
    }

    public boolean isRunning() {
        return this.clientState == ClientState.RUNNING;
    }

    public void onPartitionsRevoked(Collection<TopicPartition> partitions) {
        this.log.info("Partitions has been revoked from consumer: " + Arrays.toString(partitions.toArray()));
        this.revokedOrFailedPartitions.addAll(partitions);
        try {
            this.commitCurrentOffsets();
        }
        catch (Exception e) {
            this.log.error("Failed to commit offsets when partitions were revoked: " + this.offsetsToCommit.toString(), (Throwable)e);
        }
    }

    public void onPartitionsAssigned(Collection<TopicPartition> partitions) {
        this.log.info("New partitions has been assigned to the consumer: " + Arrays.toString(partitions.toArray()));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private void consumeTopics() {
        ArrayList<String> topicNames = new ArrayList<String>(this.config.topics.keySet());
        HashMap<TopicPartition, ExecutorService> topicConsumptionExecutors = new HashMap<TopicPartition, ExecutorService>();
        try {
            this.shutdownLatch = new CountDownLatch(1);
            this.consumer = new KafkaConsumer(this.config.properties);
            this.consumer.subscribe(topicNames, (ConsumerRebalanceListener)this);
        }
        catch (Exception e) {
            this.log.error("Unexpected exception caught from main loop. Shutting down...", (Throwable)e);
            return;
        }
        while (this.clientState == ClientState.RUNNING) {
            ConsumerRecords records;
            this.revokedOrFailedPartitions.clear();
            this.offsetsToCommit.clear();
            try {
                records = this.consumer.poll(Duration.ofMillis(this.config.pollDurationMs));
            }
            catch (WakeupException e) {
                this.log.info("Polling was cancelled by wakeup(). Stopping kafka consumer client...");
                try {
                    topicConsumptionExecutors.forEach((topicPartition, executorService) -> executorService.shutdown());
                    this.commitCurrentOffsets();
                    this.consumer.close(Duration.ofSeconds(3L));
                    return;
                }
                catch (Exception e2) {
                    this.log.error("Failed to shutdown kafka consumer client properly", (Throwable)e2);
                    return;
                }
                finally {
                    this.shutdownLatch.countDown();
                    if (this.clientState == ClientState.RUNNING) {
                        this.log.warn("Unexpected failure while kafka consumer client was running. Restarting...");
                        this.pollExecutor.submit(this::consumeTopics);
                    }
                }
            }
            catch (Exception e) {
                this.log.error("Exception occurred during polling of records. Waiting before trying again", (Throwable)e);
                Thread.sleep(5000L);
                continue;
            }
            if (records.isEmpty()) continue;
            int totalRecords = records.count();
            this.processedRecordsLatch = new CountDownLatch(totalRecords);
            for (ConsumerRecord record : records) {
                String topic = record.topic();
                TopicPartition topicPartition2 = new TopicPartition(record.topic(), record.partition());
                TopicConsumer topicConsumer = this.config.topics.get(topic);
                ExecutorService executor = topicConsumptionExecutors.computeIfAbsent(topicPartition2, tp -> Executors.newSingleThreadExecutor());
                executor.submit(() -> {
                    try {
                        if (this.clientState == ClientState.NOT_RUNNING || this.revokedOrFailedPartitions.contains(topicPartition2)) {
                            return;
                        }
                        ConsumeStatus status = ConsumerUtils.safeConsume(topicConsumer, record);
                        if (status == ConsumeStatus.OK) {
                            OffsetAndMetadata offsetAndMetadata = new OffsetAndMetadata(record.offset() + 1L);
                            this.offsetsToCommit.put(topicPartition2, offsetAndMetadata);
                        } else {
                            this.revokedOrFailedPartitions.add(topicPartition2);
                        }
                    }
                    catch (Exception e) {
                        String msg = String.format("Unexpected error occurred while processing consumer record. topic=%s partition=%d offset=%d", topic, record.partition(), record.offset());
                        this.log.error(msg, (Throwable)e);
                    }
                    finally {
                        this.processedRecordsLatch.countDown();
                    }
                });
            }
            this.processedRecordsLatch.await();
            try {
                this.commitCurrentOffsets();
            }
            catch (Exception e) {
                this.log.error("Failed to commit offsets: " + this.offsetsToCommit.toString(), (Throwable)e);
            }
        }
        return;
        finally {
            try {
                topicConsumptionExecutors.forEach((topicPartition, executorService) -> executorService.shutdown());
                this.commitCurrentOffsets();
                this.consumer.close(Duration.ofSeconds(3L));
            }
            catch (Exception e) {
                this.log.error("Failed to shutdown kafka consumer client properly", (Throwable)e);
            }
            finally {
                this.shutdownLatch.countDown();
                if (this.clientState == ClientState.RUNNING) {
                    this.log.warn("Unexpected failure while kafka consumer client was running. Restarting...");
                    this.pollExecutor.submit(this::consumeTopics);
                }
            }
        }
    }

    private void commitCurrentOffsets() {
        if (!this.offsetsToCommit.isEmpty()) {
            this.consumer.commitSync(this.offsetsToCommit, Duration.ofSeconds(3L));
            this.log.info("Offsets committed: " + this.offsetsToCommit.toString());
            this.offsetsToCommit.clear();
        }
    }

    private static void validateConfig(KafkaConsumerClientConfig<?, ?> config) {
        if (config.topics.isEmpty()) {
            throw new IllegalArgumentException("\"topics\" must contain at least 1 topic");
        }
        if (config.pollDurationMs <= 0L) {
            throw new IllegalArgumentException("\"pollDurationMs\" must be larger than 0");
        }
        if (!Boolean.FALSE.equals(config.properties.get("enable.auto.commit"))) {
            throw new IllegalArgumentException("enable.auto.commit must be false!");
        }
    }

    private static enum ClientState {
        RUNNING,
        NOT_RUNNING;

    }
}

