/*
 * Decompiled with CFR 0.152.
 */
package no.nav.common.oidc.auth;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.proc.BadJOSEException;
import com.nimbusds.jwt.JWT;
import com.nimbusds.jwt.JWTParser;
import com.nimbusds.openid.connect.sdk.validators.BadJWTExceptions;
import java.io.IOException;
import java.text.ParseException;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import no.nav.common.auth.SsoToken;
import no.nav.common.auth.Subject;
import no.nav.common.auth.SubjectHandler;
import no.nav.common.oidc.TokenRefreshClient;
import no.nav.common.oidc.auth.OidcAuthenticator;
import no.nav.common.oidc.utils.CookieUtils;
import no.nav.common.oidc.utils.TokenUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OidcAuthenticationFilter
implements Filter {
    private static final Logger logger = LoggerFactory.getLogger(OidcAuthenticationFilter.class);
    private static final long CHECK_EXPIRES_WITHIN = 300000L;
    private final List<OidcAuthenticator> oidcAuthenticators;
    private final List<String> publicPaths;
    private final TokenRefreshClient tokenRefreshClient;
    private List<Pattern> publicPatterns;

    public OidcAuthenticationFilter(List<OidcAuthenticator> oidcAuthenticators, List<String> publicPaths) {
        this.oidcAuthenticators = oidcAuthenticators;
        this.publicPaths = publicPaths;
        this.tokenRefreshClient = new TokenRefreshClient();
    }

    public void init(FilterConfig filterConfig) {
        String contextPath = this.contextPath(filterConfig);
        this.publicPatterns = this.publicPaths.stream().map(path -> "^" + contextPath + path).map(Pattern::compile).collect(Collectors.toList());
        logger.info("initialized {} with public patterns: {}", (Object)OidcAuthenticationFilter.class.getName(), this.publicPatterns);
    }

    private String contextPath(FilterConfig filterConfig) {
        String contextPath = filterConfig.getServletContext().getContextPath();
        if (contextPath == null || contextPath.length() <= 1) {
            contextPath = "";
        }
        return contextPath;
    }

    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        HttpServletRequest httpServletRequest = (HttpServletRequest)request;
        HttpServletResponse httpServletResponse = (HttpServletResponse)response;
        if (this.isPublic(httpServletRequest)) {
            chain.doFilter(request, response);
            return;
        }
        for (OidcAuthenticator authenticator : this.oidcAuthenticators) {
            Optional<String> token = authenticator.tokenLocator.getIdToken(httpServletRequest);
            if (!token.isPresent()) continue;
            try {
                JWT jwtToken = JWTParser.parse((String)token.get());
                Optional<String> refreshedIdToken = this.refreshIdTokenIfNecessary(jwtToken, authenticator, httpServletRequest);
                if (refreshedIdToken.isPresent()) {
                    jwtToken = JWTParser.parse((String)refreshedIdToken.get());
                    String idTokenCookieName = authenticator.tokenLocator.getIdTokenCookieName();
                    this.addNewIdTokenCookie(idTokenCookieName, jwtToken, httpServletRequest, httpServletResponse);
                }
                authenticator.tokenValidator.validate(jwtToken);
                SsoToken ssoToken = SsoToken.oidcToken((String)jwtToken.getParsedString(), (Map)jwtToken.getJWTClaimsSet().getClaims());
                Subject subject = new Subject(TokenUtils.getUid(jwtToken, authenticator.identType), authenticator.identType, ssoToken);
                SubjectHandler.withSubject((Subject)subject, () -> chain.doFilter(request, response));
                return;
            }
            catch (JOSEException | BadJOSEException | ParseException exception) {
                if (exception == BadJWTExceptions.EXPIRED_EXCEPTION) {
                    logger.info("Token validation failed", exception);
                    continue;
                }
                logger.error("Token validation failed", exception);
            }
        }
        httpServletResponse.setStatus(401);
    }

    private void addNewIdTokenCookie(String idTokenCookieName, JWT jwtToken, HttpServletRequest request, HttpServletResponse response) throws ParseException {
        Date cookieExpiration = jwtToken.getJWTClaimsSet().getExpirationTime();
        Cookie newIdCookie = CookieUtils.createCookie(idTokenCookieName, jwtToken.getParsedString(), cookieExpiration, request);
        response.addCookie(newIdCookie);
    }

    private Optional<String> refreshIdTokenIfNecessary(JWT token, OidcAuthenticator authenticator, HttpServletRequest request) {
        Optional<Cookie> refreshCookie;
        boolean needsToBeRefreshed;
        boolean bl = needsToBeRefreshed = TokenUtils.hasMatchingIssuer(token, authenticator.tokenValidator.getIssuer()) && TokenUtils.expiresWithin(token, 300000L);
        if (needsToBeRefreshed && (refreshCookie = authenticator.tokenLocator.getRefreshTokenCookie(request)).isPresent() && authenticator.refreshUrl != null) {
            try {
                return Optional.of(this.tokenRefreshClient.refreshIdToken(authenticator.refreshUrl, refreshCookie.get().getValue()));
            }
            catch (Exception e) {
                logger.error("Unable to refresh id token", (Throwable)e);
                return Optional.empty();
            }
        }
        return Optional.empty();
    }

    public boolean isPublic(HttpServletRequest httpServletRequest) {
        return this.publicPatterns.stream().anyMatch(p -> p.matcher(httpServletRequest.getRequestURI()).matches());
    }

    public void destroy() {
    }
}

