/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

package com.genghis.tools.cluster.services

import io.github.oshai.kotlinlogging.KotlinLogging
import io.vertx.codegen.annotations.Fluent
import io.vertx.codegen.annotations.ProxyGen
import io.vertx.core.AbstractVerticle
import io.vertx.core.AsyncResult
import io.vertx.core.Future
import io.vertx.core.Handler
import io.vertx.core.Promise
import io.vertx.core.Vertx
import io.vertx.core.eventbus.DeliveryOptions
import io.vertx.core.eventbus.Message
import io.vertx.core.eventbus.MessageConsumer
import io.vertx.core.http.HttpClient
import io.vertx.core.json.Json
import io.vertx.core.json.JsonObject
import io.vertx.kotlin.core.eventbus.deliveryOptionsOf
import io.vertx.kotlin.core.json.jsonObjectOf
import io.vertx.kotlin.coroutines.CoroutineVerticle
import io.vertx.kotlin.coroutines.coAwait
import io.vertx.servicediscovery.Record
import io.vertx.servicediscovery.ServiceDiscovery
import io.vertx.servicediscovery.ServiceDiscoveryOptions
import io.vertx.servicediscovery.Status.DOWN
import io.vertx.servicediscovery.Status.OUT_OF_SERVICE
import io.vertx.servicediscovery.Status.UNKNOWN
import io.vertx.servicediscovery.Status.UP
import io.vertx.servicediscovery.types.EventBusService
import io.vertx.servicediscovery.types.EventBusService.createRecord
import io.vertx.servicediscovery.types.HttpEndpoint
import io.vertx.serviceproxy.ServiceBinder
import io.vertx.serviceproxy.ServiceException
import io.vertx.serviceproxy.ServiceInterceptor
import io.vertx.serviceproxy.ServiceProxyBuilder
import kotlinx.coroutines.async
import java.util.concurrent.ConcurrentHashMap
import java.util.function.Consumer
import kotlin.reflect.full.allSuperclasses

private val logger = KotlinLogging.logger { }
const val GENGHIS_SERVICE_ANNOUNCE_ADDRESS = "com.genghis.services.manager.announce"
const val GENGHIS_SERVICE_SERVICE_NAME = "genghis-service-manager-service-discovery"
const val GENGHIS_SERVICE_DEFAULT_TIMEOUT = 5

@Suppress("unused")
class GenghisServices private constructor() : CoroutineVerticle() {
    val consumers = mutableSetOf<MessageConsumer<*>>()
    val records = mutableSetOf<Record>()
    val proxyMap = mutableMapOf<Class<*>, Any>()
    val discovery: ServiceDiscovery by lazy {
        ServiceDiscovery.create(
            vertx,
            ServiceDiscoveryOptions()
                .setAnnounceAddress(GENGHIS_SERVICE_ANNOUNCE_ADDRESS)
                .setUsageAddress(GENGHIS_SERVICE_ANNOUNCE_ADDRESS)
                .setName(GENGHIS_SERVICE_SERVICE_NAME),
        )
    }

    override suspend fun stop() =
        runCatching {
            val unpublishRecords =
                async {
                    records
                        .map {
                            discovery.unpublish(it.registration)
                        }.forEach {
                            it.coAwait()
                        }
                }
            val unregisterServices =
                async {
                    consumers
                        .map {
                            it.unregister()
                        }.forEach {
                            it.coAwait()
                        }
                }

            unpublishRecords.await()
            unregisterServices.await()

            discovery.close()
        }.recoverCatching { throwable ->
            logger.warn(throwable) { "Failure in closing discovery, reattempting..." }

            runCatching {
                discovery.close()
            }.onFailure {
                logger.error(it) { "Final failure in closing discovery..." }
            }.getOrElse {}
        }.getOrElse {}

    @Suppress("UNCHECKED_CAST")
    suspend inline fun <reified T> publishService(
        service: T,
        superClass: Class<in T> =
            service!!::class
                .allSuperclasses
                .first { superKlazz ->
                    superKlazz.annotations.any { it is ProxyGen }
                }.java as Class<in T>,
        customName: String = superClass.simpleName,
        timeoutSeconds: Long = GENGHIS_SERVICE_DEFAULT_TIMEOUT.toLong(),
        debugInfo: Boolean = false,
        local: Boolean = false,
        interceptors: Sequence<ServiceInterceptor> = emptySequence(),
    ): Pair<MessageConsumer<JsonObject>, Record> {
        val binder =
            ServiceBinder(vertx)
                .setTimeoutSeconds(timeoutSeconds)
                .setAddress(customName)
                .setIncludeDebugInfo(debugInfo)

        interceptors.forEach { binder.addInterceptor(it) }

        val consumer =
            when {
                local -> binder.registerLocal(superClass, service)
                else -> binder.register(superClass, service)
            }

        consumer.completion().coAwait()

        consumers.add(consumer)

        val record = createRecord(customName, customName, superClass)

        val published = discovery.publish(record).coAwait()

        records.add(published)

        return consumer to published
    }

    suspend inline fun <reified T : Any, E> consumeService(
        klazz: Class<T> = T::class.java,
        customName: String = T::class.java.simpleName,
        token: String? = null,
        deliveryOptions: DeliveryOptions = deliveryOptionsOf(),
        action: T.() -> E,
    ): E? =
        runCatching {
            discovery.getRecord(jsonObjectOf("name" to customName)).coAwait()?.let {
                @Suppress("CascadeIf") // https://youtrack.jetbrains.com/issue/KT-47475
                if (it.status == UP) {
                    val proxy = proxyMap[klazz]
                    val service: T =
                        if (proxy == null) {
                            val builder =
                                ServiceProxyBuilder(vertx)
                                    .setAddress(customName)
                                    .setOptions(deliveryOptions)
                                    .setToken(token)
                            val service = builder.build(klazz)

                            proxyMap[klazz] = service

                            service
                        } else {
                            proxy as T
                        }

                    action(service)
                } else if (it.status == DOWN) {
                    throw ServiceException(503, "Service $customName is ${it.status.name}")
                } else if (it.status == OUT_OF_SERVICE) {
                    throw ServiceException(504, "Service $customName is ${it.status.name}")
                } else if (it.status == UNKNOWN) {
                    throw ServiceException(500, "Service $customName is ${it.status.name}")
                } else {
                    throw ServiceException(500, "Service $customName is ${it.status.name}")
                }
            } ?: throw ServiceException(404, "Not found from $customName")
        }.getOrElse {
            when {
                it is ServiceException && it.failureCode() == 404 -> null
                else -> throw it
            }
        }

    suspend fun unPublishService(
        service: MessageConsumer<JsonObject>,
        record: Record,
    ): GenghisServices {
        service.unregister().coAwait()
        discovery.unpublish(record.registration).coAwait()

        records.firstOrNull { it === record }?.let {
            records.remove(it)
        }
        consumers.firstOrNull { it === service }?.let {
            consumers.remove(it)
        }

        return this
    }

    companion object {
        private val INSTANCE_MAP = mutableMapOf<Vertx, GenghisServices>()

        @Suppress("unused")
        suspend fun CoroutineVerticle.genghisServices(): GenghisServices = vertx.genghisServices()

        suspend fun Vertx.genghisServices(): GenghisServices {
            INSTANCE_MAP[this]?.let { return it }

            val services = GenghisServices()

            this.deployVerticle(services).coAwait()

            INSTANCE_MAP[this] = services

            return services
        }
    }
}

/**
 * This class defines a wrapper for publishing and consuming service declaration interfaces, and HTTP records.
 *
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */
@Suppress("unused")
class ServiceManager {
    private var serviceDiscovery: ServiceDiscovery? = null
    private val registeredServices = ConcurrentHashMap<String, MessageConsumer<JsonObject>>()
    private val registeredRecords = ConcurrentHashMap<String, Record>()
    private val fetchedServices = ConcurrentHashMap<String, MutableSet<Any>>()

    private lateinit var vertx: Vertx
    private var serviceAnnounceConsumer: MessageConsumer<JsonObject>? = null

    private constructor() {
        throw IllegalArgumentException("Should never run!")
    }

    private constructor(vertx: Vertx) {
        this.vertx = vertx
        openDiscovery()
        startServiceManagerKillVerticle()
    }

    private fun startServiceManagerKillVerticle() {
        vertx.deployVerticle(KillVerticle())
    }

    private inner class KillVerticle : AbstractVerticle() {
        @Throws(Exception::class)
        override fun stop(stopFuture: Promise<Void>) {
            logger.info { "Destroying ServiceManager" }

            when {
                serviceDiscovery != null -> {
                    logger.info { "Unpublishing all records..." }

                    val unPublishFutures = ArrayList<Promise<*>>()

                    registeredRecords.forEach { (_, v) ->
                        val unpublish = Promise.promise<Boolean>()

                        serviceDiscovery!!
                            .unpublish(v.registration)
                            .onFailure {
                                logger.info(it) { "Failed Unpublish: " + v.name }

                                unpublish.fail(it)
                            }.onSuccess {
                                logger.info { "Unpublished: " + v.name }

                                unpublish.complete()
                            }

                        unPublishFutures.add(unpublish)
                    }

                    Future.all<Any>(unPublishFutures.map { it.future() }).andThen {
                        try {
                            registeredRecords.clear()

                            logger.info { "UnPublish complete, Unregistering all services..." }

                            registeredServices.forEach { (_, v) ->
                                ServiceBinder(vertx).setAddress(v.address()).unregister(v)

                                logger.info { "Unregistering " + v.address() }
                            }

                            registeredServices.clear()

                            logger.info { "Releasing all consumed service objects..." }

                            fetchedServices.values.forEach {
                                ServiceDiscovery.releaseServiceObject(serviceDiscovery!!, it)
                            }

                            fetchedServices.clear()

                            closeDiscovery {
                                serviceAnnounceConsumer = null

                                logger.info { "Discovery Closed!" }

                                instanceMap.remove(vertx)
                                stopFuture.tryComplete()

                                logger.info { "ServiceManager destroyed..." }
                            }
                        } finally {
                            instanceMap.remove(vertx)
                            stopFuture.tryComplete()

                            logger.info { "ServiceManager destroyed..." }
                        }
                    }
                }

                else -> {
                    logger.info { "Discovery is null..." }

                    instanceMap.remove(vertx)
                    stopFuture.tryComplete()
                }
            }
        }
    }

    private fun openDiscovery() {
        logger.debug { "Opening Discovery..." }

        if (serviceDiscovery == null) {
            serviceDiscovery =
                ServiceDiscovery.create(
                    vertx,
                    ServiceDiscoveryOptions()
                        .setAnnounceAddress(GENGHIS_SERVICE_ANNOUNCE_ADDRESS)
                        .setUsageAddress(GENGHIS_SERVICE_ANNOUNCE_ADDRESS)
                        .setName(GENGHIS_SERVICE_SERVICE_NAME),
                )

            logger.debug { "Setting Discovery message consumer..." }

            serviceAnnounceConsumer =
                vertx.eventBus().consumer(GENGHIS_SERVICE_ANNOUNCE_ADDRESS) {
                    this.handleServiceEvent(it)
                }
        }

        logger.debug { "Discovery ready..." }
    }

    private fun handleServiceEvent(serviceEvent: Message<JsonObject>) {
        val headers = serviceEvent.headers()
        val body = serviceEvent.body()

        logger.trace {
            "Service Event:\n" + Json.encodePrettily(serviceEvent) +
                "\nHeaders:\n" + Json.encodePrettily(headers) +
                "\nBody:\n" + Json.encodePrettily(body)
        }

        val name = body.getString("name")
        val status = body.getString("status")

        if (status != null && status == "DOWN") {
            logger.debug { "Removing downed service: $name" }

            fetchedServices.remove(name)
        }
    }

    private fun closeDiscovery(resultHandler: Handler<AsyncResult<Void>>) {
        if (serviceDiscovery != null) serviceDiscovery!!.close()
        serviceDiscovery = null

        logger.debug { "Unregistering Service Event Listener..." }

        if (serviceAnnounceConsumer != null) {
            serviceAnnounceConsumer!!
                .unregister()
                .onFailure {
                    resultHandler.handle(ServiceException.fail(500, "Unable to unregister Service Event Listener..."))
                }.onSuccess {
                    resultHandler.handle(Future.succeededFuture())
                }
        }
    }

    @Fluent
    fun publishApi(httpRecord: Record): ServiceManager =
        publishService(
            httpRecord,
            {
                registeredRecords[it.registration] = it
            },
            { this.handlePublishResult(it) },
        )

    @Fluent
    fun publishApi(
        httpRecord: Record,
        resultHandler: Handler<AsyncResult<Record>>,
    ): ServiceManager = publishService(httpRecord, { registeredRecords[it.registration] = it }, resultHandler)

    @Fluent
    fun unPublishApi(
        service: Record,
        resultHandler: Handler<AsyncResult<Void>>,
    ): ServiceManager {
        registeredRecords.remove(service.registration)
        serviceDiscovery!!.unpublish(service.registration).onComplete(resultHandler)
        val objects = fetchedServices[service.name]

        if (objects != null && objects.size > 0) {
            val iterator = objects.iterator()
            iterator.next()
            iterator.remove()
        }

        return this
    }

    @Fluent
    fun <T> publishService(
        type: Class<T>,
        service: T,
    ): ServiceManager {
        val serviceName = type.simpleName

        return publishService(
            createRecord(serviceName, type),
            {
                registeredServices[it.registration] =
                    ServiceBinder(vertx)
                        .setTimeoutSeconds(GENGHIS_SERVICE_DEFAULT_TIMEOUT.toLong())
                        .setAddress(serviceName)
                        .register(type, service)
            },
            { this.handlePublishResult(it) },
        )
    }

    @Fluent
    fun <T> publishService(
        type: Class<T>,
        customName: String,
        service: T,
    ): ServiceManager =
        publishService(
            createRecord(customName, type),
            {
                registeredServices[it.registration] =
                    ServiceBinder(vertx)
                        .setTimeoutSeconds(GENGHIS_SERVICE_DEFAULT_TIMEOUT.toLong())
                        .setAddress(customName)
                        .register(type, service)
            },
            { this.handlePublishResult(it) },
        )

    @Fluent
    fun <T> publishService(
        type: Class<T>,
        service: T,
        resultHandler: Handler<AsyncResult<Record>>,
    ): ServiceManager =
        publishService(
            createRecord(type),
            {
                registeredServices[it.registration] =
                    ServiceBinder(vertx)
                        .setTimeoutSeconds(GENGHIS_SERVICE_DEFAULT_TIMEOUT.toLong())
                        .setAddress(type.simpleName)
                        .register(type, service)
            },
            resultHandler,
        )

    @Fluent
    fun <T> publishService(
        type: Class<T>,
        customName: String,
        service: T,
        resultHandler: Handler<AsyncResult<Record>>,
    ): ServiceManager =
        publishService(
            createRecord(customName, type),
            {
                registeredServices[it.registration] =
                    ServiceBinder(vertx)
                        .setTimeoutSeconds(GENGHIS_SERVICE_DEFAULT_TIMEOUT.toLong())
                        .setAddress(customName)
                        .register(type, service)
            },
            resultHandler,
        )

    @Fluent
    fun <T> unPublishService(
        type: Class<T>,
        service: Record,
    ): ServiceManager {
        val serviceName = type.simpleName

        return unPublishService(serviceName, service)
    }

    @Fluent
    fun <T> unPublishService(
        type: Class<T>,
        service: Record,
        resultHandler: Handler<AsyncResult<Void>>,
    ): ServiceManager {
        val serviceName = type.simpleName

        return unPublishService(serviceName, service, resultHandler)
    }

    @Fluent
    @JvmOverloads
    fun unPublishService(
        serviceName: String,
        service: Record,
        resultHandler: Handler<AsyncResult<Void>> =
            Handler {
                logger.info { "Unpublish res: " + it.succeeded() }
            },
    ): ServiceManager {
        ServiceBinder(vertx)
            .setAddress(serviceName)
            .unregister(registeredServices[service.registration])

        serviceDiscovery!!.unpublish(service.registration).onComplete {
            if (it.succeeded()) {
                registeredServices.remove(service.registration)

                val objects = fetchedServices[service.name]

                if (objects != null && objects.size > 0) {
                    val iterator = objects.iterator()
                    iterator.next()
                    iterator.remove()
                }

                resultHandler.handle(Future.succeededFuture())
            } else {
                resultHandler.handle(ServiceException.fail(500, "Unable to unpublish Service..."))
            }
        }

        return this
    }

    @Fluent
    fun consumeApi(
        name: String,
        resultHandler: Handler<AsyncResult<HttpClient>>,
    ): ServiceManager = getApi(name, resultHandler)

    @Fluent
    fun <T> consumeService(
        type: Class<T>,
        resultHandler: Handler<AsyncResult<T>>,
    ): ServiceManager = consumeService(type, type.simpleName, resultHandler)

    @Fluent
    fun <T> consumeService(
        type: Class<T>,
        customName: String,
        resultHandler: Handler<AsyncResult<T>>,
    ): ServiceManager = getService(type, customName, resultHandler)

    private fun getApi(
        name: String,
        resultHandler: Handler<AsyncResult<HttpClient>>,
    ): ServiceManager {
        logger.debug { "Getting API: $name" }

        val existingServices = fetchedServices[name]

        when {
            existingServices != null && existingServices.size > 0 -> {
                logger.debug { "Returning fetched Api..." }

                val objects = ArrayList(existingServices)
                objects.shuffle()

                resultHandler.handle(Future.succeededFuture(objects[0] as HttpClient))
            }

            else ->
                HttpEndpoint
                    .getClient(
                        serviceDiscovery!!,
                        JsonObject().put("name", name),
                    ).onComplete {
                        when {
                            it.failed() -> {
                                logger.error { "Unable to fetch API..." }

                                resultHandler.handle(ServiceException.fail(404, "API not found..."))
                            }

                            else -> {
                                val client = it.result()
                                var objects: MutableSet<Any>? = fetchedServices[name]

                                if (objects == null) {
                                    fetchedServices[name] = HashSet()
                                    objects = fetchedServices[name]
                                }

                                if (!objects!!.contains(client)) {
                                    objects.add(client)
                                }

                                fetchedServices[name] = objects

                                resultHandler.handle(Future.succeededFuture(client))
                            }
                        }
                    }
        }

        return this
    }

    private fun <T> getService(
        type: Class<T>,
        resultHandler: Handler<AsyncResult<T>>,
    ): ServiceManager = getService(type, type.simpleName, resultHandler)

    private fun <T> getService(
        type: Class<T>,
        serviceName: String,
        resultHandler: Handler<AsyncResult<T>>,
    ): ServiceManager {
        logger.debug { "Getting service: $serviceName" }

        val existingServices = fetchedServices[serviceName]

        when {
            existingServices != null && existingServices.size > 0 -> {
                logger.debug { "Returning fetched Api..." }

                val objects = ArrayList(existingServices)
                objects.shuffle()

                @Suppress("UNCHECKED_CAST")
                resultHandler.handle(Future.succeededFuture(objects[0] as T))
            }

            else ->
                EventBusService
                    .getProxy(
                        serviceDiscovery!!,
                        type,
                    ).onComplete {
                        when {
                            it.failed() -> {
                                logger.error { "ERROR: Unable to get service for $serviceName" }

                                resultHandler.handle(
                                    ServiceException.fail(
                                        NOT_FOUND,
                                        "Unable to get service for " + serviceName + " : " + it.cause(),
                                    ),
                                )
                            }

                            else -> {
                                val service = it.result()
                                var objects: MutableSet<Any>? = fetchedServices[serviceName]

                                if (objects == null) {
                                    fetchedServices[serviceName] = mutableSetOf()
                                    objects = fetchedServices[serviceName]
                                }

                                if (!objects!!.contains<Any?>(service)) {
                                    objects.add(service as Any)
                                }

                                fetchedServices[serviceName] = objects

                                logger.debug { "Successful fetch of: " + service.toString() }

                                resultHandler.handle(Future.succeededFuture(service))
                            }
                        }
                    }
        }

        return this
    }

    private fun <T> createRecord(type: Class<T>): Record = createRecord(type.simpleName, type)

    private fun <T> createRecord(
        serviceName: String,
        type: Class<T>,
    ): Record = createRecord(serviceName, serviceName, type)

    private fun publishService(
        record: Record,
        recordLogic: Consumer<Record>,
        resultHandler: Handler<AsyncResult<Record>>,
    ): ServiceManager {
        serviceDiscovery!!.publish(record).onComplete { ar ->
            when {
                ar.failed() -> {
                    logger.error {
                        "ERROR: Failed publish of " +
                            record.name + " to " +
                            record.location.encodePrettily() + " with " +
                            record.type + " : " +
                            record.status
                    }

                    resultHandler.handle(ServiceException.fail(INTERNAL_ERROR, ar.cause().message))
                }

                else -> {
                    val publishedRecord = ar.result()
                    registeredRecords[publishedRecord.registration] = publishedRecord
                    recordLogic.accept(publishedRecord)

                    logger.debug {
                        "Successful publish of: " +
                            publishedRecord.name + " to " +
                            publishedRecord.location.encodePrettily() + " with " +
                            publishedRecord.type + " : " +
                            publishedRecord.status
                    }

                    resultHandler.handle(Future.succeededFuture(publishedRecord))
                }
            }
        }

        return this
    }

    private fun handlePublishResult(publishResult: AsyncResult<Record>) {
        when {
            publishResult.failed() ->
                when {
                    publishResult.cause() is ServiceException -> {
                        val serviceException = publishResult.cause() as ServiceException

                        logger.error {
                            "Unable to publish service: " +
                                serviceException.failureCode() + " : " +
                                serviceException.message
                        }
                    }

                    else -> logger.error { "Unable to publish service: " + publishResult.cause() }
                }

            else -> {
                val record = publishResult.result()

                logger.debug {
                    "Published Service Record: " +
                        record.name + " : " +
                        record.location + " : " +
                        record.type + " : " +
                        record.status
                }
            }
        }
    }

    companion object {
        private const val NOT_FOUND = 404
        private const val INTERNAL_ERROR = 500
        private val instanceMap = mutableMapOf<Vertx, ServiceManager>()

        private lateinit var singleton: ServiceManager

        fun getInstance(): ServiceManager = getInstance(Vertx.currentContext().owner())

        fun getInstance(vertx: Vertx): ServiceManager {
            val instance: ServiceManager? = instanceMap[vertx]

            if (instance != null) return instance

            singleton = ServiceManager(vertx)

            instanceMap[vertx] = singleton

            return singleton
        }

        fun handleResultFailed(t: Throwable) {
            when (t) {
                is ServiceException -> logger.error(t) { t.failureCode().toString() + " : " + t.message }
                else -> logger.error(t) { "${t.message}" }
            }
        }
    }
}
