/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

@file:Suppress("UnstableApiUsage")

package com.genghis.tools.fcm.server

import com.genghis.tools.cluster.apis.APIManager
import com.genghis.tools.fcm.server.FcmServer.Companion.GCM_DEVICE_GROUP_HTTP_ENDPOINT_COMPLETE
import com.genghis.tools.fcm.server.data.FcmDevice
import com.genghis.tools.repository.repository.redis.GenghisRedisClient
import com.google.common.net.MediaType.JSON_UTF_8
import io.github.oshai.kotlinlogging.KotlinLogging
import io.vertx.core.AsyncResult
import io.vertx.core.Future.failedFuture
import io.vertx.core.Handler
import io.vertx.core.Vertx
import io.vertx.core.http.HttpClientOptions
import io.vertx.core.http.HttpHeaders.AUTHORIZATION
import io.vertx.core.http.HttpHeaders.CONTENT_TYPE
import io.vertx.core.http.HttpMethod.POST
import io.vertx.core.json.Json
import io.vertx.core.json.Json.encodePrettily
import io.vertx.core.json.JsonObject
import io.vertx.kotlin.core.http.requestOptionsOf
import java.util.function.Consumer

private val logger = KotlinLogging.logger { }

/**
 * @author Anders Mikkelsen
 * @version 31.03.2016
 */
class DeviceGroupManager internal constructor(
    private val server: FcmServer,
    private val sender: MessageSender,
    private val redisClient: GenghisRedisClient,
    private val GCM_SENDER_ID: String,
    private val GCM_API_KEY: String,
) {
    suspend fun addDeviceToDeviceGroupForUser(
        device: FcmDevice,
        appPackageName: String,
        channelKeyName: String,
        fcmId: String,
    ) {
        addDeviceToDeviceGroup(device, channelKeyName) {
            when {
                it.failed() -> logger.error { "Could not add device to device group..." }
                else -> {
                    logger.info { "User updated, device added..." }

                    sender.replyWithSuccessfullDeviceRegistration(appPackageName, fcmId)

                    logger.info { "Sent message of correct device registration..." }
                }
            }
        }
    }

    private suspend fun addDeviceToDeviceGroup(
        device: FcmDevice,
        channelKeyName: String,
        resultHandler: Handler<AsyncResult<Boolean>>,
    ) {
//        redisClient.performRedisWithRetry {
//            when {
//                it.failed() -> resultHandler.handle(Future.succeededFuture(FALSE))
//                else ->
//                    it.result().hgetall(channelKeyName) { hGetAllResult ->
//                        when {
//                            hGetAllResult.failed() -> {
//                                logger.error { "Unable to get Channelmap..." }
//
//                                resultHandler.handle(failedFuture(hGetAllResult.cause()))
//                            }
//
//                            else -> {
//                                val jsonMap = Json.decodeValue(hGetAllResult.result().toString(), Map::class.java)
//                                val channelMap = mutableMapOf<String, String>()
//                                @Suppress("UNCHECKED_CAST")
//                                if (jsonMap != null) channelMap.putAll(jsonMap as Map<out String, String>)
//                                val notificationKeyName = device.notificationKeyName
//
//                                when (val key = channelMap[notificationKeyName]) {
//                                    null -> {
//                                        val creationJson =
//                                            Json.encode(
//                                                MessageSender.createDeviceGroupCreationJson(
//                                                    notificationKeyName,
//                                                    device.fcmId,
//                                                ),
//                                            )
//
//                                        logger.info { "Creation Json is: " + encodePrettily(creationJson) }
//
//                                        val url = GCM_DEVICE_GROUP_HTTP_ENDPOINT_COMPLETE
//
//                                        APIManager.performRequestWithCircuitBreaker(
//                                            resultHandler,
//                                            {
//                                                val opt =
//                                                    HttpClientOptions()
//                                                        .setSsl(true)
//
//                                                logger.info { "Creation for: $url" }
//
//                                                server.vertx.createHttpClient(opt).request(POST, url) { request ->
//                                                    val req = request.result()
//                                                    req.putHeader(AUTHORIZATION, "key=$GCM_API_KEY")
//                                                    req.putHeader(CONTENT_TYPE, JSON_UTF_8.toString())
//                                                    req.putHeader("project_id", GCM_SENDER_ID)
//                                                    req.end(creationJson)
//
//                                                    req
//                                                        .response { response ->
//                                                            val clientResponse = response.result()
//                                                            val status = clientResponse.statusCode()
//                                                            logger.info { "Create Group response: " + (status == 200) }
//
//                                                            when (status) {
//                                                                200 ->
//                                                                    clientResponse.bodyHandler { bodyBuffer ->
//                                                                        logger.info { "Device Group Created..." }
//
//                                                                        val body = bodyBuffer.toJsonObject()
//
//                                                                        logger.info { "Response from GCM: ${encodePrettily(body)}" }
//
//                                                                        val notificationKey = body.getString("notification_key")
//
//                                                                        it.complete(java.lang.Boolean.TRUE)
//
//                                                                        doDeviceGroupResult(
//                                                                            notificationKey,
//                                                                            channelMap,
//                                                                            device,
//                                                                            notificationKeyName,
//                                                                            channelKeyName,
//                                                                            resultHandler,
//                                                                        )
//                                                                    }
//
//                                                                else ->
//                                                                    clientResponse.bodyHandler { body ->
//                                                                        logger.error { clientResponse.statusMessage() }
//                                                                        logger.error { body.toString() }
//
//                                                                        logger.error {
//                                                                            "Could not create Device Group for " +
//                                                                                notificationKeyName + " with " + "id: " +
//                                                                                device.fcmId
//                                                                        }
//                                                                        logger.error { "Attempting adding..." }
//
//                                                                        it.fail(
//                                                                            UnknownError(
//                                                                                "Could not create Device Group for " +
//                                                                                    notificationKeyName + " with " + "id: " +
//                                                                                    device.fcmId,
//                                                                            ),
//                                                                        )
//
//                                                                        doDeviceGroupResult(
//                                                                            null,
//                                                                            channelMap,
//                                                                            device,
//                                                                            notificationKeyName,
//                                                                            channelKeyName,
//                                                                            resultHandler,
//                                                                        )
//                                                                    }
//                                                            }
//                                                        }.exceptionHandler { message ->
//                                                            logger.error { "HTTP Error: $message" }
//
//                                                            it.fail(message)
//                                                        }
//                                                }
//                                            },
//                                        ) { logger.error { "Failed DeviceGroupAdd: $it" } }
//                                    }
//
//                                    else -> addToGroup(device.fcmId, notificationKeyName, key, resultHandler)
//                                }
//                            }
//                        }
//                    }
//            }
//        }
    }

    private fun doDeviceGroupResult(
        notificationKey: String?,
        channelMap: MutableMap<String, String>,
        device: FcmDevice,
        notificationKeyName: String,
        channelKeyName: String,
        resultHandler: Handler<AsyncResult<Boolean>>,
    ) {
        logger.info { "New key for device group is: $notificationKey" }

        when (notificationKey) {
            null -> {
                val checkKey =
                    Consumer<String> {
                        when {
                            it != null -> Unit
//                                setNewKey(
//                                    device,
//                                    channelKeyName,
//                                    channelMap,
//                                    notificationKeyName,
//                                    it,
//                                    resultHandler,
//                                )

                            else -> resultHandler.handle(failedFuture(IllegalArgumentException("Could not fetch key...")))
                        }
                    }

                val httpResultHandler =
                    Handler<AsyncResult<String>> {
                        when {
                            it.succeeded() ->
                                when {
                                    it.result() != null -> logger.info { "Completed Fetch key..." }
                                    else -> logger.error { "Failed Fetch key..." }
                                }

                            else -> logger.error { "Failed Fetch key..." }
                        }

                        checkKey.accept(it.result())
                    }

                val url = "$GCM_DEVICE_GROUP_HTTP_ENDPOINT_COMPLETE?notification_key_name=$notificationKeyName"

                APIManager.performRequestWithCircuitBreaker(
                    httpResultHandler,
                    {
                        val options =
                            HttpClientOptions()
                                .setSsl(true)

                        logger.info { "Querying: $url" }

                        Vertx
                            .currentContext()
                            .owner()
                            .createHttpClient(options)
                            .request(
                                requestOptionsOf(
                                    absoluteURI = url,
                                ),
                            ).onComplete { request ->
                                val req = request.result()
                                req.putHeader(AUTHORIZATION, "key=$GCM_API_KEY")
                                req.putHeader(CONTENT_TYPE, JSON_UTF_8.toString())
                                req.putHeader("project_id", GCM_SENDER_ID)
                                req.end()

                                req
                                    .response()
                                    .onComplete { response ->
                                        val res = response.result()
                                        val status = res.statusCode()
                                        logger.info { "Fetch Notification key response: " + (status == 200) }

                                        when {
                                            status != 200 -> {
                                                res.bodyHandler { body ->
                                                    logger.error { res.statusMessage() }
                                                    logger.error { body.toString() }
                                                }

                                                it.fail(UnknownError(res.statusMessage()))
                                            }

                                            else ->
                                                res.bodyHandler { body ->
                                                    val bodyObject = body.toJsonObject()

                                                    logger.info { "Response from GCM: " + encodePrettily(bodyObject) }

                                                    it.complete(bodyObject.getString("notification_key"))
                                                }
                                        }
                                    }
                            }
                    },
                ) { logger.error { "HttpFetchFailed: $it" } }
            }

            else -> Unit
//                setNewKey(
//                    device,
//                    channelKeyName,
//                    channelMap,
//                    notificationKeyName,
//                    notificationKey,
//                    resultHandler,
//                )
        }
    }

    private suspend fun setNewKey(
        device: FcmDevice,
        channelKeyName: String,
        channelMap: MutableMap<String, String>,
        notificationKeyName: String,
        newNotificationKey: String?,
        resultHandler: Handler<AsyncResult<Boolean>>,
    ) {
        if (newNotificationKey != null) channelMap[notificationKeyName] = newNotificationKey

        val mapAsJson = JsonObject(Json.encode(channelMap))

//        redisClient.performRedisWithRetry { redisAPIAsyncResult ->
//            when {
//                redisAPIAsyncResult.failed() -> resultHandler.handle(Future.succeededFuture(FALSE))
//                else ->
//                    redisAPIAsyncResult.result().hmset(listOf(channelKeyName, mapAsJson.encode())) {
//                        when {
//                            it.failed() -> logger.error { "Failed to set hm for device group..." }
//                            else -> addToGroup(device.fcmId, notificationKeyName, newNotificationKey, resultHandler)
//                        }
//                    }
//            }
//        }
    }

    private fun addToGroup(
        fcmId: String,
        keyName: String,
        key: String?,
        resultHandler: Handler<AsyncResult<Boolean>>,
    ) {
        val addJson = Json.encode(MessageSender.createAddDeviceGroupJson(fcmId, keyName, key))
        val url = GCM_DEVICE_GROUP_HTTP_ENDPOINT_COMPLETE

        APIManager.performRequestWithCircuitBreaker(
            resultHandler,
            {
                val opts =
                    HttpClientOptions()
                        .setSsl(true)

                server.vertx.createHttpClient(opts).request(POST, url).onComplete { request ->
                    val req = request.result()
                    req.putHeader(AUTHORIZATION, "key=$GCM_API_KEY")
                    req.putHeader(CONTENT_TYPE, JSON_UTF_8.toString())
                    req.putHeader("project_id", GCM_SENDER_ID)
                    req.end(addJson)

                    req
                        .response()
                        .onComplete { response ->
                            val clientResponse = response.result()
                            val status = clientResponse.statusCode()
                            logger.info { "Add To Group response: " + (status == 200) }

                            if (status != 200) {
                                clientResponse.bodyHandler { body ->
                                    logger.error { clientResponse.statusMessage() }
                                    logger.error { body.toString() }
                                }
                            }

                            it.complete(status == 200)
                        }
                }
            },
        ) {
            logger.error { "Failed Add to Group..." }

            resultHandler.handle(failedFuture(IllegalArgumentException()))
        }
    }
}
