/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

@file:Suppress("PrivatePropertyName")

package com.genghis.tools.fcm.server

import com.genghis.tools.fcm.server.XMPPPacketListener.Companion.GCM_ELEMENT_NAME
import com.genghis.tools.fcm.server.XMPPPacketListener.Companion.GCM_NAMESPACE
import com.genghis.tools.fcm.server.data.DataMessageHandler
import com.genghis.tools.fcm.server.data.DefaultDataMessageHandlerImpl
import com.genghis.tools.fcm.server.data.DefaultRegistrationServiceImpl
import com.genghis.tools.fcm.server.data.RegistrationService
import com.genghis.tools.fcm.server.messageutils.FcmNotification
import com.genghis.tools.fcm.server.messageutils.FcmPacketExtension
import com.genghis.tools.repository.repository.redis.GenghisRedisClient
import com.genghis.tools.repository.repository.redis.RedisUtils
import io.github.oshai.kotlinlogging.KotlinLogging
import io.vertx.codegen.annotations.Fluent
import io.vertx.core.json.JsonObject
import io.vertx.kotlin.coroutines.CoroutineVerticle
import io.vertx.kotlin.coroutines.dispatcher
import kotlinx.coroutines.Dispatchers.IO
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.withContext
import org.jivesoftware.smack.Connection
import org.jivesoftware.smack.ConnectionConfiguration
import org.jivesoftware.smack.ConnectionListener
import org.jivesoftware.smack.XMPPConnection
import org.jivesoftware.smack.XMPPException
import org.jivesoftware.smack.filter.PacketTypeFilter
import org.jivesoftware.smack.packet.Message
import org.jivesoftware.smack.provider.PacketExtensionProvider
import org.jivesoftware.smack.provider.ProviderManager
import javax.net.ssl.SSLSocketFactory

private val logger = KotlinLogging.logger { }

/**
 * @author Anders Mikkelsen
 * @version 31.03.2016
 */
class FcmServer private constructor(
    dev: Boolean,
) : CoroutineVerticle() {
    private var packageNameBase: String? = null
    private var gcmSenderId: String? = null
    private var gcmApiKey: String? = null
    private val gcmPort: Int = if (dev) 5236 else 5235
    private var messageSender: MessageSender? = null
    private var dataMessageHandler: DataMessageHandler? = null
    private var registrationService: RegistrationService? = null

    private var redisClient: GenghisRedisClient? = null

    private var connectionConfiguration: ConnectionConfiguration? = null

    private var primaryConnection: Connection? = null
    private var secondaryConnection: Connection? = null

    private var primaryIsDraining: Boolean = false
    private var primaryConnecting: Boolean = false
    private var secondaryConnecting: Boolean = false

    internal val sendingConnection: Connection?
        get() {
            logger.info {
                "GCM " +
                    (if (primaryIsDraining) "is" else "is not") +
                    " draining primary connection" +
                    if (primaryIsDraining) "!" else "..."
            }

            return when {
                primaryIsDraining -> secondaryConnection
                else -> {
                    if (primaryConnection != null &&
                        primaryConnection!!.isConnected &&
                        secondaryConnection != null &&
                        secondaryConnection!!.isConnected
                    ) {
                        primaryIsDraining = false
                        secondaryConnection!!.disconnect()
                    }

                    primaryConnection
                }
            }
        }

    val isOnline: Boolean
        get() =
            when {
                primaryConnection != null -> primaryConnection!!.isConnected && primaryConnection!!.isAuthenticated
                else ->
                    secondaryConnection != null &&
                        secondaryConnection!!.isConnected &&
                        secondaryConnection!!.isAuthenticated
            }

    class FcmServerBuilder {
        private var dev = false
        private var dataMessageHandler: DataMessageHandler? = null
        private var registrationService: RegistrationService? = null

        @Fluent
        fun withDev(dev: Boolean): FcmServerBuilder {
            this.dev = dev
            return this
        }

        @Fluent
        fun withDataMessageHandler(dataMessageHandler: DataMessageHandler): FcmServerBuilder {
            this.dataMessageHandler = dataMessageHandler

            return this
        }

        @Fluent
        fun withRegistrationService(registrationService: RegistrationService): FcmServerBuilder {
            this.registrationService = registrationService

            return this
        }

        fun build(): FcmServer {
            val fcmServer = FcmServer(dev)
            val messageSender = MessageSender(fcmServer)
            if (dataMessageHandler == null) dataMessageHandler = DefaultDataMessageHandlerImpl()
            if (registrationService == null) registrationService = DefaultRegistrationServiceImpl()
            dataMessageHandler!!.setServer(fcmServer)
            registrationService!!.setServer(fcmServer)

            dataMessageHandler!!.setSender(messageSender)
            registrationService!!.setSender(messageSender)

            fcmServer.setDataMessageHandler(dataMessageHandler)
            fcmServer.setRegistrationService(registrationService)
            fcmServer.setMessageSender(messageSender)

            return fcmServer
        }
    }

    private fun setDataMessageHandler(dataMessageHandler: DataMessageHandler?) {
        this.dataMessageHandler = dataMessageHandler
    }

    private fun setRegistrationService(registrationService: RegistrationService?) {
        this.registrationService = registrationService
    }

    private fun setMessageSender(messageSender: MessageSender) {
        this.messageSender = messageSender
    }

    @Throws(Exception::class)
    override suspend fun start() =
        withContext(IO) {
            logger.info { "Starting GCM Server: $this" }

            packageNameBase = config.getString("basePackageNameFcm")
            gcmSenderId = config.getString("gcmSenderId")
            gcmApiKey = config.getString("gcmApiKey")

            val errors = JsonObject()

            if (packageNameBase == null) errors.put("packageNameBase_error", "Cannot be null!")
            if (gcmSenderId == null) errors.put("gcmSenderId_error", "Cannot be null!")
            if (gcmApiKey == null) errors.put("gcmApiKey_error", "Cannot be null!")

            when {
                errors.isEmpty -> {
                    connectionConfiguration = ConnectionConfiguration(GCM_ENDPOINT, gcmPort)
                    redisClient = runBlocking(vertx.dispatcher()) { RedisUtils.getRedisClient(vertx, config) }
                    this@FcmServer.messageSender?.setRedisClient(redisClient!!)
                    setConfiguration()

                    try {
                        if (primaryConnection == null || !primaryConnection!!.isConnected) {
                            primaryConnection = connect()
                            addPacketListener(primaryConnection!!)
                            auth(primaryConnection!!)

                            logger.info { "GCM Connection established..." }
                        }
                    } catch (e: XMPPException) {
                        logger.error(e) { "GCM Connection could not be established!" }

                        throw e
                    }
                }

                else -> throw IllegalStateException(errors.encodePrettily())
            }
        }

    @Throws(Exception::class)
    override suspend fun stop() =
        withContext(IO) {
            logger.info { "Shutting down GCM Server: $this..." }

            primaryConnection!!.disconnect()

            if (secondaryConnection != null && secondaryConnection!!.isConnected) {
                secondaryConnection!!.disconnect()
            }
        }

    fun checkForDeadConnections() {
        when {
            primaryConnection != null &&
                !primaryConnection!!.isConnected &&
                secondaryConnection != null &&
                secondaryConnection!!.isConnected -> {
                try {
                    logger.info { "Draining on primary resolved, reconnecting..." }

                    primaryConnecting = true
                    primaryIsDraining = false
                    primaryConnection = connect()
                    addPacketListener(primaryConnection!!)
                    auth(primaryConnection!!)

                    logger.info { "Disconnecting secondary..." }
                    secondaryConnection!!.disconnect()
                } catch (e: XMPPException) {
                    logger.error { "GCM Connection could not be established!" }
                }

                primaryConnecting = false
            }

            primaryConnection != null && primaryConnection!!.isConnected ->
                logger.debug { "Primary: " + primaryConnection!!.isConnected + ", Sec is null" }

            primaryConnection != null &&
                !primaryConnection!!.isConnected &&
                secondaryConnection != null &&
                !secondaryConnection!!.isConnected ||
                primaryConnection == null &&
                secondaryConnection == null ->
                when {
                    primaryConnecting || secondaryConnecting ->
                        logger.info { "${if (!primaryConnecting) "Secondary" else "Primary"} already attempting connection..." }

                    else -> {
                        logger.info { "No connection, reconnecting..." }

                        try {
                            primaryConnecting = true
                            primaryIsDraining = false
                            primaryConnection = connect()
                            addPacketListener(primaryConnection!!)
                            auth(primaryConnection!!)
                        } catch (e: XMPPException) {
                            e.printStackTrace()

                            logger.error { "GCM Connection could not be established!" }
                        }

                        primaryConnecting = false
                    }
                }

            else -> logger.error { "UNKNOWN STATE: $primaryConnection : $secondaryConnection" }
        }
    }

    fun sendNotification(
        to: String,
        notification: FcmNotification,
    ): Boolean {
        val packageNameExtension = notification.packageNameExtension
        val appPackageName =
            if (packageNameExtension == "devApp") {
                packageNameBase
            } else {
                "$packageNameBase.$packageNameExtension"
            }

        messageSender!!.send(MessageSender.createCustomNotification(appPackageName, to, notification))

        logger.info { "Passing notification to: $to" }

        return true
    }

    private fun setConfiguration() {
        connectionConfiguration!!.isReconnectionAllowed = true
        connectionConfiguration!!.isRosterLoadedAtLogin = false
        connectionConfiguration!!.setSendPresence(false)
        connectionConfiguration!!.socketFactory = SSLSocketFactory.getDefault()

        ProviderManager.getInstance().addExtensionProvider(
            GCM_ELEMENT_NAME,
            GCM_NAMESPACE,
            PacketExtensionProvider {
                val json = it.nextText()
                FcmPacketExtension(json)
            },
        )
    }

    @Throws(XMPPException::class)
    private fun connect(): Connection {
        logger.info { "Connecting to GCM..." }

        val connection = XMPPConnection(connectionConfiguration)
        connection.connect()

        logger.info { "Adding connectionlistener..." }

        connection.addConnectionListener(
            object : ConnectionListener {
                override fun reconnectionSuccessful() {
                    logger.info { "Reconnected!" }
                }

                override fun reconnectionFailed(e: Exception) {
                    logger.info { "Reconnection failed: $e" }
                }

                override fun reconnectingIn(seconds: Int) {
                    logger.info { String.format("Reconnecting in %d secs", seconds) }
                }

                override fun connectionClosedOnError(e: Exception) {
                    logger.info { "Connection closed on error: $e" }

                    if (!primaryConnection!!.isConnected) {
                        primaryIsDraining = false
                    }
                }

                override fun connectionClosed() {
                    logger.info { "Connection closed" }
                }
            },
        )

        return connection
    }

    @Throws(XMPPException::class)
    private fun auth(connection: Connection) {
        logger.info { "Authenticating to GCM..." }

        connection.login(gcmSenderId!! + "@gcm.googleapis.com", gcmApiKey)
    }

    private fun addPacketListener(connection: Connection) {
        logger.info { "Adding packetlistener and packetinterceptor..." }

        connection.addPacketListener(
            XMPPPacketListener(
                this,
                redisClient!!,
                dataMessageHandler,
                registrationService,
                gcmSenderId,
                gcmApiKey,
            ),
            PacketTypeFilter(Message::class.java),
        )

        connection.addPacketInterceptor(
            {
                logger.info { "Sent: " + it.toXML().replace("&quot;".toRegex(), "'") }
            },
            PacketTypeFilter(Message::class.java),
        )
    }

    fun setDraining() {
        try {
            if (secondaryConnection == null || !secondaryConnection!!.isConnected) {
                secondaryConnecting = true

                for (i in 0..9) {
                    secondaryConnection = connect()
                    addPacketListener(secondaryConnection!!)
                    auth(secondaryConnection!!)

                    if (secondaryConnection!!.isConnected) {
                        break
                    } else {
                        secondaryConnection!!.disconnect()
                    }
                }

                secondaryConnecting = false
            }

            primaryIsDraining = true
        } catch (e: XMPPException) {
            logger.error { "Could not connect secondary on draining!" }
        }
    }

    companion object {
        private const val GCM_ENDPOINT = "fcm-xmpp.googleapis.com"
        const val GCM_HTTP_ENDPOINT = "https://fcm.googleapis.com/fcm/send"
        private const val GCM_DEVICE_GROUP_BASE = "android.googleapis.com"
        private const val GCM_DEVICE_GROUP_HTTP_ENDPOINT = "/gcm/notification"
        const val GCM_DEVICE_GROUP_HTTP_ENDPOINT_COMPLETE =
            "https://$GCM_DEVICE_GROUP_BASE$GCM_DEVICE_GROUP_HTTP_ENDPOINT"
    }
}
