/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

package com.genghis.tools.repository.dynamodb.operators

import com.genghis.tools.repository.dynamodb.DynamoDBRepository
import com.genghis.tools.repository.dynamodb.hash
import com.genghis.tools.repository.dynamodb.range
import com.genghis.tools.repository.models.Cacheable
import com.genghis.tools.repository.models.DynamoDBModel
import com.genghis.tools.repository.models.ETagable
import com.genghis.tools.repository.models.Model
import com.genghis.tools.repository.repository.cache.CacheManager
import com.genghis.tools.repository.repository.etag.ETagManager
import io.github.oshai.kotlinlogging.KotlinLogging
import io.vertx.core.Future.failedFuture
import io.vertx.core.Future.succeededFuture
import io.vertx.core.json.Json.encodePrettily
import io.vertx.core.json.JsonObject
import io.vertx.kotlin.core.json.jsonObjectOf
import io.vertx.kotlin.coroutines.awaitResult
import kotlinx.coroutines.async
import kotlinx.coroutines.awaitAll
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.delay
import software.amazon.awssdk.enhanced.dynamodb.Expression
import software.amazon.awssdk.enhanced.dynamodb.model.TransactPutItemEnhancedRequest
import software.amazon.awssdk.enhanced.dynamodb.model.TransactUpdateItemEnhancedRequest
import software.amazon.awssdk.enhanced.dynamodb.model.TransactWriteItemsEnhancedRequest
import software.amazon.awssdk.services.dynamodb.model.AttributeValue
import software.amazon.awssdk.services.dynamodb.model.TransactionCanceledException
import java.lang.reflect.Method
import java.time.Instant
import java.util.concurrent.CompletionException
import java.util.function.Function

private val logger = KotlinLogging.logger { }

class ItemExistsException : Throwable()

class ItemDoesNotExistException : Throwable()

/**
 * This class defines the creation operations for the DynamoDBRepository.
 *
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */
class DynamoDBCreator<E>(
    private val TYPE: Class<E>,
    private val db: DynamoDBRepository<E>,
    private val HASH_IDENTIFIER: String,
    private val HASH_FIELD: Method?,
    private val IDENTIFIER: String?,
    private val IDENTIFIER_FIELD: Method?,
    private val cacheManager: CacheManager<E>?,
    private val eTagManager: ETagManager<E>?,
) where E : DynamoDBModel, E : Model, E : ETagable, E : Cacheable {
    private val shortCacheIdSupplier: Function<E, String> =
        Function { e ->
            val hash = e.hash(HASH_FIELD)

            TYPE.simpleName + "_" + hash
        }
    private val cacheIdSupplier: Function<E, String> =
        Function { e ->
            val hash = e.hash(HASH_FIELD)
            val range = e.range(IDENTIFIER_FIELD)

            TYPE.simpleName + "_" + hash + if (range == null || range == "") "" else "/$range"
        }

    suspend fun doWrite(
        create: Boolean,
        writeMap: Map<E, suspend E.(E) -> Unit>,
        attempt: Int? = null,
    ): List<E> {
        val hashValue =
            writeMap.entries
                .first()
                .key
                .hash(HASH_FIELD)!!

        require(!db.hasRangeKey() || writeMap.all { it.key.hash(HASH_FIELD)!! == hashValue }) {
            "All hashes must match"
        }

        val transactionBuilder =
            TransactWriteItemsEnhancedRequest
                .builder()
        val existingRecords =
            when {
                !create ->
                    db
                        .index {
                            noCache(true)
                            autoPaginate(true)
                            hashAndRange {
                                hash = if (db.hasRangeKey()) hashValue else null
                                ranges =
                                    writeMap
                                        .map { (record, _) ->
                                            when {
                                                db.hasRangeKey() -> record.range(IDENTIFIER_FIELD)!!
                                                else -> record.hash(HASH_FIELD)!!
                                            }
                                        }.toSet()
                            }
                        }.itemList.items
                        .associateBy {
                            when {
                                db.hasRangeKey() -> "${it.hash(HASH_FIELD)}:${it.range(IDENTIFIER_FIELD)}"
                                else -> it.hash(HASH_FIELD)!!
                            }
                        }
                else -> emptyMap()
            }

        val result =
            runCatching<Void> {
                writeMap.forEach { (record, updateLogic) ->
                    when {
                        !create -> {
                            if (logger.isDebugEnabled()) {
                                logger.debug { "Running remoteUpdate..." }
                            }

                            val key =
                                when {
                                    db.hasRangeKey() ->
                                        "${record.hash(HASH_FIELD)}:${record.range(IDENTIFIER_FIELD)}"
                                    else -> record.hash(HASH_FIELD)!!
                                }
                            val current = existingRecords[key]

                            current?.let {
                                this.optimisticLockingSave(
                                    transaction = transactionBuilder,
                                    newerVersion = it,
                                    updateLogic = updateLogic,
                                    record = record,
                                )
                            } ?: throw NoSuchElementException()
                        }

                        else -> {
                            if (logger.isDebugEnabled()) {
                                logger.debug { "Running remoteCreate..." }
                            }

                            record.createdAt = Instant.now().toEpochMilli()
                            val validations = record.validateCreate()

                            if (validations.isNotEmpty()) {
                                throw IllegalArgumentException(encodePrettily(validations))
                            }

                            transactionBuilder.addPutItem(
                                db.dynamoDbTable,
                                TransactPutItemEnhancedRequest
                                    .builder(TYPE)
                                    .item(record)
                                    .conditionExpression(buildExistingExpression(record, false))
                                    .build(),
                            )
                        }
                    }
                }

                val transaction = transactionBuilder.build()

                awaitResult { handler ->
                    db.dynamoDbMapper.transactWriteItems(transaction).whenCompleteAsync { t, u ->
                        when (u) {
                            null -> handler.handle(succeededFuture(t))
                            else -> handler.handle(failedFuture(u))
                        }
                    }
                }
            }

        return when {
            result.isSuccess -> {
                val items =
                    db
                        .index {
                            noCache(true)
                            autoPaginate(true)
                            hashAndRange {
                                hash = if (db.hasRangeKey()) hashValue else null
                                ranges =
                                    writeMap
                                        .map { (record, _) ->
                                            when {
                                                db.hasRangeKey() -> record.range(IDENTIFIER_FIELD)!!
                                                else -> record.hash(HASH_FIELD)!!
                                            }
                                        }.toSet()
                            }
                        }.itemList.items

                runCatching {
                    cacheManager?.replaceCache(
                        records = items,
                        shortCacheIdSupplier = shortCacheIdSupplier,
                        cacheIdSupplier = cacheIdSupplier,
                    )
                }.getOrThrow()

                destroyEtagsAfterCachePurge(items.first())

                val singles =
                    items
                        .associate {
                            it.generateEtagKeyIdentifier() to it.etag
                        }

                eTagManager?.setSingleRecordEtag(singles)

                items
            }

            attempt != null && attempt > 100 -> throw IllegalArgumentException(
                "Over 100 attempt limit",
                result.exceptionOrNull(),
            )

            else -> {
                val exception = result.exceptionOrNull()

                when {
                    exception is CompletionException && exception.cause is TransactionCanceledException ->
                        when {
                            create -> throw ItemExistsException()
                            else -> throw ItemDoesNotExistException()
                        }

                    exception is NoSuchElementException ->
                        when {
                            create -> throw exception
                            else -> throw ItemDoesNotExistException()
                        }

                    else -> {
                        delay(attempt?.toLong() ?: 1L)

                        doWrite(
                            create = create,
                            writeMap = writeMap,
                            attempt = attempt?.let { it + 1 } ?: 1,
                        )
                    }
                }
            }
        }
    }

    private suspend fun optimisticLockingSave(
        transaction: TransactWriteItemsEnhancedRequest.Builder,
        newerVersion: E,
        updateLogic: (suspend E.(E) -> Unit)?,
        record: E,
    ) {
        @Suppress("NAME_SHADOWING")
        var newerVersion = newerVersion

        updateLogic!!(newerVersion, record)
        newerVersion = validateUpdateAndReturn(newerVersion)
        newerVersion.generateAndSetEtag(HashMap())

        if (logger.isDebugEnabled()) {
            logger.debug { "Performing optimistic save lock!" }
        }

        transaction.addUpdateItem(
            db.dynamoDbTable,
            TransactUpdateItemEnhancedRequest
                .builder(TYPE)
                .item(newerVersion)
                .conditionExpression(buildExistingExpression(newerVersion, true))
                .build(),
        )
    }

    private fun validateUpdateAndReturn(updatedRecord: E): E {
        updatedRecord.updatedAt = Instant.now().toEpochMilli()
        val validations = updatedRecord.validateUpdate()

        if (validations.isNotEmpty()) {
            throw IllegalArgumentException(encodePrettily(validations))
        }

        return updatedRecord
    }

    private suspend fun destroyEtagsAfterCachePurge(record: E) =
        coroutineScope {
            val hashId = JsonObject().put("hash", record.hash(HASH_FIELD)).encode().hashCode()

            eTagManager?.let {
                listOf(
                    async { eTagManager.removeProjectionsEtags(hashId) },
                    async { eTagManager.destroyEtags(hashId) },
                    async { eTagManager.destroyEtags(jsonObjectOf().encode().hashCode()) },
                ).awaitAll()
            }
        }

    private fun buildExistingExpression(
        element: E,
        exists: Boolean,
    ): Expression {
        val rangeValue = element.range(IDENTIFIER_FIELD)
        val saveExpr =
            Expression
                .builder()
                .expression(
                    when (IDENTIFIER != null && IDENTIFIER != "" && rangeValue != null) {
                        false -> "${if (exists) "#hash = " else "#hash <> "} :hash"
                        else ->
                            "${if (exists) "#hash = " else "#hash <> "} :hash AND " +
                                "${if (exists) "#range = " else "#range <> "} :range"
                    },
                ).expressionNames(
                    buildMap {
                        put("#hash", HASH_IDENTIFIER)

                        if (IDENTIFIER != null && IDENTIFIER != "" && rangeValue != null) {
                            put("#range", IDENTIFIER)
                        }
                    },
                ).expressionValues(
                    buildMap {
                        put(":hash", AttributeValue.fromS(element.hash(HASH_FIELD)))

                        if (IDENTIFIER != null && IDENTIFIER != "" && rangeValue != null) {
                            put(":range", AttributeValue.fromS(rangeValue))
                        }
                    },
                ).build()

        return saveExpr
    }
}
