/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

package com.genghis.tools.repository.dynamodb.operators

import com.genghis.tools.repository.dynamodb.DynamoDBRepository
import com.genghis.tools.repository.dynamodb.hash
import com.genghis.tools.repository.dynamodb.range
import com.genghis.tools.repository.models.Cacheable
import com.genghis.tools.repository.models.DynamoDBModel
import com.genghis.tools.repository.models.ETagable
import com.genghis.tools.repository.models.Model
import com.genghis.tools.repository.repository.cache.CacheManager
import com.genghis.tools.repository.repository.etag.ETagManager
import io.github.oshai.kotlinlogging.KotlinLogging
import io.vertx.core.Future.failedFuture
import io.vertx.core.Future.succeededFuture
import io.vertx.core.json.Json.encodePrettily
import io.vertx.kotlin.core.json.jsonObjectOf
import io.vertx.kotlin.coroutines.awaitResult
import kotlinx.coroutines.async
import kotlinx.coroutines.awaitAll
import kotlinx.coroutines.coroutineScope
import software.amazon.awssdk.enhanced.dynamodb.Expression
import software.amazon.awssdk.enhanced.dynamodb.Key
import software.amazon.awssdk.enhanced.dynamodb.model.TransactDeleteItemEnhancedRequest
import software.amazon.awssdk.enhanced.dynamodb.model.TransactWriteItemsEnhancedRequest
import software.amazon.awssdk.services.dynamodb.model.AttributeValue
import java.lang.reflect.Method
import kotlin.Int
import kotlin.collections.component1
import kotlin.collections.component2

private val logger = KotlinLogging.logger { }

/**
 * This class defines the deletion operations for the DynamoDBRepository.
 *
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */
class DynamoDBDeleter<E>(
    private val TYPE: Class<E>,
    private val db: DynamoDBRepository<E>,
    private val HASH_IDENTIFIER: String,
    private val HASH_FIELD: Method?,
    private val IDENTIFIER: String?,
    private val IDENTIFIER_FIELD: Method?,
    private val cacheManager: CacheManager<E>?,
    private val eTagManager: ETagManager<E>?,
)
    where E : Cacheable, E : ETagable, E : DynamoDBModel, E : Model {
    suspend fun doDelete(
        hashValue: String?,
        rangeValues: Set<String>? = null,
        attempt: Int? = null,
    ): List<E> =
        coroutineScope {
            val items =
                when {
                    hashValue != null && (rangeValues == null || rangeValues.isEmpty()) ->
                        listOf(
                            db
                                .show {
                                    hashAndRange {
                                        hash = hashValue
                                    }
                                }.item,
                        )

                    else ->
                        db
                            .index {
                                autoPaginate(true)
                                hashAndRange {
                                    hash = hashValue
                                    ranges = rangeValues ?: emptySet()
                                }
                            }.itemList.items
                }

            if (logger.isDebugEnabled()) {
                logger.debug { "To Delete: " + encodePrettily(items) }
            }

            val transactionBuilder =
                TransactWriteItemsEnhancedRequest
                    .builder()

            items.forEach {
                this@DynamoDBDeleter.optimisticLockingDelete(transactionBuilder, it)
            }

            val transaction = transactionBuilder.build()

            val result =
                runCatching<Void> {
                    awaitResult { handler ->
                        db.dynamoDbMapper.transactWriteItems(transaction).whenCompleteAsync { t, u ->
                            when (u) {
                                null -> handler.handle(succeededFuture(t))
                                else -> handler.handle(failedFuture(u))
                            }
                        }
                    }
                }

            when {
                result.isSuccess -> {
                    val encodedHash =
                        jsonObjectOf("hash" to hashValue)
                            .encode()
                            .hashCode()

                    eTagManager?.let {
                        buildList {
                            add(async { eTagManager.removeProjectionsEtags(encodedHash) })
                            add(async { eTagManager.destroyEtags(encodedHash) })
                            add(async { eTagManager.destroyEtags(jsonObjectOf().encode().hashCode()) })
                        }.awaitAll()
                    }

                    cacheManager?.purgeCache(items) {
                        val hash = it.hash(HASH_FIELD)
                        val range = it.range(IDENTIFIER_FIELD)

                        TYPE.simpleName + "_" + hash +
                            if (range.isNullOrBlank()) {
                                if (db.hasRangeKey()) "/null" else ""
                            } else {
                                "/$range"
                            }
                    }

                    items
                }

                attempt != null && attempt > 100 -> throw (IllegalArgumentException("Over 100 attempt limit"))
                else ->
                    doDelete(
                        hashValue = hashValue,
                        rangeValues = rangeValues,
                        attempt = attempt?.let { it + 1 } ?: 1,
                    )
            }

            return@coroutineScope items
        }

    private fun optimisticLockingDelete(
        transactionBuilder: TransactWriteItemsEnhancedRequest.Builder,
        record: E,
    ): E {
        val keyBuilder = Key.builder().partitionValue(record.hash(HASH_FIELD))
        if (db.hasRangeKey()) keyBuilder.sortValue(record.range(IDENTIFIER_FIELD))

        transactionBuilder.addDeleteItem(
            db.dynamoDbTable,
            TransactDeleteItemEnhancedRequest
                .builder()
                .key(keyBuilder.build())
                .conditionExpression(buildExistingDeleteExpression(record))
                .build(),
        )

        return record
    }

    private fun buildExistingDeleteExpression(element: E): Expression {
        val rangeValue = element.range(IDENTIFIER_FIELD)
        val saveExpr =
            Expression
                .builder()
                .expression(
                    when (IDENTIFIER != null && IDENTIFIER != "" && rangeValue != null) {
                        false -> "#hash = :hash"
                        else -> "#hash = :hash AND #range = :range"
                    },
                ).expressionNames(
                    buildMap {
                        put("#hash", HASH_IDENTIFIER)

                        if (IDENTIFIER != null && IDENTIFIER != "" && rangeValue != null) {
                            put("#range", IDENTIFIER)
                        }
                    },
                ).expressionValues(
                    buildMap {
                        put(":hash", AttributeValue.fromS(element.hash(HASH_FIELD)))

                        if (IDENTIFIER != null && IDENTIFIER != "" && rangeValue != null) {
                            put(":range", AttributeValue.fromS(rangeValue))
                        }
                    },
                ).build()

        return saveExpr
    }
}
