/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

package com.genghis.tools.repository.repository

import com.genghis.tools.repository.models.Model
import com.genghis.tools.repository.models.ModelUtils
import com.genghis.tools.repository.repository.etag.ETagManager
import com.genghis.tools.repository.repository.results.CreateResult
import com.genghis.tools.repository.repository.results.DeleteResult
import com.genghis.tools.repository.repository.results.ItemListResult
import com.genghis.tools.repository.repository.results.ItemResult
import com.genghis.tools.repository.repository.results.UpdateResult
import com.genghis.tools.repository.utils.FilterParameter
import com.genghis.tools.repository.utils.HashAndRange
import com.genghis.tools.repository.utils.Query
import com.genghis.tools.repository.utils.QueryBuilder
import com.genghis.tools.version.manager.VersionManager
import io.github.oshai.kotlinlogging.KotlinLogging
import io.vertx.core.json.Json.decodeValue
import io.vertx.core.json.JsonObject
import io.vertx.kotlin.core.json.jsonObjectOf
import software.amazon.awssdk.utils.Either
import software.amazon.awssdk.utils.Either.left
import software.amazon.awssdk.utils.Either.right
import java.lang.reflect.Field
import java.lang.reflect.Type
import java.security.NoSuchAlgorithmException
import java.util.Arrays
import java.util.concurrent.ConcurrentHashMap

private val logger = KotlinLogging.logger { }

/**
 * The repository interface is the base of the Repository tools. It defines a contract for all repositories and contains
 * standardized logic.
 *
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */
@Suppress("unused")
interface Repository<E : Model, Q : Query, QB : QueryBuilder> {
    val etagManager: ETagManager<*>?
    val type: Class<E>
    val fieldMap: ConcurrentHashMap<String, Field>
    val typeMap: ConcurrentHashMap<String, Type>

    val isCached: Boolean
    val isEtagEnabled: Boolean
    val versionManager: VersionManager

    enum class INCREMENTATION {
        ADDITION,
        SUBTRACTION,
    }

    suspend fun create(record: E): CreateResult<E> = batchCreate(listOf(record)).first()

    suspend fun batchCreate(records: List<E>): List<CreateResult<E>> =
        runCatching {
            records
                .chunked(100)
                .map {
                    doWrite(
                        true,
                        it.associateWith {
                            {
                                setInitialValues(it)
                                sanitize()
                            }
                        },
                    )
                }.flatten()
        }.map { result ->
            result.map { CreateResult(it) }
        }.getOrThrow()

    suspend fun update(
        record: E,
        updateLogic: (suspend E.(E) -> Unit)? = null,
    ): UpdateResult<E> = batchUpdate(mapOf(record to updateLogic)).first()

    suspend fun batchUpdate(records: Map<E, (suspend E.(E) -> Unit)?>): List<UpdateResult<E>> =
        runCatching {
            records.entries
                .chunked(100)
                .map {
                    doWrite(
                        create = false,
                        records =
                            it.associate { (key, value) ->
                                val replacementFunction: suspend E.(E) -> Unit =
                                    value?.let {
                                        { newModel ->
                                            setModifiables(newModel)
                                            it(this, newModel)
                                            sanitize()
                                        }
                                    } ?: { newModel ->
                                        setModifiables(newModel)
                                        sanitize()
                                    }

                                key to replacementFunction
                            },
                    )
                }.flatten()
        }.map { result ->
            result.map { UpdateResult(it) }
        }.getOrThrow()

    suspend fun delete(hashAndRange: HashAndRange): DeleteResult<E> =
        batchDelete(hashAndRange.hash, ranges = hashAndRange.ranges.toList()).first()

    suspend fun batchDelete(
        hash: String?,
        ranges: List<String>?,
    ): List<DeleteResult<E>> =
        runCatching {
            require(
                (hash != null && ranges != null && ranges.isNotEmpty()) ||
                    (hash == null && ranges != null && ranges.isNotEmpty()) ||
                    (hash != null && (ranges == null || ranges.isEmpty())),
            ) {
                "Either delete multiples hash keyed items in the ranges array or a single hash with multiple ranges, " +
                    "or a single hashed item"
            }

            buildList {
                when {
                    ranges != null && ranges.isNotEmpty() -> {
                        val rangesChunks = ranges.chunked(100)

                        rangesChunks.map {
                            addAll(doDelete(hash, it))
                        }
                    }

                    else -> {
                        addAll(doDelete(hash, ranges))
                    }
                }
            }
        }.map { result ->
            result.map { DeleteResult(it) }
        }.getOrThrow()

    @Throws(IllegalArgumentException::class)
    fun incrementField(
        record: E,
        fieldName: String,
    ): suspend E.() -> Unit

    @Throws(IllegalArgumentException::class)
    fun decrementField(
        record: E,
        fieldName: String,
    ): suspend E.() -> Unit

    suspend fun show(query: Q): ItemResult<E>

    suspend fun show(builder: QB.() -> Unit): ItemResult<E>

    suspend fun batchRead(queries: List<Q>): List<ItemResult<E>> = queries.map { show(it) }

    suspend fun index(query: Q): ItemListResult<E>

    suspend fun index(builder: QB.() -> Unit): ItemListResult<E>

    suspend fun aggregation(query: Q): String

    suspend fun aggregation(builder: QB.() -> Unit): String

    fun parseParam(
        paramJsonString: String,
        key: String,
    ): Either<FilterParameter, JsonObject> {
        val filterParameter = decodeValue(paramJsonString, FilterParameter::class.java)
        val errors = jsonObjectOf()

        return when {
            filterParameter != null -> {
                filterParameter.setField(key)

                when {
                    filterParameter.isValid -> left(filterParameter)
                    else -> {
                        filterParameter.collectErrors(errors)

                        right(errors)
                    }
                }
            }

            else -> {
                errors.put(key + "_error", "Unable to parse JSON in '$key' value!")

                right(errors)
            }
        }
    }

    suspend fun doWrite(
        create: Boolean,
        records: Map<E, suspend E.(E) -> Unit>,
    ): List<E>

    suspend fun doDelete(
        hash: String?,
        ranges: List<String>?,
    ): List<E>

    fun hasField(
        fields: Array<Field>,
        key: String,
    ): Boolean {
        val hasField = Arrays.stream(fields).anyMatch { field -> field.name.equals(key, ignoreCase = true) }

        return hasField || hasField(type.superclass, key)
    }

    private fun hasField(
        klazz: Class<*>,
        key: String,
    ): Boolean {
        try {
            var field: Field? = fieldMap[key]

            if (field == null) {
                field = klazz.getDeclaredField(key)

                if (field != null) fieldMap[key] = field
            }

            val hasField = field != null

            return hasField || hasField(klazz.superclass, key)
        } catch (e: NoSuchFieldException) {
            return false
        } catch (e: NullPointerException) {
            return false
        }
    }

    fun buildCollectionEtag(etags: List<String>): String {
        val newEtag = LongArray(1)

        return try {
            etags.forEach { etag -> newEtag[0] += etag.hashCode().toLong() }

            ModelUtils.hashString(newEtag[0].toString())
        } catch (e: NoSuchAlgorithmException) {
            e.printStackTrace()

            "noTagForCollection"
        } catch (e: NullPointerException) {
            e.printStackTrace()
            "noTagForCollection"
        }
    }

    fun extractValueAsDouble(
        field: Field,
        r: E,
    ): Double {
        try {
            return field.getLong(r).toDouble()
        } catch (e: IllegalArgumentException) {
            try {
                return (field.get(r) as Long).toDouble()
            } catch (ignored: IllegalArgumentException) {
            } catch (ignored: IllegalAccessException) {
            }
        } catch (e: IllegalAccessException) {
            try {
                return (field.get(r) as Long).toDouble()
            } catch (ignored: IllegalArgumentException) {
            } catch (ignored: IllegalAccessException) {
            }
        }

        try {
            return field.getInt(r).toDouble()
        } catch (e: IllegalArgumentException) {
            try {
                return (field.get(r) as Int).toDouble()
            } catch (ignored: IllegalArgumentException) {
            } catch (ignored: IllegalAccessException) {
            }
        } catch (e: IllegalAccessException) {
            try {
                return (field.get(r) as Int).toDouble()
            } catch (ignored: IllegalArgumentException) {
            } catch (ignored: IllegalAccessException) {
            }
        }

        try {
            return field.getShort(r).toDouble()
        } catch (e: IllegalArgumentException) {
            try {
                return (field.get(r) as Short).toDouble()
            } catch (ignored: IllegalArgumentException) {
            } catch (ignored: IllegalAccessException) {
            }
        } catch (e: IllegalAccessException) {
            try {
                return (field.get(r) as Short).toDouble()
            } catch (ignored: IllegalArgumentException) {
            } catch (ignored: IllegalAccessException) {
            }
        }

        try {
            return field.getDouble(r)
        } catch (ignored: IllegalArgumentException) {
        } catch (ignored: IllegalAccessException) {
        }

        try {
            return field.getFloat(r).toDouble()
        } catch (e: IllegalArgumentException) {
            try {
                return (field.get(r) as Float).toDouble()
            } catch (ignored: IllegalArgumentException) {
            } catch (ignored: IllegalAccessException) {
            }
        } catch (e: IllegalAccessException) {
            try {
                return (field.get(r) as Float).toDouble()
            } catch (ignored: IllegalArgumentException) {
            } catch (ignored: IllegalAccessException) {
            }
        }

        logger.error { "Conversion is null!" }

        throw IllegalArgumentException()
    }
}
