/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

package com.genghis.tools.repository.repository.cache

import com.fasterxml.jackson.annotation.JsonTypeInfo
import com.genghis.tools.repository.models.Cacheable
import com.genghis.tools.repository.models.Model
import com.genghis.tools.repository.utils.ItemList
import com.genghis.tools.repository.utils.ItemListMeta
import com.genghis.tools.repository.utils.PageTokens
import com.hazelcast.cache.CacheNotExistsException
import com.hazelcast.cache.HazelcastCachingProvider
import com.hazelcast.cache.ICache
import com.hazelcast.core.Hazelcast
import io.github.oshai.kotlinlogging.KotlinLogging
import io.vertx.core.Future
import io.vertx.core.Promise
import io.vertx.core.Vertx
import io.vertx.core.json.Json.decodeValue
import io.vertx.core.json.Json.encode
import io.vertx.core.json.JsonObject
import io.vertx.kotlin.coroutines.awaitBlocking
import io.vertx.kotlin.coroutines.coAwait
import io.vertx.kotlin.coroutines.dispatcher
import io.vertx.serviceproxy.ServiceException
import kotlinx.coroutines.TimeoutCancellationException
import kotlinx.coroutines.async
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.withContext
import kotlinx.coroutines.withTimeout
import java.lang.Boolean.TRUE
import java.util.Arrays
import java.util.concurrent.Callable
import java.util.concurrent.TimeoutException
import java.util.function.Function
import java.util.function.Supplier
import java.util.stream.Collectors.toList
import javax.cache.Caching
import javax.cache.configuration.CompleteConfiguration
import javax.cache.configuration.MutableConfiguration
import javax.cache.expiry.AccessedExpiryPolicy
import javax.cache.expiry.Duration.ONE_HOUR

private val logger = KotlinLogging.logger { }

/**
 * The ClusterCacheManagerImpl contains the logic for setting, removing, and replace caches.
 *
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */
@Suppress("PrivatePropertyName")
class ClusterCacheManagerImpl<E>(
    private val TYPE: Class<E>,
    private val vertx: Vertx,
) : CacheManager<E> where E : Cacheable, E : Model {
    private val itemListKeyMap: String = TYPE.simpleName + "/ITEMLIST"
    private val aggregationKeyMap: String = TYPE.simpleName + "/AGGREGATION"

    private val cacheReadTimeout = 1000L
    private val cacheWriteTimeout = 10000L
    private val expiryPolicy = AccessedExpiryPolicy.factoryOf(ONE_HOUR).create()

    private val hasTypeJsonField: Boolean = Arrays.stream(TYPE.declaredAnnotations).anyMatch { a -> a is JsonTypeInfo }

    override suspend fun initializeCache(): Boolean {
        if (cachesCreated) return true

        withContext(vertx.dispatcher()) {
            objectCache = createCache("object")
            itemListCache = createCache("itemList")
            aggregationCache = createCache("aggregation")
            cachesCreated = true
        }

        return cachesCreated
    }

    private fun createCache(cacheName: String): ICache<String, String>? {
        val instances = Hazelcast.getAllHazelcastInstances()
        val hzOpt = instances.stream().findFirst()

        when {
            hzOpt.isPresent -> {
                val hz = hzOpt.get()

                try {
                    val cache = hz.cacheManager.getCache<String, String>(cacheName)

                    logger.info { "Initialized cache: ${cache.name} ok!" }

                    return cache
                } catch (cnee: CacheNotExistsException) {
                    val cachingProvider = Caching.getCachingProvider(HazelcastCachingProvider::class.java.name)
                    val config =
                        MutableConfiguration<String, String>()
                            .setTypes(String::class.java, String::class.java)
                            .setManagementEnabled(false)
                            .setStatisticsEnabled(false)
                            .setReadThrough(false)
                            .setWriteThrough(false)

                    @Suppress("UNCHECKED_CAST")
                    return cachingProvider.cacheManager
                        .createCache<String, String, CompleteConfiguration<String, String>>(
                            cacheName,
                            config,
                        ).unwrap(ICache::class.java) as ICache<String, String>?
                } catch (ilse: IllegalStateException) {
                    logger.error { "JCache not available!" }

                    return null
                }
            }

            else -> {
                logger.error { "Cannot find hazelcast instance!" }

                return null
            }
        }
    }

    override suspend fun checkObjectCache(
        cacheId: String,
        typeName: String,
    ): E =
        when {
            isObjectCacheAvailable() ->
                runCatching {
                    withTimeout(cacheReadTimeout) {
                        runCatching {
                            val fetchFuture = Promise.promise<String>()

                            objectCache!!.getAsync(cacheId).whenCompleteAsync { t, u ->
                                when (u) {
                                    null -> fetchFuture.tryComplete(t)
                                    else -> {
                                        logger.error {
                                            "$u : ${u.message} : ${u.stackTrace?.joinToString {
                                                "${it.fileName}:${it.lineNumber}\n"
                                            }}"
                                        }

                                        fetchFuture.fail(u)
                                    }
                                }
                            }

                            fetchFuture.future().coAwait()
                        }.map {
                            runCatching {
                                logger.debug { "Cached Content is: ${it!!}" }

                                when (it) {
                                    null -> throw ServiceException(404, "Cache result is null!")
                                    else -> decodeValue(it, TYPE)
                                }
                            }.recoverCatching {
                                when (it) {
                                    is ServiceException -> throw it
                                    else -> {
                                        logger.error {
                                            "$it : ${it.message} : ${it.stackTrace?.joinToString {
                                                "${it.fileName}:${it.lineNumber}\n"
                                            }}"
                                        }

                                        throw it
                                    }
                                }
                            }.getOrThrow()
                        }.getOrThrow()
                    }
                }.getOrElse(cacheFetchFail(objectCache?.name, "$cacheId -> $typeName"))

            else -> {
                logger.error { "ObjectCache is null, recreating..." }

                throw ServiceException(404, "Unable to retrieve from cache, cache was null...")
            }
        }

    override suspend fun checkItemListCache(
        cacheId: String,
        projections: Set<String>,
        typeName: String,
    ): ItemList<E> =
        when {
            isItemListCacheAvailable() ->
                runCatching {
                    logger.debug { "Checking Item List Cache" }

                    withTimeout(cacheReadTimeout) {
                        val fetchFuture = Promise.promise<String>()

                        itemListCache!!.getAsync(cacheId).whenCompleteAsync { t, u ->
                            when (u) {
                                null -> fetchFuture.tryComplete(t)
                                else -> {
                                    logger.error {
                                        "$u : ${u.message} : ${u.stackTrace?.joinToString {
                                            "${it.fileName}:${it.lineNumber}\n"
                                        }}"
                                    }

                                    throw ServiceException(500, "Cache fetch failed...", JsonObject(encode(u)))
                                }
                            }
                        }

                        runCatching {
                            val cacheResult = fetchFuture.future().coAwait() ?: throw ServiceException(404, "Null")
                            val jsonObject = JsonObject(cacheResult)
                            val jsonArray = jsonObject.getJsonArray("items")
                            val meta = ItemListMeta(jsonObject.getJsonObject("meta"))
                            val pageToken = PageTokens(jsonObject.getJsonObject("paging"))
                            val items =
                                jsonArray
                                    .stream()
                                    .map { json ->
                                        val obj = JsonObject(json.toString())

                                        if (hasTypeJsonField) {
                                            obj.put("@type", TYPE.simpleName)
                                        }

                                        decodeValue(obj.encode(), TYPE)
                                    }.collect(toList())

                            ItemList<E>(
                                items = items,
                                meta = meta,
                                paging = pageToken,
                            )
                        }.recoverCatching {
                            when (it) {
                                is ServiceException -> throw it
                                else -> {
                                    logger.error {
                                        "$it : ${it.message} : ${it.stackTrace?.joinToString {
                                            "${it.fileName}:${it.lineNumber}\n"
                                        }}"
                                    }

                                    throw it
                                }
                            }
                        }.getOrThrow()
                    }
                }.getOrElse(cacheFetchFail(itemListCache?.name, "$cacheId -> $typeName"))

            else -> {
                logger.error { "ItemList Cache is null, recreating..." }

                throw ServiceException(404, "Unable to perform cache fetch, cache was null...")
            }
        }

    override suspend fun checkAggregationCache(
        cacheKey: String,
        typeName: String,
    ): String =
        when {
            isAggregationCacheAvailable() ->
                runCatching {
                    logger.debug { "Checking Item List Cache" }

                    withTimeout(cacheReadTimeout) {
                        val fetchFuture = Promise.promise<String>()

                        itemListCache!!.getAsync(cacheKey).whenCompleteAsync { t, u ->
                            when (u) {
                                null -> fetchFuture.tryComplete(t)
                                else -> {
                                    logger.error {
                                        "$u : ${u.message} : ${u.stackTrace?.joinToString {
                                            "${it.fileName}:${it.lineNumber}\n"
                                        }}"
                                    }

                                    throw ServiceException(500, "Cache fetch failed...", JsonObject(encode(u)))
                                }
                            }
                        }

                        runCatching {
                            fetchFuture.future().coAwait()
                        }.recoverCatching {
                            logger.error {
                                "$it : ${it.message} : ${it.stackTrace?.joinToString {
                                    "${it.fileName}:${it.lineNumber}\n"
                                }}"
                            }

                            throw ServiceException(404, "Cache result is null...", JsonObject(encode(it)))
                        }.getOrThrow()
                    }
                }.getOrElse(cacheFetchFail(aggregationCache?.name, "$cacheKey -> $typeName"))

            else -> {
                logger.error { "Aggregation Cache is null, recreating..." }

                throw ServiceException(404, "Unable to perform cache fetch, cache was null...")
            }
        }

    private fun <T> cacheFetchFail(
        cacheName: String?,
        infoMessage: String,
    ): (exception: Throwable) -> T =
        {
            when (it) {
                is TimeoutCancellationException ->
                    throw ServiceException(502, "Cache ($cacheName) timeout: $infoMessage!")

                is ServiceException -> throw it

                else ->
                    throw ServiceException(502, "Cache ($cacheName) error: $infoMessage!", JsonObject(encode(it)))
            }
        }

    override suspend fun replaceObjectCache(
        cacheId: String,
        item: E,
        future: Promise<E>,
        projections: Set<String>,
    ) {
        when {
            isObjectCacheAvailable() -> {
                val fullCacheContent = encode(item)
                val jsonRepresentationCache =
                    vertx
                        .executeBlocking(
                            Callable {
                                item.toJsonFormat(projections).encode()
                            },
                        ).coAwait()
                val fullCacheFuture = Promise.promise<Boolean>()
                val jsonFuture = Promise.promise<Boolean>()

                vertx.setTimer(cacheWriteTimeout) {
                    if (!fullCacheFuture.future().isComplete) {
                        objectCache!!.removeAsync("FULL_CACHE_$cacheId")
                        fullCacheFuture.tryComplete()

                        logger.error { "Cache timeout!" }
                    }

                    if (!jsonFuture.future().isComplete) {
                        objectCache!!.removeAsync(cacheId)
                        jsonFuture.tryComplete()

                        logger.error { "Cache timeout!" }
                    }
                }

                objectCache!!
                    .putAsync("FULL_CACHE_$cacheId", fullCacheContent, expiryPolicy)
                    .thenAccept {
                        logger.debug { "Set new object cache on: $cacheId is $it" }

                        fullCacheFuture.tryComplete(TRUE)
                    }.exceptionally {
                        logger.error {
                            "$it : ${it.message} : ${it.stackTrace?.joinToString {
                                "${it.fileName}:${it.lineNumber}\n"
                            }}"
                        }

                        fullCacheFuture.tryFail(it)

                        null
                    }

                objectCache!!
                    .putAsync(cacheId, jsonRepresentationCache, expiryPolicy)
                    .thenAccept {
                        logger.debug { "Set new object cache on: $cacheId is $it" }

                        jsonFuture.tryComplete(TRUE)
                    }.exceptionally {
                        logger.error {
                            "$it : ${it.message} : ${it.stackTrace?.joinToString {
                                "${it.fileName}:${it.lineNumber}\n"
                            }}"
                        }

                        jsonFuture.tryFail(it)

                        null
                    }

                Future.all(fullCacheFuture.future(), jsonFuture.future()).andThen {
                    if (it.failed()) {
                        future.fail(it.cause())
                    } else {
                        future.complete(item)
                    }
                }
            }

            else -> {
                logger.error { "ObjectCache is null, recreating..." }

                future.complete(item)
            }
        }
    }

    override suspend fun replaceCache(
        records: List<E>,
        shortCacheIdSupplier: Function<E, String>,
        cacheIdSupplier: Function<E, String>,
    ) = coroutineScope {
        when {
            isObjectCacheAvailable() -> {
                when {
                    isObjectCacheAvailable() -> {
                        val purgeSecondaries =
                            async {
                                purgeSecondaryCaches()
                            }
                        val toPurge =
                            buildMap {
                                records.forEach { record ->
                                    val shortCacheId = shortCacheIdSupplier.apply(record)
                                    val cacheId = cacheIdSupplier.apply(record)
                                    val jsonValue = encode(record)
                                    val secondaryCache = "FULL_CACHE_$cacheId"
                                    val secondaryShortCache = "FULL_CACHE_$shortCacheId"

                                    put(cacheId, record.toJsonString())
                                    put(shortCacheId, record.toJsonString())
                                    put(secondaryCache, jsonValue)
                                    put(secondaryShortCache, jsonValue)
                                }
                            }

                        val purgeMain =
                            async {
                                vertx
                                    .executeBlocking(
                                        Callable {
                                            objectCache!!.putAll(toPurge, expiryPolicy)
                                        },
                                    ).coAwait()
                            }

                        purgeMain.await()
                        purgeSecondaries.await()
                    }

                    else -> false
                }

                purgeSecondaryCaches()
            }

            else -> {
                logger.error { "ObjectCache is null, recreating..." }

                purgeSecondaryCaches()
            }
        }
    }

    override suspend fun replaceItemListCache(
        content: String,
        cacheIdSupplier: Supplier<String>,
    ) = when {
        isItemListCacheAvailable() -> {
            val cacheId = cacheIdSupplier.get()

            runCatching {
                withTimeout(cacheWriteTimeout) {
                    val cacheFuture = Promise.promise<Boolean>()

                    itemListCache!!.putAsync(cacheId, content, expiryPolicy).whenCompleteAsync { _, u ->
                        when (u) {
                            null -> {
                                logger.debug { "Set new itemlist cache on: $cacheId" }

                                cacheFuture.tryComplete(true)
                            }

                            else -> {
                                logger.error {
                                    "$u : ${u.message} : ${u.stackTrace?.joinToString {
                                        "${it.fileName}:${it.lineNumber}\n"
                                    }}}"
                                }

                                cacheFuture.fail(u)
                            }
                        }
                    }

                    runCatching {
                        cacheFuture.future().coAwait()

                        replaceMapValues(cacheFuture, itemListKeyMap, cacheId)

                        true
                    }.onFailure {
                        throw ServiceException(500, it.cause?.message)
                    }.getOrDefault(false)
                }
            }.getOrElse(replaceCacheTimeout(itemListCache!!, cacheId))
        }

        else -> {
            logger.error { "ItemListCache is null, recreating..." }

            throw ServiceException(500, "Itemlist cache does not exist!")
        }
    }

    override suspend fun replaceAggregationCache(
        content: String,
        cacheIdSupplier: Supplier<String>,
    ) = when {
        isAggregationCacheAvailable() -> {
            val cacheKey = cacheIdSupplier.get()

            runCatching {
                withTimeout(cacheWriteTimeout) {
                    val cacheFuture = Promise.promise<Boolean>()

                    aggregationCache!!.putAsync(cacheKey, content, expiryPolicy).whenCompleteAsync { _, u ->
                        when (u) {
                            null -> {
                                logger.debug { "Set new aggregation cache on: $cacheKey" }

                                cacheFuture.tryComplete(true)
                            }

                            else -> {
                                logger.error {
                                    "$u : ${u.message} : ${u.stackTrace?.joinToString {
                                        "${it.fileName}:${it.lineNumber}\n"
                                    }}}"
                                }

                                cacheFuture.fail(u)
                            }
                        }
                    }

                    runCatching {
                        cacheFuture.future().coAwait()

                        replaceMapValues(cacheFuture, aggregationKeyMap, cacheKey)

                        true
                    }.onFailure {
                        throw ServiceException(500, it.cause?.message)
                    }.getOrDefault(false)
                }
            }.getOrElse(replaceCacheTimeout(aggregationCache!!, cacheKey))
        }

        else -> {
            logger.error { "AggregationCache is null, recreating..." }

            throw ServiceException(500, "Aggregation cache does not exist!")
        }
    }

    private fun replaceCacheTimeout(
        cache: ICache<String, String>,
        cacheId: String,
    ): (exception: Throwable) -> Boolean =
        {
            when (it) {
                is TimeoutCancellationException -> {
                    logger.error { "Cache timeout when replacing ${cache.name} for: $cacheId!" }

                    cache.removeAsync(cacheId)

                    throw TimeoutException("Cache request timed out, above: $cacheWriteTimeout!")
                }

                else -> {
                    logger.warn(it) { "Cache error!" }

                    throw ServiceException(502, "Cache error!", JsonObject(encode(it)))
                }
            }
        }

    private suspend fun replaceMapValues(
        cacheIdFuture: Promise<Boolean>,
        aggKeyMap: String,
        cacheKey: String,
    ) {
        val map =
            runCatching {
                vertx.sharedData().getClusterWideMap<String, Set<String>>(aggKeyMap).coAwait()
            }.onFailure {
                logger.error(it) { "Cannot set cachemap..." }

                cacheIdFuture.tryComplete()
            }.getOrNull()

        val set =
            runCatching {
                map?.get(TYPE.simpleName)?.coAwait()
            }.onFailure {
                logger.error(it) { "Unable to get TYPE id set!" }

                cacheIdFuture.tryComplete()
            }.getOrNull()

        var idSet: MutableSet<String>? = set?.toMutableSet()

        when (idSet) {
            null -> {
                idSet = HashSet()

                idSet.add(cacheKey)

                runCatching {
                    map?.put(TYPE.simpleName, idSet)?.coAwait()
                }.onFailure {
                    logger.error(it) { "Unable to set cacheIdSet!" }
                }

                cacheIdFuture.tryComplete()
            }

            else -> {
                idSet.add(cacheKey)

                runCatching {
                    map?.replace(TYPE.simpleName, idSet)?.coAwait()
                }.onFailure {
                    logger.error(it) { "Unable to set cacheIdSet!" }
                }

                cacheIdFuture.tryComplete()
            }
        }
    }

    override suspend fun purgeCache(
        records: List<E>,
        cacheIdSupplier: (E) -> String,
    ) = coroutineScope {
        when {
            isObjectCacheAvailable() -> {
                val purgeSecondaries =
                    async {
                        purgeSecondaryCaches()
                    }
                val toPurge =
                    buildSet {
                        records.forEach { record ->
                            val cacheId = cacheIdSupplier(record)
                            val secondaryCache = "FULL_CACHE_$cacheId"

                            add(cacheId)
                            add(secondaryCache)
                        }
                    }

                val purgeMain =
                    async {
                        vertx
                            .executeBlocking(
                                Callable {
                                    objectCache!!.removeAll(toPurge)
                                },
                            ).coAwait()
                    }

                purgeMain.await()
                purgeSecondaries.await()
            }

            else -> {
                logger.error { "ObjectCache is null, recreating..." }

                purgeSecondaryCaches()
            }
        }
    }

    private suspend fun purgeSecondaryCaches() {
        logger.debug { "Purging secondary caches..." }

        val itemListFuture = Promise.promise<Boolean>()
        val aggregationFuture = Promise.promise<Boolean>()

        when {
            isItemListCacheAvailable() -> {
                vertx.setTimer(cacheWriteTimeout) {
                    if (!itemListFuture.future().isComplete && !itemListCache!!.isDestroyed) {
                        itemListCache!!.clear()

                        itemListFuture.tryComplete()

                        logger.error { "Cache Timeout purging secondary caches for itemlist!" }
                    }
                }

                runCatching {
                    purgeMap(itemListKeyMap, itemListCache)
                }.onFailure {
                    logger.warn(it) { "Failed itemlist purge" }

                    itemListCache = null
                }

                itemListFuture.tryComplete()
            }

            else -> {
                logger.error { "ItemListCache is null, recreating..." }

                itemListFuture.tryComplete()
            }
        }

        when {
            isAggregationCacheAvailable() -> {
                vertx.setTimer(cacheWriteTimeout) {
                    if (!aggregationFuture.future().isComplete && !aggregationCache!!.isDestroyed) {
                        aggregationCache!!.clear()

                        aggregationFuture.tryComplete()

                        logger.error { "Cache timeout purging aggregationcache!" }
                    }
                }

                runCatching {
                    purgeMap(aggregationKeyMap, aggregationCache)
                }.onFailure {
                    aggregationCache = null
                }

                aggregationFuture.tryComplete()
            }

            else -> {
                logger.error { "AggregateCache is null, recreating..." }

                aggregationFuture.tryComplete()
            }
        }

        Future
            .any(
                itemListFuture.future(),
                aggregationFuture.future(),
            ).coAwait()
    }

    private suspend fun purgeMap(
        mapKey: String,
        cache: ICache<String, String>?,
    ) {
        logger.debug { "Now purging cache: $mapKey" }

        try {
            val map = vertx.sharedData().getClusterWideMap<String, Set<String>>(mapKey).coAwait()
            val cachePartitionKey = TYPE.getDeclaredConstructor().newInstance().cachePartitionKey
            val cacheSet = map.get(cachePartitionKey).coAwait<Set<String>?>()

            cacheSet?.let { cache!!.removeAll(it) } ?: map.put(cachePartitionKey, HashSet())

            logger.debug { "Cache cleared: " + cache!!.size() }
        } catch (e: Exception) {
            logger.error(e) { "Error purging cache" }
            logger.error { "Unable to purge cache, nulling..." }

            awaitBlocking { cache!!.clear() }
        }
    }

    override suspend fun isObjectCacheAvailable(): Boolean =
        awaitBlocking {
            val available = objectCache != null

            if (!available) {
                objectCache = createCache("object")
            }

            available
        }

    override suspend fun isItemListCacheAvailable(): Boolean =
        awaitBlocking {
            val available = itemListCache != null

            if (!available) {
                itemListCache = createCache("itemList")
            }

            available
        }

    override suspend fun isAggregationCacheAvailable(): Boolean =
        awaitBlocking {
            val available = aggregationCache != null

            if (!available) {
                aggregationCache = createCache("aggregation")
            }

            available
        }

    companion object {
        private var cachesCreated = false
        private var objectCache: ICache<String, String>? = null
        private var itemListCache: ICache<String, String>? = null
        private var aggregationCache: ICache<String, String>? = null
    }
}
