/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

package com.genghis.tools.repository.repository.etag

import com.genghis.tools.repository.models.ETagable
import com.genghis.tools.repository.models.Model
import com.genghis.tools.repository.repository.redis.GenghisRedisClient
import com.genghis.tools.repository.utils.ItemList
import io.github.oshai.kotlinlogging.KotlinLogging
import io.vertx.core.json.Json
import io.vertx.kotlin.coroutines.coAwait
import io.vertx.redis.client.ResponseType.ATTRIBUTE
import io.vertx.redis.client.ResponseType.BOOLEAN
import io.vertx.redis.client.ResponseType.BULK
import io.vertx.redis.client.ResponseType.ERROR
import io.vertx.redis.client.ResponseType.MULTI
import io.vertx.redis.client.ResponseType.NUMBER
import io.vertx.redis.client.ResponseType.PUSH
import io.vertx.redis.client.ResponseType.SIMPLE
import io.vertx.redis.client.impl.types.MultiType
import java.lang.IllegalArgumentException
import kotlin.collections.component1
import kotlin.collections.component2

private val logger = KotlinLogging.logger { }

/**
 * The RedisETagManagerImpl contains the logic for setting, removing, and replace etags.
 *
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */
class RedisETagManagerImpl<E>(
    private val TYPE: Class<E>,
    private val REDIS_CLIENT: GenghisRedisClient,
) : ETagManager<E> where E : ETagable, E : Model {
    override suspend fun removeProjectionsEtags(hash: Int): Boolean {
        val etagKeyBase = TYPE.simpleName + "_" + hash + "/projections"

        return doEtagRemovalWithRetry(etagKeyBase)
    }

    override suspend fun destroyEtags(hash: Int): Boolean {
        val etagItemListHashKey = TYPE.simpleName + "_" + hash + "_" + "itemListEtags"

        return doEtagRemovalWithRetry(etagItemListHashKey)
    }

    private suspend fun doEtagRemovalWithRetry(etagKeyBase: String): Boolean =
        runCatching {
            val getResult =
                REDIS_CLIENT.performRedisWithRetry {
                    it.hgetall(etagKeyBase).coAwait()
                }
            val itemsToRemove =
                buildList {
                    when (getResult.type()) {
                        SIMPLE -> add(getResult.toString())
                        MULTI -> addAll((getResult as MultiType).keys)
                        ERROR,
                        BOOLEAN,
                        NUMBER,
                        BULK,
                        PUSH,
                        ATTRIBUTE,
                        null,
                        -> throw IllegalArgumentException("Unknown type!")
                    }
                }

            if (itemsToRemove.isNotEmpty()) {
                REDIS_CLIENT.performRedisWithRetry {
                    val args =
                        buildList {
                            add(etagKeyBase)

                            itemsToRemove.forEach {
                                add(it)
                            }
                        }

                    it.hdel(args).coAwait()
                }
            }

            true
        }.onFailure {
            logger.error(it) { "Failed to do multi removal: $etagKeyBase" }
        }.getOrElse {
            false
        }

    override suspend fun replaceAggregationEtag(
        etagItemListHashKey: String,
        etagKey: String,
        newEtag: String,
    ): Boolean =
        runCatching {
            REDIS_CLIENT.performRedisWithRetry {
                it.hset(listOf(etagItemListHashKey, etagKey, newEtag)).coAwait()
            }

            true
        }.onFailure {
            logger.error(it) { "Failed to set aggreation etag map: $etagItemListHashKey : $etagKey : $newEtag" }
        }.getOrElse {
            false
        }

    override suspend fun setSingleRecordEtag(etagMap: Map<String, String>): Boolean =
        runCatching {
            REDIS_CLIENT.performRedisWithRetry { redis ->
                val args =
                    buildList {
                        etagMap.forEach { (key, value) ->
                            add(key)
                            add(value)
                        }
                    }

                redis.mset(args).coAwait()

                true
            }
        }.onFailure {
            logger.error(it) { "Failed to set etag map: ${Json.encodePrettily(etagMap)}" }
        }.getOrElse {
            false
        }

    override suspend fun setProjectionEtags(
        projections: Array<String>,
        hash: Int,
        item: E,
    ): Boolean =
        if (projections.isNotEmpty()) {
            val etagKeyBase = TYPE.simpleName + "_" + hash + "/projections"
            val key = TYPE.simpleName + "_" + hash + "/projections" + projections.contentHashCode()
            val etag = item.etag

            runCatching {
                REDIS_CLIENT.performRedisWithRetry {
                    it.hset(listOf(etagKeyBase, key, etag)).coAwait()
                }

                true
            }.onFailure {
                logger.error(it) { "Unable to store projection etag: $hash!" }
            }.getOrElse { false }
        } else {
            false
        }

    override suspend fun setItemListEtags(
        etagItemListHashKey: String,
        etagKey: String,
        itemList: ItemList<E>,
    ): Boolean =
        runCatching {
            REDIS_CLIENT.performRedisWithRetry {
                it.hset(listOf(etagItemListHashKey, etagKey, itemList.meta.etag)).coAwait()

                true
            }
        }.onFailure {
            logger.error(it) { "Unable to store itemList etag: $etagItemListHashKey : $etagKey!" }
        }.getOrElse {
            false
        }

    override suspend fun checkItemEtag(
        etagKeyBase: String,
        key: String,
        requestEtag: String,
    ): Boolean =
        runCatching {
            REDIS_CLIENT.performRedisWithRetry {
                it.hget(etagKeyBase, key).coAwait().toString() == requestEtag
            }
        }.onFailure {
            logger.warn(it) { "Unable to check item etag: $etagKeyBase, key: $key, etag: $requestEtag!" }
        }.getOrElse { false }

    override suspend fun checkItemListEtag(
        etagItemListHashKey: String,
        etagKey: String,
        etag: String,
    ): Boolean =
        runCatching {
            REDIS_CLIENT.performRedisWithRetry {
                it.hget(etagItemListHashKey, etagKey).coAwait().toString() == etag
            }
        }.onFailure {
            logger.warn(it) { "Unable to check itemList etag: $etagItemListHashKey, key: $etagKey, etag: $etag!" }
        }.getOrElse { false }

    override suspend fun checkAggregationEtag(
        etagItemListHashKey: String,
        etagKey: String,
        etag: String,
    ): Boolean =
        runCatching {
            REDIS_CLIENT.performRedisWithRetry {
                it.hget(etagItemListHashKey, etagKey).coAwait().toString() == etag
            }
        }.onFailure {
            logger.warn(it) { "Unable to check aggregation etag: $etagItemListHashKey, key: $etagKey, etag: $etag!" }
        }.getOrElse { false }
}
