package com.genghis.tools.repository.repository.redis

import io.github.oshai.kotlinlogging.KotlinLogging
import io.vertx.core.Future
import io.vertx.core.Promise
import io.vertx.core.Vertx
import io.vertx.kotlin.redis.client.redisOptionsOf
import io.vertx.redis.client.Redis
import io.vertx.redis.client.RedisAPI
import io.vertx.redis.client.RedisAPI.api
import io.vertx.redis.client.RedisClientType
import io.vertx.redis.client.RedisConnection
import io.vertx.redis.client.RedisOptions
import io.vertx.redis.client.RedisReplicas
import kotlinx.coroutines.delay
import java.util.concurrent.atomic.AtomicBoolean
import kotlin.math.pow

private val logger = KotlinLogging.logger {}

data class GenghisRedisConfiguration(
    val redisServer: String = "localhost",
    val redisPort: Int = 6380,
    val tls: Boolean = false,
    val clustered: Boolean = false,
    val replication: Boolean = false,
    val redisOptions: RedisOptions =
        redisOptionsOf(
            maxPoolSize = 48,
            type =
                when {
                    clustered -> RedisClientType.CLUSTER
                    replication -> RedisClientType.REPLICATION
                    else -> RedisClientType.STANDALONE
                },
            useReplicas =
                when {
                    clustered || replication -> RedisReplicas.ALWAYS
                    else -> RedisReplicas.NEVER
                },
        ).setConnectionString("redis${if (tls) "s" else ""}://$redisServer:$redisPort"),
)

class GenghisRedisClient(
    private val vertx: Vertx,
    private val conf: GenghisRedisConfiguration,
) {
    private val maxRetries = 16
    private var connection: RedisConnection? = null
    private val connecting = AtomicBoolean()
    private val clientOptions by lazy {
        val netOptions = conf.redisOptions.netClientOptions
        netOptions.setSsl(conf.tls)
        netOptions.hostnameVerificationAlgorithm = if (conf.tls) "HTTPS" else "NONE"

        conf.redisOptions
    }

    init {
        connecting.set(false)

        createRedisClient()
            .onSuccess { connection = it }
            .onFailure { logger.warn(it) { "Unable to start RedisClient!" } }
    }

    suspend fun <T> performRedisWithRetry(handler: suspend (RedisAPI) -> T) = handler(attemptConnection(0))

    private fun createRedisClient(): Future<RedisConnection> {
        val promise: Promise<RedisConnection> = Promise.promise()

        when {
            connecting.compareAndSet(false, true) -> {
                runCatching {
                    logger.debug { "Connecting to Redis: ${conf.redisServer}" }

                    Redis
                        .createClient(vertx, clientOptions)
                        .connect()
                        .onSuccess { redisConnection ->
                            logger.info { "Redis connected ${conf.redisServer}!" }

                            // make sure to invalidate old connection if present
                            connection?.close()

                            // make sure the client is reconnected on error
                            // eg, the underlying TCP connection is closed but the client side doesn't know it yet
                            // the client tries to use the staled connection to talk to server.
                            // An exceptions will be raised
                            redisConnection.exceptionHandler {
                                logger.debug(it) { "Redis reconnecting: ${conf.redisServer}!" }

                                attemptReconnect(0)
                            }

                            // make sure the client is reconnected on connection close
                            // eg, the underlying TCP connection is closed with normal 4-Way-Handshake
                            // this handler will be notified instantly
                            redisConnection.endHandler {
                                logger.debug { "Redis reconnecting at closed connection: ${conf.redisServer}!" }

                                attemptReconnect(0)
                            }

                            // allow further processing
                            promise.complete(redisConnection)
                            connection = redisConnection
                            connecting.set(false)
                        }.onFailure { t ->
                            logger.warn(t) { "Redis failed ${conf.redisServer} : ${conf.redisPort}!" }
                            promise.fail(t)
                            connecting.set(false)
                        }
                }.onFailure {
                    logger.warn(it) { "Redis failed ${conf.redisServer} : ${conf.redisPort}!" }
                    promise.fail(it)
                    connecting.set(false)
                }
            }
            else -> promise.complete()
        }

        return promise.future()
    }

    private suspend fun attemptConnection(retry: Int): RedisAPI =
        when {
            retry > maxRetries -> throw IllegalStateException("Redis dead")
            connection != null && !connecting.get() -> api(connection)
            else -> {
                val backoff = (2.0.pow(retry.coerceAtMost(10).toDouble()) * 10).toLong()

                runCatching {
                    delay(backoff)
                    attemptConnection(retry + 1)
                }.onFailure {
                    logger.info { "Cannot connect timer for redis $retry" }
                }.getOrThrow()
            }
        }

    private fun attemptReconnect(retry: Int) {
        when {
            retry > maxRetries -> connecting.set(false)
            else -> {
                val backoff = (2.0.pow(retry.coerceAtMost(10).toDouble()) * 10).toLong()

                runCatching {
                    vertx.setTimer(backoff) {
                        runCatching {
                            createRedisClient().onFailure { attemptReconnect(retry + 1) }
                        }.onFailure {
                            logger.info { "Cannot set connect timer for redis" }
                        }
                    }
                }.onFailure {
                    logger.info { "Cannot set reconnect timer for redis" }
                }
            }
        }
    }
}
