/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

package com.genghis.tools.repository.utils

import com.fasterxml.jackson.annotation.JsonEnumDefaultValue
import com.genghis.tools.repository.utils.OrderByDirection.ASC
import com.genghis.tools.repository.utils.OrderByDirection.DESC
import io.github.oshai.kotlinlogging.KotlinLogging

@Suppress("unused")
private val logger = KotlinLogging.logger { }

enum class OrderByDirection {
    @JsonEnumDefaultValue
    DESC,
    ASC,
}

/**
 * This class defines an orderByParameter, which is used for sorting results.
 *
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */
data class OrderByParameter(
    val field: String,
    val direction: OrderByDirection? = DESC,
) {
    val isAsc: Boolean
        get() = direction == ASC

    val isDesc: Boolean
        get() = direction == DESC

    val isValid: Boolean
        get() = isAsc && !isDesc || isDesc && !isAsc

    class OrderByParameterBuilder internal constructor() {
        private var field: String? = null
        private var direction: OrderByDirection = DESC

        fun build(): OrderByParameter =
            field?.let {
                OrderByParameter(
                    field = field!!,
                    direction = direction,
                )
            } ?: throw IllegalArgumentException("Field cannot be null for an order by parameter!")

        fun field(field: String) {
            this.field = field
        }

        fun direction(direction: OrderByDirection) {
            this.direction = direction
        }
    }

    companion object {
        fun builder(init: OrderByParameterBuilder.() -> Unit): OrderByParameter {
            val builder = OrderByParameterBuilder()
            builder.init()

            return builder.build()
        }
    }
}
