/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

@file:Suppress("unused")

package com.genghis.tools.repository.utils

import com.genghis.tools.repository.models.ModelUtils
import com.genghis.tools.repository.utils.HashAndRange.HashAndRangeBuilder
import io.github.oshai.kotlinlogging.KotlinLogging
import io.vertx.codegen.annotations.DataObject
import io.vertx.codegen.annotations.Fluent
import io.vertx.codegen.json.annotations.JsonGen
import io.vertx.core.json.Json.decodeValue
import io.vertx.core.json.JsonArray
import io.vertx.core.json.JsonObject

@Suppress("unused")
private val logger = KotlinLogging.logger { }

/**
 * This class defines the querypack. A querypack includes the orderByQueue, the map of filterparameters to be performed,
 * and any aggregate function.
 *
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */
sealed class Query(
    open val pageToken: String?,
    open val etag: String?,
    open val orderBy: OrderByParameter?,
    open val filter: Map<String, List<FilterParameter>>,
    open val aggregate: AggregateFunction?,
    open val projections: Set<String>,
    open val autoPaginate: Boolean,
    open val limit: Int,
) {
    fun baseEtagKey() = ModelUtils.returnNewEtag(hashCode().toLong())
}

data class DynamoDBIdentifiers(
    val hash: String? = null,
    val ranges: Set<String> = emptySet(),
    val gsi: String? = null,
    val lsi: String? = null,
)

@DataObject
data class HashAndRange(
    val hash: String? = null,
    val ranges: Set<String> = emptySet(),
) {
    constructor(json: JsonObject) : this(
        hash = json.getString("hash"),
        ranges =
            json
                .getJsonArray("ranges")
                ?.map {
                    it.toString()
                }?.toSet() ?: emptySet(),
    )

    fun toJson(): JsonObject = JsonObject.mapFrom(this)

    fun isEmpty() = hash == null && ranges.isEmpty()

    class HashAndRangeBuilder internal constructor() {
        var hash: String? = null
        var range: String? = null
        var ranges: Set<String?> = emptySet()

        fun build() =
            HashAndRange(
                hash,
                (ranges + range)
                    .filterNotNull()
                    .filterNot { it.isBlank() }
                    .toSet(),
            )
    }
}

fun hashAndRange(init: HashAndRangeBuilder.() -> Unit): HashAndRange {
    val hashAndRange = HashAndRangeBuilder()
    hashAndRange.init()

    return hashAndRange.build()
}

@DataObject
data class DynamoDBQuery internal constructor(
    val identifiers: DynamoDBIdentifiers,
    val consistent: Boolean,
    val noCache: Boolean,
    override val pageToken: String?,
    override val etag: String?,
    override val orderBy: OrderByParameter?,
    override val filter: Map<String, List<FilterParameter>>,
    override val aggregate: AggregateFunction?,
    override val projections: Set<String>,
    override val autoPaginate: Boolean,
    override val limit: Int,
) : Query(pageToken, etag, orderBy, filter, aggregate, projections, autoPaginate, limit) {
    constructor(json: JsonObject) : this(
        identifiers =
            json.getJsonObject("identifiers")?.let { ids ->
                val obj = JsonObject(ids.toString())

                DynamoDBIdentifiers(
                    hash = obj.getString("hash"),
                    ranges =
                        obj
                            .getJsonArray("ranges")
                            ?.map {
                                it.toString()
                            }?.toSet() ?: emptySet(),
                )
            } ?: DynamoDBIdentifiers(),
        consistent = json.getBoolean("consistent") ?: false,
        noCache = json.getBoolean("noCache") ?: false,
        pageToken = json.getString("pageToken"),
        etag = json.getString("etag"),
        orderBy =
            json.getJsonObject("orderBy")?.let {
                decodeValue(it.encode(), OrderByParameter::class.java)
            },
        filter =
            json
                .getJsonObject("params")
                ?.map
                ?.map { entry ->
                    entry.key to
                        JsonArray(entry.toString()).map {
                            decodeValue(it.toString(), FilterParameter::class.java)
                        }
                }?.toMap() ?: emptyMap(),
        aggregate =
            json.getJsonObject("aggregateFunction")?.let {
                decodeValue(it.encode(), AggregateFunction::class.java)
            },
        projections = json.getJsonArray("projections")?.map { it.toString() }?.toSet() ?: emptySet(),
        autoPaginate = json.getBoolean("autoPaginate") ?: false,
        limit = json.getInteger("limit") ?: 50,
    )

    fun toJson(): JsonObject = JsonObject.mapFrom(this)

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (javaClass != other?.javaClass) return false

        other as DynamoDBQuery

        if (identifiers != other.identifiers) return false
        if (consistent != other.consistent) return false
        if (noCache != other.noCache) return false
        if (pageToken != other.pageToken) return false
        if (orderBy != other.orderBy) return false
        if (filter != other.filter) return false
        if (aggregate != other.aggregate) return false
        if (projections != other.projections) return false
        if (autoPaginate != other.autoPaginate) return false
        if (limit != other.limit) return false

        return true
    }

    override fun hashCode(): Int {
        var result = identifiers.hashCode()
        result = 31 * result + consistent.hashCode()
        result = 31 * result + (pageToken?.hashCode() ?: 0)
        result = 31 * result + (orderBy?.hashCode() ?: 0)
        result = 31 * result + filter.hashCode()
        result = 31 * result + (aggregate?.hashCode() ?: 0)
        result = 31 * result + projections.hashCode()
        result = 31 * result + autoPaginate.hashCode()
        result = 31 * result + limit
        return result
    }

    class DynamoDBQuerybuilder internal constructor() : QueryBuilder() {
        private var identifiers: DynamoDBIdentifiers? = null
        private var consistent: Boolean? = null
        private var noCache: Boolean? = null
        private var gsi: String? = null
        var lsi: String? = null
            private set

        override fun build(): DynamoDBQuery =
            DynamoDBQuery(
                identifiers = identifiers ?: DynamoDBIdentifiers(),
                consistent = consistent == true,
                noCache = noCache == true,
                pageToken = pageToken,
                etag = etag,
                orderBy = orderBy,
                filter = params,
                aggregate = aggregateFunction,
                projections = projections,
                autoPaginate = autoPaginate,
                limit = limit,
            )

        fun hashAndRange(hashAndRange: HashAndRange) {
            this.identifiers = (
                identifiers?.copy(
                    hash = hashAndRange.hash,
                    ranges = hashAndRange.ranges,
                )
                    ?: DynamoDBIdentifiers(
                        hash = hashAndRange.hash,
                        ranges = hashAndRange.ranges,
                    )
            )
        }

        fun hashAndRange(init: HashAndRangeBuilder.() -> Unit) {
            val hashAndRange = HashAndRangeBuilder()
            hashAndRange.init()
            val result = hashAndRange.build()

            this.identifiers = (
                identifiers?.copy(
                    hash = result.hash,
                    ranges = result.ranges,
                )
                    ?: DynamoDBIdentifiers(
                        hash = result.hash,
                        ranges = result.ranges,
                    )
            )
        }

        fun noCache(noCache: Boolean) {
            this.noCache = noCache
        }

        fun consistent(consistent: Boolean) {
            this.consistent = consistent
        }

        fun gsi(gsi: String) {
            this.identifiers = (
                identifiers?.copy(
                    gsi = gsi,
                ) ?: DynamoDBIdentifiers(
                    gsi = gsi,
                )
            )
        }

        fun lsi(lsi: String) {
            this.identifiers = (
                identifiers?.copy(
                    lsi = lsi,
                ) ?: DynamoDBIdentifiers(
                    lsi = lsi,
                )
            )
        }
    }

    companion object {
        fun builder(init: DynamoDBQuerybuilder.() -> Unit): DynamoDBQuery {
            val builder = DynamoDBQuerybuilder()
            builder.init()

            return builder.build()
        }
    }
}

sealed class QueryBuilder {
    internal var pageToken: String? = null
    internal var etag: String? = null
    internal var orderBy: OrderByParameter? = null
    internal var params: Map<String, List<FilterParameter>> = emptyMap()
    internal var aggregateFunction: AggregateFunction? = null
    internal var projections: Set<String> = emptySet()
    internal var autoPaginate: Boolean = false
    internal var limit: Int = 50

    abstract fun build(): Query

    fun pageToken(pageToken: String) {
        this.pageToken = pageToken
    }

    fun projections(projections: Set<String>) {
        this.projections = projections
    }

    fun etag(etag: String) {
        this.etag = etag
    }

    fun filter(params: Map<String, List<FilterParameter>>) {
        when {
            params.isEmpty() -> this.params = params
            else -> this.params += params
        }
    }

    fun orderBy(orderBy: OrderByParameter) {
        this.orderBy = orderBy
    }

    fun aggregateFunction(aggregateFunction: AggregateFunction) {
        this.aggregateFunction = aggregateFunction
    }

    fun autoPaginate(autoPaginate: Boolean) {
        this.autoPaginate = autoPaginate
    }

    fun limit(limit: Int) {
        this.limit = limit
    }

    @Fluent
    fun addFilterParameter(
        field: String,
        param: FilterParameter,
    ): QueryBuilder {
        this.params =
            buildMap {
                putAll(this@QueryBuilder.params)
                this@QueryBuilder.params[field]?.let {
                    put(field, it + param)
                } ?: put(field, listOf(param))
            }

        return this
    }

    @Fluent
    fun addFilterParameters(
        field: String,
        parameters: List<FilterParameter>,
    ): QueryBuilder {
        this.params =
            buildMap {
                putAll(this@QueryBuilder.params)
                this@QueryBuilder.params[field]?.let {
                    put(field, it + parameters)
                } ?: put(field, parameters)
            }

        return this
    }
}
