/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

@file:OptIn(DelicateCoroutinesApi::class)

package com.genghis.tools.repository.dynamodb

import com.genghis.tools.repository.dynamodb.operators.DynamoDBAggregates
import com.genghis.tools.repository.dynamodb.operators.DynamoDBCreator
import com.genghis.tools.repository.dynamodb.operators.DynamoDBDeleter
import com.genghis.tools.repository.dynamodb.operators.DynamoDBParameters
import com.genghis.tools.repository.dynamodb.operators.DynamoDBReader
import com.genghis.tools.repository.dynamodb.operators.DynamoDBUpdater
import com.genghis.tools.repository.models.Cacheable
import com.genghis.tools.repository.models.DynamoDBModel
import com.genghis.tools.repository.models.ETagable
import com.genghis.tools.repository.models.Model
import com.genghis.tools.repository.repository.Repository
import com.genghis.tools.repository.repository.cache.CacheManager
import com.genghis.tools.repository.repository.etag.ETagManager
import com.genghis.tools.repository.repository.redis.GenghisRedisClient
import com.genghis.tools.repository.repository.results.ItemListResult
import com.genghis.tools.repository.repository.results.ItemResult
import com.genghis.tools.repository.services.internal.InternalRepositoryService
import com.genghis.tools.repository.utils.DynamoDBQuery
import com.genghis.tools.repository.utils.DynamoDBQuery.DynamoDBQuerybuilder
import com.genghis.tools.repository.utils.HashAndRange
import com.genghis.tools.version.manager.VersionManager
import com.genghis.tools.version.manager.VersionManagerImpl
import io.github.oshai.kotlinlogging.KotlinLogging
import io.netty.channel.ConnectTimeoutException
import io.vertx.core.AsyncResult
import io.vertx.core.Future
import io.vertx.core.Future.failedFuture
import io.vertx.core.Future.succeededFuture
import io.vertx.core.Handler
import io.vertx.core.Promise
import io.vertx.core.Vertx
import io.vertx.core.VertxException
import io.vertx.core.http.HttpClosedException
import io.vertx.core.json.Json
import io.vertx.core.json.JsonArray
import io.vertx.core.json.JsonObject
import io.vertx.kotlin.core.http.httpClientOptionsOf
import io.vertx.kotlin.coroutines.coAwait
import io.vertx.kotlin.coroutines.dispatcher
import kotlinx.coroutines.DelicateCoroutinesApi
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import no.solibo.oss.vertx.client.VertxSdkClient.Companion.withVertx
import org.apache.commons.lang3.ArrayUtils
import software.amazon.awssdk.auth.credentials.AwsBasicCredentials
import software.amazon.awssdk.auth.credentials.StaticCredentialsProvider
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy
import software.amazon.awssdk.enhanced.dynamodb.DynamoDbAsyncTable
import software.amazon.awssdk.enhanced.dynamodb.DynamoDbEnhancedAsyncClient
import software.amazon.awssdk.enhanced.dynamodb.TableSchema.fromBean
import software.amazon.awssdk.enhanced.dynamodb.extensions.annotations.DynamoDbVersionAttribute
import software.amazon.awssdk.enhanced.dynamodb.mapper.annotations.DynamoDbPartitionKey
import software.amazon.awssdk.enhanced.dynamodb.mapper.annotations.DynamoDbSecondaryPartitionKey
import software.amazon.awssdk.enhanced.dynamodb.mapper.annotations.DynamoDbSecondarySortKey
import software.amazon.awssdk.enhanced.dynamodb.mapper.annotations.DynamoDbSortKey
import software.amazon.awssdk.enhanced.dynamodb.model.EnhancedGlobalSecondaryIndex
import software.amazon.awssdk.enhanced.dynamodb.model.EnhancedLocalSecondaryIndex
import software.amazon.awssdk.enhanced.dynamodb.model.GetItemEnhancedRequest
import software.amazon.awssdk.enhanced.dynamodb.model.QueryConditional
import software.amazon.awssdk.enhanced.dynamodb.model.QueryEnhancedRequest
import software.amazon.awssdk.regions.Region
import software.amazon.awssdk.services.dynamodb.DynamoDbAsyncClient
import software.amazon.awssdk.services.dynamodb.DynamoDbAsyncClient.builder
import software.amazon.awssdk.services.dynamodb.model.AttributeValue
import software.amazon.awssdk.services.dynamodb.model.ComparisonOperator
import software.amazon.awssdk.services.dynamodb.model.ProjectionType.ALL
import java.lang.reflect.Field
import java.lang.reflect.Method
import java.lang.reflect.Type
import java.net.URI
import java.text.ParseException
import java.text.SimpleDateFormat
import java.util.Arrays.stream
import java.util.Calendar
import java.util.Date
import java.util.TimeZone
import java.util.concurrent.Callable
import java.util.concurrent.ConcurrentHashMap
import java.util.stream.Collectors.toList
import java.util.stream.IntStream

private val logger = KotlinLogging.logger { }

/**
 * This class defines DynamoDBRepository class. It handles almost all cases of use with the DynamoDB of AWS.
 *
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */
@Suppress("LeakingThis")
open class DynamoDBRepository<E>(
    private val vertx: Vertx = Vertx.currentContext().owner(),
    private val modelType: Class<E>,
    private val appConfig: JsonObject,
    cacheManager: CacheManager<E>? = null,
    eTagManager: ETagManager<E>? = null,
) : Repository<E, DynamoDBQuery, DynamoDBQuerybuilder>,
    InternalRepositoryService<E, DynamoDBQuery>
    where E : DynamoDBModel, E : Model, E : Cacheable, E : ETagable {
    final override var isCached = false
    final override var isEtagEnabled = false

    override val versionManager: VersionManager = VersionManagerImpl()
    override val type: Class<E> = modelType
    private var isVersioned = false
    var hashIdentifier: String? = null
        private set
    var hashField: Method? = null
    private var identifier: String? = null
    var identifierField: Method? = null
        private set
    var paginationIdentifier: String? = null
        private set
    private var paginationField: Method? = null

    private var hasRangeKey: Boolean = false

    lateinit var dynamoDbMapper: DynamoDbEnhancedAsyncClient
        private set
    lateinit var dynamoDbTable: DynamoDbAsyncTable<E>
        private set

    var redisClient: GenghisRedisClient? = null
        private set

    private val parameters: DynamoDBParameters<E>
    private val aggregates: DynamoDBAggregates<E>
    private val creator: DynamoDBCreator<E>
    private val reader: DynamoDBReader<E>
    private val updater: DynamoDBUpdater<E>
    private val deleter: DynamoDBDeleter<E>
    val modelName: String by lazy { type.simpleName }

    final override var etagManager: ETagManager<E>? = eTagManager
        protected set
    var cacheManager: CacheManager<E>? = cacheManager
        protected set

    override val fieldMap = ConcurrentHashMap<String, Field>()
    override val typeMap = ConcurrentHashMap<String, Type>()

    init {
        val tableSuffix =
            appConfig.getString("table-suffix")?.let {
                "-$it"
            } ?: ""
        val tableName = "${appConfig.getString("table")}$tableSuffix"
        setMapper(appConfig, tableName)

        if (etagManager != null) {
            this.etagManager = etagManager
            isEtagEnabled = true
        } else {
            isEtagEnabled = false
        }

        if (cacheManager != null) {
            this.cacheManager = cacheManager
            isCached = true
        } else {
            isCached = false
        }

        isVersioned =
            stream(type.declaredMethods)
                .anyMatch { m ->
                    stream(m.declaredAnnotations)
                        .anyMatch { a -> a is DynamoDbVersionAttribute }
                }
        setHashAndRange(type)
        val gsiKeyMap = setGsiKeys(type)
        if (isCached) isCached = runBlocking { this@DynamoDBRepository.cacheManager!!.initializeCache() }
        this.parameters = DynamoDBParameters(this, hashIdentifier!!, identifier, paginationIdentifier)
        this.aggregates =
            DynamoDBAggregates(type, this, hashIdentifier!!, identifier, this.cacheManager, etagManager)
        this.creator =
            DynamoDBCreator(
                type,
                this,
                hashIdentifier!!,
                hashField,
                identifier,
                identifierField,
                this.cacheManager,
                etagManager,
            )
        this.reader =
            DynamoDBReader(
                type,
                this,
                hashIdentifier!!,
                hashField,
                identifier,
                identifierField,
                paginationIdentifier,
                gsiKeyMap,
                parameters,
                this.cacheManager,
                this.etagManager,
            )
        this.updater = DynamoDBUpdater(this)
        this.deleter =
            DynamoDBDeleter(
                type,
                this,
                hashIdentifier!!,
                hashField,
                identifier,
                identifierField,
                this.cacheManager,
                etagManager,
            )
    }

    private fun setMapper(
        appConfig: JsonObject,
        tableName: String,
    ) {
        val dynamoDBId = appConfig.getString("dynamo_db_iam_id")
        val dynamoDBKey = appConfig.getString("dynamo_db_iam_key")
        val endPoint = fetchEndPoint(appConfig)
        val region = fetchRegion(appConfig)
        val dynamo =
            withVertx(
                builder()
                    .region(Region.of(region))
                    .endpointOverride(URI.create(endPoint))
                    .overrideConfiguration {
                        it.retryStrategy(
                            AwsRetryStrategy
                                .defaultRetryStrategy()
                                .toBuilder()
                                .retryOnExceptionOrCause(VertxException::class.java)
                                .retryOnExceptionOrCause(HttpClosedException::class.java)
                                .retryOnExceptionOrCause(ConnectTimeoutException::class.java)
                                .retryOnExceptionOrCauseInstanceOf(VertxException::class.java)
                                .retryOnExceptionOrCauseInstanceOf(HttpClosedException::class.java)
                                .retryOnExceptionOrCauseInstanceOf(ConnectTimeoutException::class.java)
                                .build(),
                        )
                    },
                httpClientOptionsOf(
                    tcpKeepAlive = true,
                    keepAliveTimeout = 30,
                    http2KeepAliveTimeout = 30,
                    connectTimeout = 30,
                    idleTimeout = 30,
                    writeIdleTimeout = 30,
                    readIdleTimeout = 30,
                ),
                vertx.orCreateContext,
            )!!

        dynamoDbMapper =
            when {
                dynamoDBId != null && dynamoDBKey != null -> {
                    val creds = AwsBasicCredentials.create(dynamoDBId, dynamoDBKey)
                    val statCreds = StaticCredentialsProvider.create(creds)

                    DynamoDbEnhancedAsyncClient
                        .builder()
                        .dynamoDbClient(dynamo.credentialsProvider(statCreds).build())
                        .build()
                }

                else ->
                    DynamoDbEnhancedAsyncClient
                        .builder()
                        .dynamoDbClient(dynamo.build())
                        .build()
            }
        dynamoDbTable = dynamoDbMapper.table(tableName, fromBean(type))
    }

    private fun setHashAndRange(type: Class<E>) {
        val allMethods = getAllMethodsOnType(type)
        hashIdentifier = ""
        identifier = ""
        paginationIdentifier = ""

        stream(allMethods)
            .filter { method ->
                stream(method.annotations)
                    .anyMatch { annotation -> annotation is DynamoDbPartitionKey }
            }.findFirst()
            .ifPresent { method ->
                run {
                    hashIdentifier = stripGet(method.name)
                    hashField = method
                }
            }

        stream(allMethods)
            .filter { method ->
                stream(method.annotations)
                    .anyMatch { annotation -> annotation is DynamoDbSortKey }
            }.findFirst()
            .ifPresent { method ->
                run {
                    identifier = stripGet(method.name)
                    identifierField = method
                }
            }

        stream(allMethods)
            .filter { method ->
                stream(method.annotations)
                    .anyMatch { annotation ->
                        annotation is DynamoDbSecondarySortKey &&
                            annotation.indexNames.any {
                                it.equals(PAGINATION_INDEX, ignoreCase = true)
                            }
                    }
            }.findFirst()
            .ifPresent { method ->
                run {
                    paginationIdentifier = stripGet(method.name)
                    paginationField = method
                }
            }

        hasRangeKey = !identifier.isNullOrBlank()
    }

    @Throws(IllegalArgumentException::class)
    fun getField(fieldName: String): Field {
        try {
            var field: Field? = fieldMap[fieldName]
            if (field != null) return field

            field = type.getDeclaredField(fieldName)
            if (field != null) fieldMap[fieldName] = field
            field!!.isAccessible = true

            return field
        } catch (e: NoSuchFieldException) {
            if (type.superclass != null && type.superclass != java.lang.Object::class.java) {
                return getField(fieldName, type.superclass)
            }

            throw UnknownError("Cannot get field " + fieldName + " from " + type.simpleName + "!")
        } catch (e: NullPointerException) {
            if (type.superclass != null && type.superclass != java.lang.Object::class.java) {
                return getField(fieldName, type.superclass)
            }
            throw UnknownError("Cannot get field " + fieldName + " from " + type.simpleName + "!")
        }
    }

    @Throws(IllegalArgumentException::class)
    private fun getField(
        fieldName: String,
        klazz: Class<*>,
    ): Field {
        try {
            var field: Field? = fieldMap[fieldName]
            if (field != null) return field

            field = klazz.getDeclaredField(fieldName)
            if (field != null) fieldMap[fieldName] = field
            field!!.isAccessible = true

            return field
        } catch (e: NoSuchFieldException) {
            when {
                verifyKlazz(klazz) -> return getField(fieldName, klazz.superclass)
                else -> logger.error(e) { "Cannot get field " + fieldName + " from " + klazz.simpleName + "!" }
            }

            throw UnknownError("Cannot find field!")
        } catch (e: NullPointerException) {
            when {
                verifyKlazz(klazz) -> return getField(fieldName, klazz.superclass)
                else -> logger.error(e) { "Cannot get field " + fieldName + " from " + klazz.simpleName + "!" }
            }

            throw UnknownError("Cannot find field!")
        }
    }

    private fun verifyKlazz(klazz: Class<*>) = klazz.superclass != null && klazz.superclass != Object::class.java

    @Suppress("UNCHECKED_CAST")
    fun <T, O : Any> getFieldAsObject(
        fieldName: String,
        `object`: O,
    ): T {
        try {
            var field: Field? = fieldMap[fieldName]
            if (field != null) return field.get(`object`) as T

            field = `object`.javaClass.getDeclaredField(fieldName)
            if (field != null) fieldMap[fieldName] = field
            field!!.isAccessible = true

            return field.get(`object`) as T
        } catch (e: Exception) {
            if (`object`.javaClass.superclass != null && `object`.javaClass.superclass != java.lang.Object::class.java) {
                return getFieldAsObject(fieldName, `object`, `object`.javaClass.superclass)
            } else {
                logger.error(e) { "Cannot get field " + fieldName + " from " + `object`.javaClass.simpleName + "!" }
            }

            throw UnknownError("Cannot find field: $fieldName!")
        }
    }

    @Suppress("UNCHECKED_CAST")
    private fun <T, O : Any> getFieldAsObject(
        fieldName: String,
        `object`: O,
        klazz: Class<*>,
    ): T {
        try {
            var field: Field? = fieldMap[fieldName]
            if (field != null) return field.get(`object`) as T

            field = `object`.javaClass.getDeclaredField(fieldName)
            if (field != null) fieldMap[fieldName] = field
            field!!.isAccessible = true

            return field.get(`object`) as T
        } catch (e: Exception) {
            if (klazz.superclass != null && klazz.superclass != java.lang.Object::class.java) {
                return getFieldAsObject(fieldName, `object`, klazz.superclass)
            } else {
                logger.error(e) { "Cannot get field " + fieldName + " from " + klazz.simpleName + "!" }
            }

            throw UnknownError("Cannot find field: $fieldName!")
        }
    }

    fun <T : Any> getFieldAsString(
        fieldName: String,
        `object`: T,
    ): String? {
        if (logger.isTraceEnabled()) {
            logger.trace { "Getting " + fieldName + " from " + `object`.javaClass.simpleName }
        }

        try {
            var field: Field? = fieldMap[fieldName]

            if (field != null) {
                field.isAccessible = true

                return field.get(`object`).toString()
            }

            field = type.getDeclaredField(fieldName)
            if (field != null) fieldMap[fieldName] = field
            field!!.isAccessible = true

            val fieldObject = field.get(`object`)

            return fieldObject.toString()
        } catch (e: Exception) {
            return when {
                type.superclass != null && type.superclass != java.lang.Object::class.java ->
                    getFieldAsString(fieldName, `object`, type.superclass)

                else -> {
                    logger.error(e) { "Cannot get " + fieldName + " as string from: " + Json.encodePrettily(`object`) }

                    throw UnknownError("Cannot find field: $fieldName!")
                }
            }
        }
    }

    private fun <T> getFieldAsString(
        fieldName: String,
        `object`: T,
        klazz: Class<*>,
    ): String? {
        if (logger.isTraceEnabled()) {
            logger.trace { "Getting " + fieldName + " from " + klazz.simpleName }
        }

        try {
            var field: Field? = fieldMap[fieldName]

            if (field != null) {
                field.isAccessible = true

                return field.get(`object`)?.toString()
            }

            field = klazz.getDeclaredField(fieldName)
            if (field != null) fieldMap[fieldName] = field
            field!!.isAccessible = true

            val fieldObject = field.get(`object`)

            return fieldObject?.toString()
        } catch (e: Exception) {
            return when {
                klazz.superclass != null && klazz.superclass != java.lang.Object::class.java ->
                    getFieldAsString(fieldName, `object`, klazz.superclass)

                else -> {
                    logger.error(e) {
                        "Cannot get " + fieldName + " as string from: " + Json.encodePrettily(`object`) + ", klazzwise!"
                    }

                    throw UnknownError("Cannot find field: $fieldName!")
                }
            }
        }
    }

    @Throws(IllegalArgumentException::class)
    fun checkAndGetField(fieldName: String): Field {
        try {
            var field: Field? = fieldMap[fieldName]

            if (field == null) {
                field = type.getDeclaredField(fieldName)

                if (field != null) fieldMap[fieldName] = field
            }

            var fieldType: Type? = typeMap[fieldName]

            if (fieldType == null) {
                fieldType = field!!.type

                if (fieldType != null) typeMap[fieldName] = fieldType
            }

            when {
                fieldType === java.lang.Long::class.java ||
                    fieldType === java.lang.Integer::class.java ||
                    fieldType === java.lang.Double::class.java ||
                    fieldType === java.lang.Float::class.java ||
                    fieldType === java.lang.Short::class.java ||
                    fieldType === java.lang.Long::class.javaPrimitiveType ||
                    fieldType === java.lang.Integer::class.javaPrimitiveType ||
                    fieldType === java.lang.Double::class.javaPrimitiveType ||
                    fieldType === java.lang.Float::class.javaPrimitiveType ||
                    fieldType === java.lang.Short::class.javaPrimitiveType -> {
                    field!!.isAccessible = true

                    return field
                }

                else -> throw IllegalArgumentException("Not an incrementable field: $fieldName!")
            }
        } catch (e: NoSuchFieldException) {
            return when {
                type.superclass != null && type.superclass != java.lang.Object::class.java ->
                    checkAndGetField(fieldName, type.superclass)

                else -> throw IllegalArgumentException(
                    "Field does not exist: $fieldName!",
                )
            }
        } catch (e: NullPointerException) {
            return when {
                type.superclass != null && type.superclass != java.lang.Object::class.java ->
                    checkAndGetField(fieldName, type.superclass)

                else -> throw IllegalArgumentException(
                    "Field does not exist: $fieldName!",
                )
            }
        }
    }

    @Throws(IllegalArgumentException::class)
    private fun checkAndGetField(
        fieldName: String,
        klazz: Class<*>,
    ): Field {
        try {
            var field: Field? = fieldMap[fieldName]

            if (field == null) {
                field = klazz.getDeclaredField(fieldName)

                if (field != null) fieldMap[fieldName] = field
            }

            var fieldType: Type? = typeMap[fieldName]

            if (fieldType == null) {
                fieldType = field!!.type

                if (fieldType != null) typeMap[fieldName] = fieldType
            }

            when {
                fieldType === java.lang.Long::class.java ||
                    fieldType === java.lang.Integer::class.java ||
                    fieldType === java.lang.Double::class.java ||
                    fieldType === java.lang.Float::class.java ||
                    fieldType === java.lang.Short::class.java ||
                    fieldType === java.lang.Long::class.javaPrimitiveType ||
                    fieldType === java.lang.Integer::class.javaPrimitiveType ||
                    fieldType === java.lang.Double::class.javaPrimitiveType ||
                    fieldType === java.lang.Float::class.javaPrimitiveType ||
                    fieldType === java.lang.Short::class.javaPrimitiveType -> {
                    field!!.isAccessible = true

                    return field
                }

                else -> throw IllegalArgumentException("Not an incrementable field: $fieldName!")
            }
        } catch (e: NoSuchFieldException) {
            return when {
                klazz.superclass != null && klazz.superclass != java.lang.Object::class.java ->
                    checkAndGetField(fieldName, klazz)

                else -> throw IllegalArgumentException(
                    "Field does not exist: $fieldName!",
                )
            }
        }
    }

    fun getAlternativeIndexIdentifier(indexName: String): String? {
        val identifier = arrayOfNulls<String>(1)

        stream(type.methods)
            .filter { method ->
                stream(method.annotations)
                    .anyMatch { annotation ->
                        annotation is DynamoDbSecondarySortKey &&
                            annotation.indexNames.any {
                                it.equals(indexName, ignoreCase = true)
                            }
                    }
            }.findFirst()
            .ifPresent { method -> identifier[0] = stripGet(method.name) }

        return identifier[0]
    }

    fun getAlternativeIndexIdentifierReturnType(indexName: String): Class<*>? {
        val identifier = arrayOfNulls<Class<*>>(1)

        stream(type.methods)
            .filter { method ->
                stream(method.annotations)
                    .anyMatch { annotation ->
                        annotation is DynamoDbSecondarySortKey &&
                            annotation.indexNames.any {
                                it.equals(indexName, ignoreCase = true)
                            }
                    }
            }.findFirst()
            .ifPresent { method -> identifier[0] = method.returnType }

        return identifier[0]
    }

    fun <T : Any> getIndexValue(
        alternateIndex: String,
        `object`: T,
    ): AttributeValue {
        try {
            var field: Field? = fieldMap[alternateIndex]

            if (field == null) {
                field = `object`.javaClass.getDeclaredField(alternateIndex)

                if (field != null) fieldMap[alternateIndex] = field
            }

            field?.isAccessible = true

            var fieldType: Type? = typeMap[alternateIndex]

            if (fieldType == null) {
                fieldType = field?.type

                if (fieldType != null) typeMap[alternateIndex] = fieldType
            }

            return when {
                fieldType === java.util.Date::class.java -> {
                    val dateObject = field?.get(`object`) as Date

                    createAttributeValue(alternateIndex, dateObject.time.toString())
                }

                else ->
                    createAttributeValue(
                        alternateIndex,
                        field?.get(`object`).toString(),
                    )
            }
        } catch (e: NoSuchFieldException) {
            if (`object`.javaClass.superclass != null && `object`.javaClass.superclass != java.lang.Object::class.java) {
                return getIndexValue(alternateIndex, `object`, `object`.javaClass.superclass)
            }
        } catch (e: NullPointerException) {
            if (`object`.javaClass.superclass != null && `object`.javaClass.superclass != java.lang.Object::class.java) {
                return getIndexValue(alternateIndex, `object`, `object`.javaClass.superclass)
            }
        } catch (e: IllegalAccessException) {
            if (`object`.javaClass.superclass != null && `object`.javaClass.superclass != java.lang.Object::class.java) {
                return getIndexValue(alternateIndex, `object`, `object`.javaClass.superclass)
            }
        }

        throw UnknownError("Cannot find field!")
    }

    private fun <T> getIndexValue(
        alternateIndex: String,
        `object`: T,
        klazz: Class<*>,
    ): AttributeValue {
        try {
            var field: Field? = fieldMap[alternateIndex]

            if (field == null) {
                field = klazz.getDeclaredField(alternateIndex)

                if (field != null) fieldMap[alternateIndex] = field
            }

            field!!.isAccessible = true

            var fieldType: Type? = typeMap[alternateIndex]

            if (fieldType == null) {
                fieldType = field.type

                if (fieldType != null) typeMap[alternateIndex] = fieldType
            }

            return when {
                fieldType === java.util.Date::class.java -> {
                    val dateObject = field.get(`object`) as Date

                    createAttributeValue(alternateIndex, dateObject.time.toString())
                }

                else ->
                    createAttributeValue(
                        alternateIndex,
                        field.get(`object`).toString(),
                    )
            }
        } catch (e: NoSuchFieldException) {
            if (klazz.superclass != null && klazz.superclass != java.lang.Object::class.java) {
                return getIndexValue(alternateIndex, `object`, klazz.superclass)
            }
        } catch (e: NullPointerException) {
            if (klazz.superclass != null && klazz.superclass != java.lang.Object::class.java) {
                return getIndexValue(alternateIndex, `object`, klazz.superclass)
            }
        } catch (e: IllegalAccessException) {
            if (klazz.superclass != null && klazz.superclass != java.lang.Object::class.java) {
                return getIndexValue(alternateIndex, `object`, klazz.superclass)
            }
        }

        throw UnknownError("Cannot find field!")
    }

    @JvmOverloads
    fun createAttributeValue(
        fieldName: String?,
        valueAsString: String,
        modifier: ComparisonOperator? = null,
    ): AttributeValue {
        if (fieldName == null) throw IllegalArgumentException("Fieldname cannot be null!")

        val field = getField(fieldName)
        var fieldType: Type? = typeMap[fieldName]

        if (fieldType == null) {
            fieldType = field.type

            if (fieldType != null) typeMap[fieldName] = fieldType
        }

        val fieldTypeName = fieldType?.typeName

        when {
            fieldTypeName.equals("String", ignoreCase = true) ||
                fieldTypeName.equals("java.lang.String", ignoreCase = true) -> return AttributeValue.fromS(valueAsString)
            fieldTypeName.equals("Int", ignoreCase = true) ||
                fieldTypeName.equals("java.lang.Int", ignoreCase = true) ||
                fieldTypeName.equals("Integer", ignoreCase = true) ||
                fieldTypeName.equals("java.lang.Integer", ignoreCase = true) ||
                fieldTypeName.equals("Double", ignoreCase = true) ||
                fieldTypeName.equals("java.lang.Double", ignoreCase = true) ||
                fieldTypeName.equals("Long", ignoreCase = true) ||
                fieldTypeName.equals("java.lang.Long", ignoreCase = true) -> {
                try {
                    return when {
                        fieldTypeName.equals("Int", ignoreCase = true) ||
                            fieldTypeName.equals("Integer", ignoreCase = true) ||
                            fieldTypeName.equals("java.lang.Integer", ignoreCase = true) -> {
                            var value = Integer.parseInt(valueAsString)

                            if (modifier == ComparisonOperator.GE) value -= 1
                            if (modifier == ComparisonOperator.LE) value += 1

                            return AttributeValue.fromN(value.toString())
                        }
                        fieldTypeName.equals("Double", ignoreCase = true) ||
                            fieldTypeName.equals("java.lang.Double", ignoreCase = true) -> {
                            var value = java.lang.Double.parseDouble(valueAsString)

                            if (modifier == ComparisonOperator.GE) value -= 0.1
                            if (modifier == ComparisonOperator.LE) value += 0.1

                            return AttributeValue.fromN(value.toString())
                        }
                        fieldTypeName.equals("Long", ignoreCase = true) ||
                            fieldTypeName.equals("java.lang.Long", ignoreCase = true) -> {
                            var value = java.lang.Long.parseLong(valueAsString)

                            if (modifier == ComparisonOperator.GE) value -= 1
                            if (modifier == ComparisonOperator.LE) value += 1

                            return AttributeValue.fromN(value.toString())
                        }
                        else -> AttributeValue.fromN(valueAsString)
                    }
                } catch (nfe: NumberFormatException) {
                    logger.error(nfe) { "Cannot recreate attribute!" }
                }

                return AttributeValue.fromN(valueAsString)
            }

            fieldTypeName.equals("Boolean", ignoreCase = true) ||
                fieldTypeName.equals("java.lang.Boolean", ignoreCase = true) -> {
                if (valueAsString.equals("true", ignoreCase = true)) {
                    return AttributeValue.fromN("1")
                } else if (valueAsString.equals("false", ignoreCase = true)) {
                    return AttributeValue.fromN("0")
                }

                try {
                    val boolValue = Integer.parseInt(valueAsString)

                    if (boolValue == 1 || boolValue == 0) {
                        return AttributeValue.fromN(boolValue.toString())
                    }

                    throw UnknownError("Cannot create AttributeValue: $fieldName!")
                } catch (nfe: NumberFormatException) {
                    logger.error(nfe) { "Cannot receate attribute!" }
                }
            }

            else -> return when {
                fieldTypeName.equals("Date", ignoreCase = true) ||
                    fieldTypeName.equals("java.util.Date", ignoreCase = true) ->
                    try {
                        if (logger.isDebugEnabled()) {
                            logger.debug { "Date received: $valueAsString" }
                        }

                        val date: Date =
                            try {
                                Date(java.lang.Long.parseLong(valueAsString))
                            } catch (nfe: NumberFormatException) {
                                val df1 = SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSX")
                                df1.parse(valueAsString)
                            }

                        val calendar = Calendar.getInstance()
                        calendar.time = date

                        if (modifier == ComparisonOperator.LE) calendar.add(Calendar.MILLISECOND, 1)
                        if (modifier == ComparisonOperator.GE) calendar.time = Date(calendar.time.time - 1)

                        val df2 = SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSX")
                        df2.timeZone = TimeZone.getTimeZone("Z")
                        if (logger.isDebugEnabled()) {
                            logger.debug { "DATE IS: " + df2.format(calendar.time) }
                        }

                        AttributeValue.fromS(df2.format(calendar.time))
                    } catch (e: ParseException) {
                        AttributeValue.fromS(valueAsString)
                    }

                else -> AttributeValue.fromS(valueAsString)
            }
        }

        throw UnknownError("Cannot create attributevalue: $fieldName!")
    }

    suspend fun fetchNewestRecord(
        hash: String,
        range: String?,
    ): E? =
        when {
            !hasRangeKey() || range != null -> {
                if (logger.isDebugEnabled()) {
                    logger.debug { "Loading newest with ${if (hasRangeKey()) "range" else "hash"}!" }
                }

                val fetchFuture = Promise.promise<E?>()

                dynamoDbTable
                    .getItem(
                        GetItemEnhancedRequest
                            .builder()
                            .consistentRead(true)
                            .key {
                                it.partitionValue(hash)

                                if (hasRangeKey()) it.sortValue(range)
                            }.build(),
                    ).whenComplete { t, u ->
                        when (u) {
                            null -> fetchFuture.complete(t)
                            else -> fetchFuture.fail(u)
                        }
                    }

                fetchFuture.future().coAwait()
            }

            else ->
                runCatching {
                    if (logger.isDebugEnabled()) {
                        logger.debug { "Loading newest by hash query!" }
                    }

                    val timeBefore = System.currentTimeMillis()
                    val items =
                        dynamoDbTable.query(
                            QueryEnhancedRequest
                                .builder()
                                .consistentRead(true)
                                .limit(1)
                                .queryConditional(
                                    QueryConditional.keyEqualTo {
                                        it.partitionValue(hash)
                                    },
                                ).build(),
                        )

                    if (logger.isDebugEnabled()) {
                        logger.debug { "Results received in: " + (System.currentTimeMillis() - timeBefore) + " ms" }
                    }
                    val result = mutableListOf<E>()
                    val fetchFuture = Promise.promise<E?>()

                    items.items().subscribe { result.add(it) }.whenComplete { _, u ->
                        when (u) {
                            null -> fetchFuture.complete(if (result.isNotEmpty()) result.first() else null)
                            else -> fetchFuture.fail(u)
                        }
                    }

                    fetchFuture.future().coAwait()
                }.onFailure {
                    logger.error(it) { "Error fetching newest!" }
                }.getOrThrow()
        }

    @Throws(IllegalArgumentException::class)
    override fun incrementField(
        record: E,
        fieldName: String,
    ): suspend E.() -> Unit =
        {
            updater.incrementField(this, fieldName)
        }

    @Throws(IllegalArgumentException::class)
    override fun decrementField(
        record: E,
        fieldName: String,
    ): suspend E.() -> Unit =
        {
            updater.decrementField(this, fieldName)
        }

    override suspend fun show(query: DynamoDBQuery): ItemResult<E> = reader.show(query)

    override suspend fun show(builder: DynamoDBQuerybuilder.() -> Unit): ItemResult<E> {
        val queryBuilder = DynamoDBQuerybuilder()
        queryBuilder.builder()

        val queryResult = queryBuilder.build()

        return show(queryResult)
    }

    override suspend fun index(query: DynamoDBQuery): ItemListResult<E> = reader.index(query)

    override suspend fun index(builder: DynamoDBQuerybuilder.() -> Unit): ItemListResult<E> {
        val queryBuilder = DynamoDBQuerybuilder()
        queryBuilder.builder()

        val queryResult = queryBuilder.build()

        return index(queryResult)
    }

    override suspend fun aggregation(query: DynamoDBQuery): String = aggregates.aggregation(query)

    override suspend fun aggregation(builder: DynamoDBQuerybuilder.() -> Unit): String {
        val queryBuilder = DynamoDBQuerybuilder()
        queryBuilder.builder()

        val queryResult = queryBuilder.build()

        return aggregation(queryResult)
    }

    override suspend fun doWrite(
        create: Boolean,
        records: Map<E, suspend E.(E) -> Unit>,
    ): List<E> = creator.doWrite(create, records)

    override suspend fun doDelete(
        hash: String?,
        ranges: List<String>?,
    ): List<E> = deleter.doDelete(hash, ranges?.toSet())

    override fun remoteCreate(record: E): Future<E> {
        val promise = Promise.promise<E>()

        GlobalScope.launch(vertx.dispatcher()) {
            runCatching {
                create(record)
            }.onFailure {
                promise.fail(it)
            }.onSuccess {
                promise.complete(it.item)
            }
        }

        return promise.future()
    }

    override fun remoteRead(query: DynamoDBQuery): Future<E> {
        val promise = Promise.promise<E>()

        GlobalScope.launch(vertx.dispatcher()) {
            runCatching {
                show(query)
            }.onFailure {
                promise.fail(it)
            }.onSuccess {
                promise.complete(it.item)
            }
        }

        return promise.future()
    }

    override fun remoteIndex(query: DynamoDBQuery?): Future<List<E>> {
        val promise = Promise.promise<List<E>>()

        GlobalScope.launch(vertx.dispatcher()) {
            runCatching {
                val result = index(query ?: DynamoDBQuery.builder {})
                val items =
                    vertx
                        .executeBlocking(
                            Callable {
                                result.itemList.items
                            },
                        ).coAwait()

                promise.tryComplete(items)
            }.onFailure {
                promise.tryFail(it)
            }
        }

        return promise.future()
    }

    override fun remoteUpdate(record: E): Future<E> {
        val promise = Promise.promise<E>()

        GlobalScope.launch(vertx.dispatcher()) {
            runCatching {
                show(
                    DynamoDBQuery.builder {
                        hashAndRange {
                            hash = record.hash(hashField)
                            range = record.range(identifierField)
                        }
                    },
                )
            }.onFailure {
                promise.tryFail(it)
            }.onSuccess {
                runCatching {
                    update(record)
                }.onFailure {
                    promise.tryFail(it)
                }.onSuccess {
                    promise.complete(it.item)
                }
            }
        }

        return promise.future()
    }

    override fun remoteDelete(hashAndRange: HashAndRange): Future<E> {
        val promise = Promise.promise<E>()

        GlobalScope.launch(vertx.dispatcher()) {
            runCatching {
                delete(hashAndRange)
            }.onFailure {
                promise.fail(it)
            }.onSuccess {
                promise.complete(it.item)
            }
        }

        return promise.future()
    }

    fun buildEventbusProjections(projectionArray: JsonArray?): Array<String> {
        if (projectionArray == null) return arrayOf()

        val projections =
            projectionArray
                .stream()
                .map { it.toString() }
                .collect(toList())

        val projectionArrayStrings = arrayOfNulls<String>(projections?.size ?: 0)

        if (projections != null) {
            IntStream.range(0, projections.size).forEach { i -> projectionArrayStrings[i] = projections[i] }
        }

        return projectionArrayStrings.requireNoNulls()
    }

    fun hasRangeKey(): Boolean = hasRangeKey

    companion object {
        const val PAGINATION_INDEX = "PAGINATION_INDEX"

        inline fun <reified T> create(
            vertx: Vertx = Vertx.currentContext().owner(),
            appConfig: JsonObject,
            cacheManager: CacheManager<T>? = null,
            eTagManager: ETagManager<T>? = null,
        ) where T : DynamoDBModel, T : Model, T : Cacheable, T : ETagable =
            DynamoDBRepository(
                vertx = vertx,
                modelType = T::class.java,
                appConfig = appConfig,
                cacheManager = cacheManager,
                eTagManager = eTagManager,
            )

        private fun fetchEndPoint(appConfig: JsonObject?): String {
            val endPoint: String =
                when (
                    val config =
                        appConfig ?: if (Vertx.currentContext() == null) null else Vertx.currentContext().config()
                ) {
                    null -> "http://localhost:8001"
                    else -> config.getString("dynamo_endpoint")
                }

            return endPoint
        }

        private fun fetchRegion(appConfig: JsonObject?): String {
            val config = appConfig ?: if (Vertx.currentContext() == null) null else Vertx.currentContext().config()
            var region: String?

            when (config) {
                null -> region = "eu-west-1"
                else -> {
                    region = config.getString("dynamo_signing_region")
                    if (region == null) region = "eu-west-1"
                }
            }

            return region
        }

        private fun <E> setGsiKeys(type: Class<E>): Map<String, JsonObject> {
            val allMethods = getAllMethodsOnType(type)
            val gsiMap = ConcurrentHashMap<String, JsonObject>()

            stream(allMethods).forEach { method ->
                if (stream(method.declaredAnnotations)
                        .anyMatch { annotation -> annotation is DynamoDbSecondaryPartitionKey }
                ) {
                    val hashName =
                        method
                            .getDeclaredAnnotation(DynamoDbSecondaryPartitionKey::class.java)
                            .indexNames
                            .first()
                    val hash = stripGet(method.name)
                    val range = arrayOfNulls<String>(1)

                    if (hashName != "") {
                        stream(allMethods).forEach { rangeMethod ->
                            if (stream(rangeMethod.declaredAnnotations)
                                    .anyMatch { annotation -> annotation is DynamoDbSecondarySortKey }
                            ) {
                                rangeMethod
                                    .getDeclaredAnnotation(DynamoDbSecondarySortKey::class.java)
                                    .indexNames
                                    .firstOrNull {
                                        it == hashName
                                    }?.let {
                                        range[0] = stripGet(rangeMethod.name)
                                    }
                            }
                        }

                        val hashKeyObject =
                            JsonObject()
                                .put("hash", hash)

                        if (range[0] != null) hashKeyObject.put("range", range[0])

                        gsiMap[hashName] = hashKeyObject

                        logger.debug { "Detected GSI: " + hashName + " : " + hashKeyObject.encodePrettily() }
                    }
                }
            }

            return gsiMap
        }

        private fun getAllMethodsOnType(klazz: Class<*>): Array<Method> {
            val methods = klazz.declaredMethods

            return if (klazz.superclass != null && klazz.superclass != java.lang.Object::class.java) {
                ArrayUtils.addAll(methods, *getAllMethodsOnType(klazz.superclass))
            } else {
                methods
            }
        }

        fun stripGet(string: String): String {
            val newString = string.replace("get", "")
            val c = newString.toCharArray()
            c[0] = c[0] + 32

            return String(c)
        }

        fun initializeDynamoDb(
            vertx: Vertx = Vertx.currentContext().owner(),
            appConfig: JsonObject,
            collectionMap: Map<String, Class<*>>,
            resultHandler: Handler<AsyncResult<Void>>,
        ) {
            if (logger.isDebugEnabled()) {
                logger.debug { "Initializing DynamoDB" }
            }

            try {
                silenceDynamoDBLoggers()
                val futures = ArrayList<Promise<*>>()

                collectionMap.forEach { (_, v) -> futures.add(initialize(vertx, appConfig, v)) }

                Future.all<Any>(futures.map { it.future() }).andThen { res ->
                    if (logger.isDebugEnabled()) {
                        logger.debug { "DynamoDB Ready" }
                    }

                    when {
                        res.failed() -> resultHandler.handle(failedFuture(res.cause()))
                        else -> resultHandler.handle(succeededFuture())
                    }
                }
            } catch (e: Exception) {
                logger.error(e) { "Unable to initialize!" }
            }
        }

        private fun silenceDynamoDBLoggers() {
            java.util.logging.Logger
                .getLogger("com.amazonaws")
                .level = java.util.logging.Level.WARNING
        }

        private fun initialize(
            vertx: Vertx,
            appConfig: JsonObject,
            type: Class<*>,
        ): Promise<Void> {
            val future = Promise.promise<Void>()

            val tableSuffix =
                appConfig.getString("table-suffix")?.let {
                    "-$it"
                } ?: ""
            val tableName = "${appConfig.getString("table")}$tableSuffix"
            val (dynamoDBId, dynamoDBKey) = iam(appConfig)
            val endPoint = fetchEndPoint(appConfig)
            val region = fetchRegion(appConfig)
            val dynamo =
                withVertx(
                    builder()
                        .region(Region.of(region))
                        .endpointOverride(URI.create(endPoint)),
                    vertx.orCreateContext,
                )!!
            val client =
                when {
                    dynamoDBId != null && dynamoDBKey != null -> {
                        val creds = AwsBasicCredentials.create(dynamoDBId, dynamoDBKey)
                        val statCreds = StaticCredentialsProvider.create(creds)

                        dynamo.credentialsProvider(statCreds).build()
                    }

                    else -> dynamo.build()
                }
            val dynamoDbMapper =
                DynamoDbEnhancedAsyncClient
                    .builder()
                    .dynamoDbClient(client)
                    .build()

            @Suppress("UNCHECKED_CAST")
            initialize(
                client = client,
                mapper = dynamoDbMapper.table(tableName, fromBean(type)) as DynamoDbAsyncTable<Class<*>>,
                tableName = tableName,
                gsi = appConfig.getJsonArray("GSI"),
                lsi = appConfig.getJsonArray("LSI"),
                resultHandler = future,
            )

            return future
        }

        private fun iam(appConfig: JsonObject): Pair<String?, String?> {
            val dynamoDBId = appConfig.getString("dynamo_db_iam_id")
            val dynamoDBKey = appConfig.getString("dynamo_db_iam_key")

            return Pair(dynamoDBId, dynamoDBKey)
        }

        private fun initialize(
            client: DynamoDbAsyncClient,
            mapper: DynamoDbAsyncTable<Class<*>>,
            tableName: String,
            gsi: JsonArray?,
            lsi: JsonArray?,
            resultHandler: Promise<Void>,
        ) {
            mapper
                .createTable { builder ->
                    gsi?.let { gsi ->
                        builder.globalSecondaryIndices(
                            gsi.map { it.toString() }.map {
                                EnhancedGlobalSecondaryIndex
                                    .builder()
                                    .indexName(it)
                                    .projection { p -> p.projectionType(ALL) }
                                    .build()
                            },
                        )
                    }
                    lsi?.let { lsi ->
                        builder.localSecondaryIndices(
                            lsi.map { it.toString() }.map {
                                EnhancedLocalSecondaryIndex
                                    .builder()
                                    .indexName(it)
                                    .projection { p -> p.projectionType(ALL) }
                                    .build()
                            },
                        )
                    }
                }.whenComplete { _, u ->
                    when (u) {
                        null -> {
                            if (logger.isDebugEnabled()) {
                                logger.debug { "Table creation for: $tableName is success!" }
                            }

                            client.waiter().waitUntilTableExists { b -> b.tableName(tableName) }.whenComplete { _, u2 ->
                                when (u2) {
                                    null -> {
                                        if (logger.isDebugEnabled()) {
                                            logger.debug { "Table wait for: $tableName is success!" }
                                        }

                                        resultHandler.handle(succeededFuture())
                                    }

                                    else -> resultHandler.handle(failedFuture(u2))
                                }
                            }
                        }

                        else -> resultHandler.handle(succeededFuture())
                    }
                }
        }
    }
}

fun DynamoDBModel.hash(field: Method?) = field?.let { it(this)?.toString() }

fun DynamoDBModel.range(field: Method?) = field?.let { it(this)?.toString() }
