/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package com.genghis.tools.repository.repository.cache

import com.genghis.tools.repository.models.Cacheable
import com.genghis.tools.repository.models.Model
import com.genghis.tools.repository.utils.ItemList
import io.vertx.core.Promise
import java.util.function.Function
import java.util.function.Supplier

/**
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */
interface CacheManager<E> where E : Cacheable, E : Model {
    suspend fun isObjectCacheAvailable(): Boolean

    suspend fun isItemListCacheAvailable(): Boolean

    suspend fun isAggregationCacheAvailable(): Boolean

    suspend fun initializeCache(): Boolean

    suspend fun checkObjectCache(
        cacheId: String,
        typeName: String,
    ): E

    suspend fun checkItemListCache(
        cacheId: String,
        projections: Set<String>,
        typeName: String,
    ): ItemList<E>

    suspend fun checkAggregationCache(
        cacheKey: String,
        typeName: String,
    ): String

    suspend fun replaceCache(
        records: List<E>,
        shortCacheIdSupplier: Function<E, String>,
        cacheIdSupplier: Function<E, String>,
    )

    suspend fun replaceObjectCache(
        cacheId: String,
        item: E,
        future: Promise<E>,
        projections: Set<String>,
    )

    suspend fun replaceItemListCache(
        content: String,
        cacheIdSupplier: Supplier<String>,
    ): Boolean

    suspend fun replaceAggregationCache(
        content: String,
        cacheIdSupplier: Supplier<String>,
    ): Boolean

    suspend fun purgeCache(
        records: List<E>,
        cacheIdSupplier: (E) -> String,
    )
}
