/*
 * MIT License
 *
 * Copyright (c) 2017 Anders Mikkelsen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package com.genghis.tools.repository.repository.etag

import com.genghis.tools.repository.models.ETagable
import com.genghis.tools.repository.models.Model
import com.genghis.tools.repository.utils.ItemList
import io.github.oshai.kotlinlogging.KotlinLogging
import io.vertx.core.AsyncResult
import io.vertx.core.Future
import io.vertx.core.Future.failedFuture
import io.vertx.core.Future.succeededFuture
import io.vertx.core.Handler
import io.vertx.core.Promise
import io.vertx.core.Vertx
import io.vertx.core.shareddata.AsyncMap
import io.vertx.core.shareddata.LocalMap
import io.vertx.kotlin.coroutines.awaitResult
import io.vertx.kotlin.coroutines.coAwait
import java.lang.Boolean.TRUE

private val logger = KotlinLogging.logger { }

/**
 * The InMemoryETagManagerImpl contains the logic for setting, removing, and replace etags.
 *
 * @author Anders Mikkelsen
 * @version 17.11.2017
 */
class InMemoryETagManagerImpl<E>(
    private val vertx: Vertx,
    private val TYPE: Class<E>,
) : ETagManager<E> where E : ETagable, E : Model {
    private val localObjectMap: LocalMap<String, String>
        get() = vertx.sharedData().getLocalMap(OBJECT_ETAG_MAP)

    private fun getLocalObjectMap(etagKeyBase: String): LocalMap<String, String> = vertx.sharedData().getLocalMap(etagKeyBase)

    private fun getLocalItemListMap(itemListHashKey: String): LocalMap<String, String> = vertx.sharedData().getLocalMap(itemListHashKey)

    private fun getClusteredObjectMap(resultHandler: Handler<AsyncResult<AsyncMap<String, String>>>) {
        if (!vertx.isClustered) throw IllegalStateException("Vertx is not clustered!")
        vertx.sharedData().getClusterWideMap<String, String>(OBJECT_ETAG_MAP).onComplete(resultHandler)
    }

    private fun getClusteredObjectMap(
        etagKeyBase: String,
        resultHandler: Handler<AsyncResult<AsyncMap<String, String>>>,
    ) {
        if (!vertx.isClustered) throw IllegalStateException("Vertx is not clustered!")
        vertx.sharedData().getClusterWideMap<String, String>(etagKeyBase).onComplete(resultHandler)
    }

    private fun getClusteredItemListMap(
        itemListHashKey: String,
        resultHandler: Handler<AsyncResult<AsyncMap<String, String>>>,
    ) {
        if (!vertx.isClustered) throw IllegalStateException("Vertx is not clustered!")
        vertx.sharedData().getClusterWideMap<String, String>(itemListHashKey).onComplete(resultHandler)
    }

    override suspend fun removeProjectionsEtags(hash: Int): Boolean {
        val etagKeyBase = TYPE.simpleName + "_" + hash + "/projections"

        return awaitResult { handler ->
            when {
                vertx.isClustered -> getClusteredObjectMap(etagKeyBase) { clearClusteredMap(handler, it) }
                else -> {
                    getLocalItemListMap(etagKeyBase).clear()

                    handler.handle(succeededFuture(TRUE))
                }
            }
        }
    }

    override suspend fun destroyEtags(hash: Int): Boolean {
        val etagItemListHashKey = TYPE.simpleName + "_" + hash + "_" + "itemListEtags"

        return awaitResult { handler ->
            when {
                vertx.isClustered ->
                    getClusteredItemListMap(
                        etagItemListHashKey,
                    ) {
                        clearClusteredMap(handler, it)
                    }

                else -> {
                    getLocalItemListMap(etagItemListHashKey).clear()

                    handler.handle(succeededFuture(TRUE))
                }
            }
        }
    }

    private fun clearClusteredMap(
        resultHandler: Handler<AsyncResult<Boolean>>,
        mapRes: AsyncResult<AsyncMap<String, String>>,
    ) {
        when {
            mapRes.failed() -> resultHandler.handle(failedFuture(mapRes.cause()))
            else ->
                mapRes.result().clear().onComplete { clearRes ->
                    if (clearRes.failed()) {
                        resultHandler.handle(failedFuture(clearRes.cause()))
                    } else {
                        resultHandler.handle(succeededFuture(TRUE))
                    }
                }
        }
    }

    override suspend fun replaceAggregationEtag(
        etagItemListHashKey: String,
        etagKey: String,
        newEtag: String,
    ): Boolean =
        awaitResult { resultHandler ->
            when {
                vertx.isClustered ->
                    getClusteredItemListMap(
                        etagItemListHashKey,
                    ) {
                        when {
                            it.failed() -> resultHandler.handle(failedFuture(it.cause()))
                            else ->
                                it.result().put(etagKey, newEtag).onComplete { setRes ->
                                    when {
                                        setRes.failed() -> {
                                            logger.error(setRes.cause()) { "Unable to set etag!" }

                                            resultHandler.handle(failedFuture(setRes.cause()))
                                        }

                                        else -> resultHandler.handle(succeededFuture(TRUE))
                                    }
                                }
                        }
                    }

                else -> {
                    getLocalItemListMap(etagItemListHashKey)[etagKey] = newEtag

                    resultHandler.handle(succeededFuture(TRUE))
                }
            }
        }

    override suspend fun setSingleRecordEtag(etagMap: Map<String, String>): Boolean =
        awaitResult { resultHandler ->
            when {
                vertx.isClustered ->
                    getClusteredObjectMap { asyncResult ->
                        when {
                            asyncResult.failed() -> {
                                logger.error(asyncResult.cause()) { "Failed etag setting for objects!" }

                                resultHandler.handle(failedFuture(asyncResult.cause()))
                            }

                            else -> {
                                val map = asyncResult.result()
                                val setFutures = ArrayList<Promise<*>>()

                                etagMap.forEach { (k, v) ->
                                    val setFuture = Promise.promise<Void>()

                                    map.put(k, v).onComplete(setFuture)

                                    setFutures.add(setFuture)
                                }

                                Future.all<Any>(setFutures.map { it.future() }).andThen { result ->
                                    when {
                                        result.failed() -> resultHandler.handle(failedFuture(result.cause()))
                                        else -> resultHandler.handle(succeededFuture())
                                    }
                                }
                            }
                        }
                    }

                else -> {
                    val localObjectMap = localObjectMap
                    etagMap.forEach { (k, v) -> localObjectMap[k] = v }

                    resultHandler.handle(succeededFuture(TRUE))
                }
            }
        }

    override suspend fun setProjectionEtags(
        projections: Array<String>,
        hash: Int,
        item: E,
    ): Boolean =
        awaitResult { resultHandler ->
            if (projections.isNotEmpty()) {
                val etagKeyBase = TYPE.simpleName + "_" + hash + "/projections"
                val key = TYPE.simpleName + "_" + hash + "/projections" + projections.contentHashCode()
                val etag = item.etag

                when {
                    vertx.isClustered ->
                        getClusteredObjectMap(
                            etagKeyBase,
                        ) { mapRes ->
                            when {
                                mapRes.succeeded() -> {
                                    mapRes.result().put(key, etag).onComplete { setRes ->
                                        when {
                                            setRes.failed() -> {
                                                logger.warn(setRes.cause()) { "Etag Set failed!" }

                                                resultHandler.handle(failedFuture(setRes.cause()))
                                            }

                                            else -> resultHandler.handle(succeededFuture(TRUE))
                                        }
                                    }
                                }
                                else -> resultHandler.handle(failedFuture(mapRes.cause()))
                            }
                        }

                    else -> {
                        getLocalObjectMap(etagKeyBase)[key] = etag

                        resultHandler.handle(succeededFuture(TRUE))
                    }
                }
            }
        }

    override suspend fun setItemListEtags(
        etagItemListHashKey: String,
        etagKey: String,
        itemList: ItemList<E>,
    ): Boolean = setItemListEtags(etagItemListHashKey, etagKey, itemList.meta.etag)

    private suspend fun setItemListEtags(
        etagItemListHashKey: String,
        etagKey: String,
        etag: String?,
    ): Boolean {
        val itemListEtagFuture = Promise.promise<Boolean>()

        when {
            vertx.isClustered ->
                getClusteredItemListMap(
                    etagItemListHashKey,
                ) {
                    when {
                        it.failed() -> itemListEtagFuture.fail(it.cause())
                        else ->
                            it.result().put(etagKey, etag).onComplete { result ->
                                when {
                                    result.failed() -> {
                                        logger.error(result.cause()) { "Unable to set etag!" }

                                        itemListEtagFuture.fail(result.cause())
                                    }

                                    else -> itemListEtagFuture.complete(TRUE)
                                }
                            }
                    }
                }

            else -> {
                getLocalItemListMap(etagItemListHashKey)[etagKey] = etag

                itemListEtagFuture.complete(TRUE)
            }
        }

        return itemListEtagFuture.future().coAwait()
    }

    override suspend fun checkItemEtag(
        etagKeyBase: String,
        key: String,
        requestEtag: String,
    ): Boolean =
        awaitResult { resultHandler ->
            when {
                vertx.isClustered ->
                    getClusteredObjectMap(
                        etagKeyBase,
                    ) {
                        checkEtagFromMap(key, requestEtag, resultHandler, it)
                    }

                else -> {
                    val s = localObjectMap[key]

                    when {
                        s != null -> resultHandler.handle(succeededFuture(s == requestEtag))
                        else -> resultHandler.handle(succeededFuture(java.lang.Boolean.FALSE))
                    }
                }
            }
        }

    override suspend fun checkItemListEtag(
        etagItemListHashKey: String,
        etagKey: String,
        etag: String,
    ): Boolean =
        awaitResult { resultHandler ->
            when {
                vertx.isClustered ->
                    getClusteredItemListMap(
                        etagItemListHashKey,
                    ) {
                        checkEtagFromMap(etagKey, etag, resultHandler, it)
                    }

                else -> {
                    val s = getLocalItemListMap(etagItemListHashKey)[etagKey]

                    when {
                        s != null -> resultHandler.handle(succeededFuture(s == etag))
                        else -> resultHandler.handle(succeededFuture(java.lang.Boolean.FALSE))
                    }
                }
            }
        }

    override suspend fun checkAggregationEtag(
        etagItemListHashKey: String,
        etagKey: String,
        etag: String,
    ): Boolean = checkItemListEtag(etagItemListHashKey, etagKey, etag)

    private fun checkEtagFromMap(
        key: String,
        etag: String,
        resultHandler: Handler<AsyncResult<Boolean>>,
        res: AsyncResult<AsyncMap<String, String>>,
    ) {
        when {
            res.failed() -> resultHandler.handle(failedFuture(res.cause()))
            else ->
                res.result().get(key).onComplete { getRes ->
                    when {
                        getRes.failed() -> {
                            logger.error(getRes.cause()) { "Unable to set etag!" }

                            resultHandler.handle(failedFuture(getRes.cause()))
                        }

                        else ->
                            when {
                                getRes.result() != null && getRes.result() == etag ->
                                    resultHandler.handle(succeededFuture(TRUE))

                                else ->
                                    resultHandler.handle(
                                        succeededFuture(
                                            java.lang.Boolean.FALSE,
                                        ),
                                    )
                            }
                    }
                }
        }
    }

    companion object {
        private const val OBJECT_ETAG_MAP = "OBJECT_ETAG_MAP"
    }
}
